package de.fraunhofer.sit.c2x.pki.etsi_ts103097v1114.impl;

import java.io.DataInputStream;
import java.io.DataOutputStream;
import java.io.IOException;

import de.fraunhofer.sit.c2x.pki.etsi_ts103097v1114.serializer.External;
import de.fraunhofer.sit.c2x.pki.etsi_ts103097v1114.serializer.Internal;
import de.fraunhofer.sit.c2x.pki.etsi_ts103097v1114.visitor.EtsiVisitor;

/**
 * 
 * @author Daniel Quanz (daniel.quanz@sit.fraunhofer.de)
 * 
 */
public class Duration extends WaveElement {
	public enum DurationType {
		SECONDS(1), MINUTES(60), HOURS(3600), SIXTY_HOUR_BLOCKS(216000), YEARS(31556925);

		private final int seconds;

		private DurationType(int seconds) {
			this.seconds = seconds;
		}

		public int getSeconds() {
			return seconds;
		}
	}

	// ---- fields ----

	@Internal(order = 0)
	private final UInt16 duration;
	@External
	private final DurationType type;
	@External
	private final int durationValue;

	// ---- constructors ----
	/**
	 * 
	 * @param type
	 *            Duration type (sec, min, hours, sixty hour blocks, years)
	 * @param value
	 *            value, but do not use the UInt16 value of duration
	 */
	public Duration(DurationType type, int value) {
		this.type = type;
		this.durationValue = value;
		this.duration = computeDuration();
	}

	public Duration(long seconds) {

		if (seconds / DurationType.YEARS.getSeconds() > 8191) {
			throw new IllegalArgumentException();
		}
		DurationType[] types = { DurationType.SECONDS, DurationType.MINUTES, DurationType.HOURS,
				DurationType.SIXTY_HOUR_BLOCKS, DurationType.YEARS };

		DurationType t = null;
		int dV = -1;
		for (DurationType durationType : types) {

			long newVal = seconds / durationType.getSeconds();
			if (newVal <= 8191) {
				t = durationType;
				dV = (int) newVal;
				break;
			}
		}
		this.type = t;
		this.durationValue = dV;
		this.duration = computeDuration();
	}

	public Duration(DataInputStream in) throws IOException {
		duration = new UInt16(in);
		this.type = this.computeType();
		this.durationValue = this.computeDurationValue();
	}

	private UInt16 computeDuration() {

		System.out.println(type);
		int code = DurationTypeImpl.getInstance().getCode(type);
		code = code << 13;

		return new UInt16(code | durationValue);
	}

	private int computeDurationValue() {
		return duration.get() & 0x1FFF;
	}

	private DurationType computeType() {

		int type = this.duration.get() >> 13;
		return DurationTypeImpl.getInstance().getEnumType(type);

	}

	// ---- accept ----

	public <T> T accept(EtsiVisitor<T> visitor) {
		return visitor.visit(this);
	}

	// ---- getter ----

	public UInt16 getDuration() {
		return this.duration;
	}

	// XXX: test me
	public int getDurationInSeconds() {
		return durationValue * type.getSeconds();
	}

	// ---- setter ----

	@Override
	public int writeData(DataOutputStream out) throws IOException {
		if (duration == null)
			throw new IllegalArgumentException("args may not be null");
		return duration.writeData(out);
	}

	public int getDurationValue() {
		return durationValue;
	}

	public DurationType getType() {
		return type;
	}
}