package de.fraunhofer.sit.c2x.pki.ca.module.webserver.servlets;

import java.text.SimpleDateFormat;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

import org.apache.commons.codec.binary.Hex;

import com.google.inject.Inject;

import de.fraunhofer.sit.c2x.pki.ca.module.webserver.interfaces.HtmlProvider;
import de.fraunhofer.sit.c2x.pki.ca.provider.ProviderException;
import de.fraunhofer.sit.c2x.pki.ca.provider.entities.DBInterval;
import de.fraunhofer.sit.c2x.pki.ca.provider.entities.DBIntervalId;
import de.fraunhofer.sit.c2x.pki.ca.provider.interfaces.ConfigProvider;
import de.fraunhofer.sit.c2x.pki.ca.provider.interfaces.PPEExtendedProvider;
import de.fraunhofer.sit.c2x.pki.ca.utils.ByteUtils;

/**
 * A simple servlet that outputs the certificate content of issued certificates.
 * Following arguments are allowed per GET
 * 
 * <ul>
 * <li>CertId8 or CertId10: <b>/cacert?cert=HEX or Base64</b></li>
 * </ul>
 * 
 * @author Norbert Bissmeyer (norbert.bissmeyer@sit.fraunhofer.de)
 * 
 */
public class PPEServlet extends AbstractServlet {

	private static final long serialVersionUID = 1L;
	protected static final String URL = "/ppe";
	protected static final String TITLE = "Search pseudonym intervals";

	private final PPEExtendedProvider ppeProvider;
	private final ConfigProvider configProvider;

	@Inject
	public PPEServlet(HtmlProvider htmlProvider, PPEExtendedProvider ppeProvider,
			ConfigProvider configProvider) {
		super(htmlProvider);
		this.ppeProvider = ppeProvider;
		this.configProvider = configProvider;
	}

	@Override
	public String getUrl() {
		return URL;
	}

	@Override
	public String getTitle() {
		return TITLE;
	}

	@Override
	protected String contentHtml(HttpServletRequest req, HttpServletResponse resp) {
		StringBuilder sb = new StringBuilder();

		try {
			if (req.getParameter("cert") != null && req.getParameter("cert").length() > 0) {

				int plp = configProvider.getInt("maxPseudonymLifetime");
				byte[] certId = Hex.decodeHex(req.getParameter("cert").toCharArray());

				listDetail(sb, plp, certId);
			} else {
				if (req.getParameter("cert") != null && req.getParameter("cert").length() > 0)
					listAll(req, sb, Hex.decodeHex(req.getParameter("cert").toCharArray()));
				else
					listAll(req, sb, null);
			}
		} catch (Exception e) {
			sb = new StringBuilder();
			sb.append("<div id=\"errorbox\">Error: Can not fetch Elements from DB! " + e.getMessage()
					+ "</div>");
		}

		return sb.toString();
	}

	private void listDetail(StringBuilder sb, int plp, byte[] certId) throws ProviderException {
		// details
		// list all
		DBInterval[] intervals = ppeProvider.getAll(certId);

		if (certId != null && intervals.length == 0) {
			sb.append("<div id=\"errorbox\">No interval found in database for certId="
					+ Hex.encodeHexString(certId) + "</div>");
			return;
		}
		sb.append("<table border=\"0\" style=\"border: solid 1px #ccc;\">\n");
		sb.append("<tr style=\"background-color:#ddd; font-weight:bold;\">\n<td width=\"150px\">LTC-CertId (HashedId8)</td>\n<td width=\"80px\"></td></tr>\n");
		sb.append("<tr>\n<td><span style=\"font-family:courier new;\">" + ByteUtils.getHex(certId)
				+ "</span></td>\n<td align=\"center\">");
		sb.append("<form action=\"" + PPEDeletionServlet.URL
				+ "\" method=\"get\"><input type=\"hidden\" name=\"cert\" value=\""
				+ ByteUtils.getHex(certId) + "\"/><input type=\"submit\" value=\"Delete\" /></form>");
		// sb.append("<a style=\"color:#F00;\" href=\"/ppe?delete=true&amp;cert="
		// + ByteUtils.getHex(certId) + "&amp;region=" +
		// ByteUtils.getHex(region)
		// + "\">Delete</a>");
		sb.append("</td>\n</tr>\n");
		sb.append("</table><br /><br />");

		String ppeimageString = create(intervals, plp);
		sb.append("<img src=\"/ppeImage?plp=" + plp + "&amp;" + ppeimageString + "\" /><br /><br />");

		sb.append("<table border=\"0\" style=\"border: solid 1px #ccc;\">\n");
		sb.append("<tr style=\"background-color:#ddd; font-weight:bold;\">\n<td width=\"70px\">interval</td>\n<td width=\"160px\">start time (UTC)</td>\n<td width=\"160px\">expiration time (UTC)</td>\n<td width=\"150px\">number of PLP intervals</td>\n<td width=\"100px\">activated certs</td>\n<td width=\"100px\">reserved certs</td>\n</tr>\n");
		int i = 0;
		for (DBInterval interval : intervals) {
			long noI = ((interval.getId().getEndTime() - interval.getId().getStartTime()) / (plp * 1000l));
			sb.append("<tr>\n<td># " + (++i) + "</td>\n<td>"
					+ new SimpleDateFormat("yyyy-MM-dd HH:mm:ss").format(interval.getId().getStartTime())
					+ "</td>\n<td>"
					+ new SimpleDateFormat("yyyy-MM-dd HH:mm:ss").format(interval.getId().getEndTime())
					+ "</td>\n<td>" + noI + "</td>\n<td>" + (noI * interval.getActivatedCerts())
					+ "</td>\n<td>" + (noI * interval.getReservedCerts()) + "</td>\n</tr>\n");
		}
		sb.append("</table>\n");

		sb.append("<br /><br />");
		sb.append("<form action=\"" + URL
				+ "\" method=\"get\"><input type=\"submit\" value=\"Back\" /></form>");
		// sb.append("<a href=\"" + URL + "\">Back</a>");
	}

	private void listAll(HttpServletRequest req, StringBuilder sb, byte[] cert) throws ProviderException {

		sb.append("<form name=\"certForm\" action=\"\" method=\"get\">Search interval with LTC-CertId encoded as HEX string: ");
		sb.append("<input type=\"text\" id=\"cert\" name=\"cert\" value=\"\" maxlength=\"20\" style=\"width:200px;\" />&nbsp;<input type=\"submit\" value=\"Find\" /></form><br /><br />");

		// list all
		DBIntervalId[] intervals = new DBIntervalId[0];
		if (cert != null)
			intervals = ppeProvider.getList(cert);

		if (intervals.length == 0) {
			if (cert != null) {
				sb.append("<div id=\"errorbox\">No interval found in database for certId="
						+ Hex.encodeHexString(cert) + "</div>");
			}
		} else {
			sb.append("<table border=\"0\" style=\"border: solid 1px #ccc;\">\n");
			sb.append("<tr style=\"background-color:#ddd; font-weight:bold;\">\n<td width=\"250px\">LTC-CertId (HEX encoded HashedId8)</td>\n<td width=\"80px\"></td>\n<td width=\"80px\"></td>\n</tr>\n");
			for (DBIntervalId interval : intervals) {
				String certId = ByteUtils.getHex(interval.getCertId());
				sb.append("<tr>\n<td><span style=\"font-family:courier new;\">" + certId + "</span></td>\n");
				sb.append("<td align=\"center\"><form action=\"" + URL
						+ "\" method=\"get\"><input type=\"hidden\" name=\"cert\" value=\"" + certId
						+ "\"/><input type=\"submit\" value=\"Show details\" /></form></td>\n");
				sb.append("<td align=\"center\"><form action=\"" + PPEDeletionServlet.URL
						+ "\" method=\"get\"><input type=\"hidden\" name=\"cert\" value=\"" + certId
						+ "\"/><input type=\"submit\" value=\"Delete\" /></form></td>\n");
				// sb.append("<td align=\"center\"><a style=\"color:#F00;\" href=\"/ppe?delete=true&amp;cert="
				// + certId + "&amp;region=" + regionHash +
				// "\">Delete</a></td>\n");
				sb.append("</tr>\n");
			}
			sb.append("</table>\n");
		}
	}

	private static String create(DBInterval[] intervals, int plp) {
		StringBuilder sb = new StringBuilder();
		for (DBInterval dbInterval : intervals) {
			sb.append("&amp;start=" + dbInterval.getId().getStartTime() / 1000);
			sb.append("&amp;end=" + (dbInterval.getId().getEndTime()) / 1000);
			long noI = ((dbInterval.getId().getEndTime() - dbInterval.getId().getStartTime()) / (plp * 1000l));
			sb.append("&amp;n=" + (noI  * dbInterval.getActivatedCerts()));
		}
		return sb.toString().substring(5);
	}

	@Override
	public boolean isProtected() {
		return true;
	}

	@Override
	public String[] getAllowedRoles() {
		return new String[] { UserRole.ADMINISTRATOR.toString(), UserRole.REGISTRATION_USER.toString(),
				UserRole.DEVELOPER.toString() };
	}

	@Override
	public Category getCategory() {
		return Category.LTC;
	}
}
