package de.fraunhofer.sit.c2x.pki.etsi_ts103097v1114.impl;

import java.io.DataInputStream;
import java.io.DataOutputStream;
import java.io.IOException;

import de.fraunhofer.sit.c2x.pki.ca.utils.WaveUtils;
import de.fraunhofer.sit.c2x.pki.etsi_ts103097v1114.impl.HeaderFieldTypeImpl.HeaderFieldType;
import de.fraunhofer.sit.c2x.pki.etsi_ts103097v1114.serializer.Internal;
import de.fraunhofer.sit.c2x.pki.etsi_ts103097v1114.visitor.EtsiVisitor;

/**
 * 
 * @author Daniel Quanz (daniel.quanz@sit.fraunhofer.de)
 * 
 */
public class HeaderField extends WaveElement {

	// ---- fields ----

	@Internal(order = 0)
	private HeaderFieldType type;

	@Internal(order = 1)
	private Time64 generationTime;

	@Internal(order = 2)
	private Time64WithStandardDeviation generationTimeWithStandardDeviation;

	@Internal(order = 3)
	private Time32 expiryTime;

	@Internal(order = 4)
	private ThreeDLocation generationLocation;

	@Internal(order = 5)
	private HashedId3[] digests;

	@Internal(order = 6)
	private IntX itsAid;

	@Internal(order = 7)
	private SignerInfo signer;
	
	@Internal(order = 8)
	private EncryptionParameters encParams;
	
	@Internal(order = 9)
	private RecipientInfo[] recipients;

	@Internal(order = 10)
	private Opaque otherHeader;

	// ---- constructors ----

	public HeaderField() {
	}

	public HeaderField(DataInputStream in) throws IOException {
		type = HeaderFieldTypeImpl.getInstance().getEnumType(in.readByte());
		if (type == HeaderFieldType.GENERATION_TIME) {
			generationTime = new Time64(in);
		} else if (type == HeaderFieldType.GENERATION_TIME_STANDARD_DEVIATION) {
			generationTimeWithStandardDeviation = new Time64WithStandardDeviation(in);
		} else if (type == HeaderFieldType.EXPIRATION) {
			expiryTime = new Time32(in);
		} else if (type == HeaderFieldType.GENERATION_LOCATION) {
			generationLocation = new ThreeDLocation(in);
		} else if (type == HeaderFieldType.REQUEST_UNRECOGNIZED_CERTIFICATE) {
			digests = WaveUtils.getArrayFromStream(in, HashedId3.class);
		} else if (type == HeaderFieldType.ITS_AID) {
			itsAid = new IntX(in);
		} else if (type == HeaderFieldType.SIGNER_INFO) {
			signer = new SignerInfo(in);
		} else if (type == HeaderFieldType.ENCRYPTION_PARAMETERS) {
			encParams = new EncryptionParameters(in);
		} else if (type == HeaderFieldType.RECIPIENT_INFO) {
			recipients = WaveUtils.getArrayFromStream(in, RecipientInfo.class);
		} else {
			otherHeader = new Opaque(in);
		}
	}

	// ---- accept ----

	public <T> T accept(EtsiVisitor<T> visitor) {
		return visitor.visit(this);
	}


	public HeaderFieldType getType() {
		return type;
	}

	public Time64 getGenerationTime() {
		return generationTime;
	}

	public Time64WithStandardDeviation getGenerationTimeWithStandardDeviation() {
		return generationTimeWithStandardDeviation;
	}

	public Time32 getExpiryTime() {
		return expiryTime;
	}

	public ThreeDLocation getGenerationLocation() {
		return generationLocation;
	}

	public HashedId3[] getDigests() {
		return digests;
	}

	public IntX getItsAid() {
		return itsAid;
	}

	public SignerInfo getSigner() {
		return signer;
	}

	public RecipientInfo[] getRecipients() {
		return recipients;
	}

	public Opaque getOtherHeader() {
		return otherHeader;
	}

	public void setType(HeaderFieldType type) {
		this.type = type;
	}

	public void setGenerationTime(Time64 generationTime) {
		this.generationTime = generationTime;
	}

	public void setGenerationTimeWithStandardDeviation(
			Time64WithStandardDeviation generationTimeWithStandardDeviation) {
		this.generationTimeWithStandardDeviation = generationTimeWithStandardDeviation;
	}

	public void setExpiryTime(Time32 expiryTime) {
		this.expiryTime = expiryTime;
	}

	public void setGenerationLocation(ThreeDLocation generationLocation) {
		this.generationLocation = generationLocation;
	}

	public void setDigests(HashedId3[] digests) {
		this.digests = digests;
	}

	public void setItsAid(IntX itsAid) {
		this.itsAid = itsAid;
	}

	public void setSigner(SignerInfo signer) {
		this.signer = signer;
	}

	public void setRecipients(RecipientInfo[] recipients) {
		this.recipients = recipients;
	}

	public void setOtherHeader(Opaque otherHeader) {
		this.otherHeader = otherHeader;
	}

	@Override
	public int writeData(DataOutputStream out) throws IOException {
		if (type == null)
			throw new IllegalArgumentException("arg may not be null");
		int written = HeaderFieldTypeImpl.getInstance().writeData(out, type);
		if (type == HeaderFieldType.GENERATION_TIME) {
			if (generationTime == null)
				throw new IllegalArgumentException("arg may not be null");
			written += generationTime.writeData(out);
		} else if (type == HeaderFieldType.GENERATION_TIME_STANDARD_DEVIATION) {
			if (generationTimeWithStandardDeviation == null)
				throw new IllegalArgumentException("arg may not be null");
			written += generationTimeWithStandardDeviation.writeData(out);
		} else if (type == HeaderFieldType.EXPIRATION) {
			if (expiryTime == null)
				throw new IllegalArgumentException("arg may not be null");
			written += expiryTime.writeData(out);
		} else if (type == HeaderFieldType.GENERATION_LOCATION) {
			if (generationLocation == null)
				throw new IllegalArgumentException("arg may not be null");
			written += generationLocation.writeData(out);
		} else if (type == HeaderFieldType.REQUEST_UNRECOGNIZED_CERTIFICATE) {
			if (digests == null)
				throw new IllegalArgumentException("arg may not be null");
			written += WaveUtils.writeArrayToStream(out, digests);
		} else if (type == HeaderFieldType.ITS_AID) {
			if (itsAid == null)
				throw new IllegalArgumentException("arg may not be null");
			written += itsAid.writeData(out);
		} else if (type == HeaderFieldType.SIGNER_INFO) {
			if (signer == null)
				throw new IllegalArgumentException("arg may not be null");
			written += signer.writeData(out);
		} else if (type == HeaderFieldType.ENCRYPTION_PARAMETERS) {
			if (encParams == null)
				throw new IllegalArgumentException("arg may not be null");
			written += encParams.writeData(out);
		} else if (type == HeaderFieldType.RECIPIENT_INFO) {
			if (recipients == null)
				throw new IllegalArgumentException("arg may not be null");
			written += WaveUtils.writeArrayToStream(out, recipients);
		} else {
			if (otherHeader == null)
				throw new IllegalArgumentException("arg may not be null");
			written += otherHeader.writeData(out);
		}
		return written;
	}
}