package de.fraunhofer.sit.c2x.pki.etsi_ts103097v1114.handler;

import java.io.ByteArrayOutputStream;
import java.io.DataOutputStream;
import java.io.IOException;
import java.math.BigInteger;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;

import org.apache.commons.codec.binary.Hex;
import org.apache.log4j.Logger;
import org.bouncycastle.util.encoders.Base64;

import com.google.inject.Inject;

import de.fraunhofer.sit.c2x.pki.ca.certificates.datacontainers.GeographicRegionDataContainer;
import de.fraunhofer.sit.c2x.pki.ca.certificates.datacontainers.PsidSspPriorityDataContainer;
import de.fraunhofer.sit.c2x.pki.ca.core.exceptions.HandlerException;
import de.fraunhofer.sit.c2x.pki.ca.core.interfaces.HandlerWithResult;
import de.fraunhofer.sit.c2x.pki.ca.core.logging.InjectLogger;
import de.fraunhofer.sit.c2x.pki.ca.measuring.MeasuringKey;
import de.fraunhofer.sit.c2x.pki.ca.module.webserver.servlets.interfaces.ItsStationRegistrationHandler;
import de.fraunhofer.sit.c2x.pki.ca.module.webservice.datatypes.ItsStationRegistrationRequest;
import de.fraunhofer.sit.c2x.pki.ca.module.webservice.datatypes.ItsStationRegistrationRequestData;
import de.fraunhofer.sit.c2x.pki.ca.module.webservice.datatypes.PsidSspPriority;
import de.fraunhofer.sit.c2x.pki.ca.module.webservice.datatypes.Region;
import de.fraunhofer.sit.c2x.pki.ca.provider.ProviderException;
import de.fraunhofer.sit.c2x.pki.ca.provider.entities.Authenticator;
import de.fraunhofer.sit.c2x.pki.ca.provider.entities.AuthorizedDevice;
import de.fraunhofer.sit.c2x.pki.ca.provider.entities.HttpUser;
import de.fraunhofer.sit.c2x.pki.ca.provider.entities.PublicKey;
import de.fraunhofer.sit.c2x.pki.ca.provider.interfaces.AuthorizedDeviceProvider;
import de.fraunhofer.sit.c2x.pki.ca.provider.interfaces.CaInfoProvider;
import de.fraunhofer.sit.c2x.pki.ca.provider.interfaces.ConfigProvider;
import de.fraunhofer.sit.c2x.pki.ca.provider.interfaces.HttpUserProvider;
import de.fraunhofer.sit.c2x.pki.ca.provider.interfaces.MeasuringProvider;
import de.fraunhofer.sit.c2x.pki.ca.utils.WaveUtils;
import de.fraunhofer.sit.c2x.pki.ca.validator.region.GeorgraphicRegionValidator;
import de.fraunhofer.sit.c2x.pki.etsi_ts103097v1114.impl.Certificate;
import de.fraunhofer.sit.c2x.pki.etsi_ts103097v1114.impl.GeographicRegion;
import de.fraunhofer.sit.c2x.pki.etsi_ts103097v1114.impl.IntX;
import de.fraunhofer.sit.c2x.pki.etsi_ts103097v1114.impl.ItsAidPriority;
import de.fraunhofer.sit.c2x.pki.etsi_ts103097v1114.impl.ItsAidPrioritySsp;
import de.fraunhofer.sit.c2x.pki.etsi_ts103097v1114.impl.ItsAidSsp;
import de.fraunhofer.sit.c2x.pki.etsi_ts103097v1114.impl.RegionDictionaryImpl.RegionDictionary;
import de.fraunhofer.sit.c2x.pki.etsi_ts103097v1114.impl.RegionTypeImpl.RegionType;
import de.fraunhofer.sit.c2x.pki.etsi_ts103097v1114.impl.SubjectAssurance;
import de.fraunhofer.sit.c2x.pki.etsi_ts103097v1114.impl.SubjectAttribute;
import de.fraunhofer.sit.c2x.pki.etsi_ts103097v1114.impl.ValidityRestriction;
import de.fraunhofer.sit.c2x.pki.etsi_ts103097v1114.impl.ValidityRestrictionTypeImpl.ValidityRestrictionType;
import de.fraunhofer.sit.c2x.pki.etsi_ts103097v1114.utils.EtsiPerrmissionUtils;
import de.fraunhofer.sit.c2x.pki.etsi_ts103097v1114.utils.EtsiRegionUtils;

public class EtsiItsStationRegistrationHandler implements HandlerWithResult<AuthorizedDevice, Boolean>,
		ItsStationRegistrationHandler {

	@InjectLogger
	private Logger logger;

	@Inject
	private AuthorizedDeviceProvider adp;

	@Inject
	private CaInfoProvider<Certificate> caInfo;

	@Inject
	private MeasuringProvider measuringProvider;

	@Inject
	private GeorgraphicRegionValidator geoValidator;

	@Inject
	private HttpUserProvider httpUserProvider;

	@Inject
	private ConfigProvider configProvider;

	@Override
	public Boolean handle(AuthorizedDevice in) throws HandlerException {

		if (in.getCanonicalId() != null && in.getCanonicalId().length > 0) {

			byte[] pk = in.getPublicKey().getPublicKey();
			if (pk.length == 65) {
				try {
					return adp.save(in);
				} catch (ProviderException e) {
					throw new HandlerException(e);
				}
			} else {
				throw new HandlerException("Invalid length of public key: " + pk.length + ". Must be 65");
			}

		} else {
			throw new HandlerException("No canonical ID given");
		}
	}

	@Override
	public boolean handleItsStationRegistration(byte[] canonicalId, PublicKey publicKey,
			byte[] assuranceLevel, List<PsidSspPriorityDataContainer> requestedPermissions,
			GeographicRegionDataContainer region, String httpUserId) throws HandlerException {
		try {
			// get prefix from DB
			HttpUser user = httpUserProvider.getUser(httpUserId);

			AuthorizedDevice device = getDeviceFromRequest(canonicalId, publicKey, assuranceLevel,
					requestedPermissions, region, null, user.getItsSRegPrefix());
			boolean result = adp.save(device);
			if (result == true) {
				if (logger.isDebugEnabled()) {
					logger.debug("ITS-S with canonical ID " + Hex.encodeHexString(canonicalId)
							+ " stored in DB.");
				}
			}
			return result;
		} catch (Exception e) {
			throw new HandlerException(e);
		}

	}

	public BigInteger handleItsStationRegistration(ItsStationRegistrationRequest in,
			Authenticator authenticator) throws HandlerException {
		long timeRequestReceived = System.currentTimeMillis();
		byte[] caCertDigest;
		try {
			caCertDigest = caInfo.getCaCertificate().getHashedId8().getCertID().get();
		} catch (ProviderException e) {
			throw new HandlerException(e);
		}
		byte[] requestHash = String.valueOf(in.hashCode()).getBytes();
		int numRequestedItsS = in.getItsStationRegistrationRequestData().size();

		BigInteger n = BigInteger.ZERO;

		ArrayList<AuthorizedDevice> devices = new ArrayList<AuthorizedDevice>();

		// check the permissions and handle the request
		try {

			// get the CA certificate in order to check permissions
			Certificate caCert = caInfo.getCaCertificate();
			if (caCert == null)
				throw new HandlerException("No CA certificate available!");

			// retrieve the data of each device and check for duplicates in DB
			for (ItsStationRegistrationRequestData data : in.getItsStationRegistrationRequestData()) {

				byte[] canonicalId = data.getModuleId();
				PublicKey publicKey = new PublicKey();
				publicKey.setPublicKey(data.getDeviceIdentityEccPublicKeyX(),
						data.getDeviceIdentityEccPublicKeyY(), "ECDSA_NISTP256_WITH_SHA256");
				byte[] assuranceLevel = data.getSubjectAssurance();
				List<PsidSspPriorityDataContainer> requestedPermissions = convertPsidFormat(data
						.getPsidSspPriority());
				GeographicRegionDataContainer region = convertRegion(data.getRegion());

				// get prefix from DB
				byte[] authenticatedCanonicalIdPrefix = null;
				if (authenticator != null) {
					HttpUser user = httpUserProvider.getUser(authenticator.getHttpUserId());
					authenticatedCanonicalIdPrefix = user.getItsSRegPrefix();

					if (logger.isDebugEnabled())
						logger.debug("Authenticator with email: " + authenticator.getEmail()
								+ " was created by http user: " + authenticator.getHttpUserId()
								+ ". ITS-S registration canonical ID prefix found: "
								+ Hex.encodeHexString(authenticatedCanonicalIdPrefix));
				}

				AuthorizedDevice device = getDeviceFromRequest(canonicalId, publicKey, assuranceLevel,
						requestedPermissions, region, authenticator, authenticatedCanonicalIdPrefix);

				for (AuthorizedDevice toBeComparedDevice : devices) {
					if (device.getCanonicalId().equals(toBeComparedDevice.getCanonicalId())) {
						throw new HandlerException("Duplicate device within a single request!");
					}
				}

				devices.add(device);

				// check if the device is already registered in DB
				if (adp.get(device.getCanonicalId()) != null) {
					throw new HandlerException("Device " + new String(Base64.encode(data.getModuleId()))
							+ " is already registered in DB.");
				}
			}

			// create time measurement
			measuringProvider.add(caCertDigest, requestHash,
					MeasuringKey.ITS_S_REGISTRATION_PERMISSION_CHECKS,
					(System.currentTimeMillis() - timeRequestReceived), numRequestedItsS);
			long time = System.currentTimeMillis();

			// Store all devices in DB
			for (Iterator<AuthorizedDevice> iterator = devices.iterator(); iterator.hasNext();) {
				AuthorizedDevice authorizedDevice = (AuthorizedDevice) iterator.next();

				if (adp.save(authorizedDevice)) {
					n = n.add(BigInteger.ONE);
				} else {
					// TODO remove already stored devices after adding a device
					// failed
					throw new HandlerException("Failed to store device "
							+ new String(Base64.encode(authorizedDevice.getCanonicalId())) + " in DB.");
				}

			}

			if (logger.isDebugEnabled()) {
				String itsStationModuleIds = "";
				for (AuthorizedDevice ad : devices) {
					if (itsStationModuleIds.isEmpty() == false)
						itsStationModuleIds += ", ";
					itsStationModuleIds += Hex.encodeHexString(ad.getCanonicalId());
				}
				logger.debug(n + " device(s) registered and stored in DB. The IDs of registered ITS-S are: "
						+ itsStationModuleIds);
			}

			// create time measurement
			measuringProvider.add(caCertDigest, requestHash, MeasuringKey.ITS_S_REGISTRATION_STORAGE,
					(System.currentTimeMillis() - time), numRequestedItsS);
			measuringProvider.add(caCertDigest, requestHash, MeasuringKey.ITS_S_REGISTRATION_PROCESSING,
					(System.currentTimeMillis() - timeRequestReceived), numRequestedItsS);

			return n;

		} catch (Exception e) {
			throw new HandlerException(e.getMessage(), e);
		}

	}

	private List<PsidSspPriorityDataContainer> convertPsidFormat(List<PsidSspPriority> in) {

		List<PsidSspPriorityDataContainer> out = new ArrayList<>();

		if (in == null)
			return out;

		for (PsidSspPriority pIn : in) {
			PsidSspPriorityDataContainer pOut = new PsidSspPriorityDataContainer();
			Integer maxPriority = null;
			Long psid = null;
			if (pIn.getMaxPriority() != null)
				maxPriority = pIn.getMaxPriority().intValue();
			if (pIn.getPsid() != null)
				psid = pIn.getPsid().longValue();

			pOut.init(psid, maxPriority, pIn.getServiceSpecificPermissions());
			out.add(pOut);
		}

		return out;
	}

	private GeographicRegionDataContainer convertRegion(Region in) throws HandlerException {

		if (in == null) {
			return null;
		}

		if (in.getGeographicRegion() != null && in.getIdentifiedRegion() != null) {
			throw new HandlerException("Multiple regions detected - only a single region is allowed!");
		}

		if (in.getGeographicRegion() != null
				&& in.getGeographicRegion().getRegionType().intValue() == RegionType.CIRCLE.ordinal()) {
			if (in.getGeographicRegion().getCoordinates() == null
					|| in.getGeographicRegion().getCoordinates().size() != 1)
				throw new HandlerException("Invalid region parameters - wrong number of coordinates");

			return new GeographicRegionDataContainer(in.getGeographicRegion().getCoordinates().get(0)
					.getLatitudeDouble(), in.getGeographicRegion().getCoordinates().get(0)
					.getLongitudeDouble(), in.getGeographicRegion().getRadius().intValue());
		} else if (in.getIdentifiedRegion() != null
				&& in.getGeographicRegion().getRegionType().intValue() == RegionType.ID.ordinal()) {

			String dictionary = "unknown";
			if (in.getIdentifiedRegion().getRegionDictionary().intValue() == RegionDictionary.ISO_3166_1
					.ordinal()) {
				dictionary = "iso_3166_1";
			} else if (in.getIdentifiedRegion().getRegionDictionary().intValue() == RegionDictionary.UN_STATS
					.ordinal()) {
				dictionary = "un_stats";
			}

			return new GeographicRegionDataContainer(dictionary, in.getIdentifiedRegion()
					.getRegionIdentifier().longValue(), in.getIdentifiedRegion().getLocalRegion().longValue());
		} else if (in.getIdentifiedRegion() == null && in.getGeographicRegion() == null
				&& in.getGeographicRegion().getRegionType().intValue() == RegionType.NONE.ordinal()) {
			// no region
			return null;
		} else {
			throw new HandlerException("Unsupported Region");
		}

	}

	/**
	 * @param canonicalId
	 * @param publicKey
	 * @param assuranceLevel
	 * @param requestedPermissions
	 * @param region
	 * @param withAuthenticator
	 * @return
	 * @throws HandlerException
	 */
	private AuthorizedDevice getDeviceFromRequest(byte[] canonicalId, PublicKey publicKey,
			byte[] assuranceLevel, List<PsidSspPriorityDataContainer> requestedPermissions,
			GeographicRegionDataContainer region, Authenticator authenticator,
			byte[] authenticatedCanonicalIdPrefix) throws HandlerException {

		if (canonicalId == null || assuranceLevel == null || publicKey == null) {
			throw new HandlerException("Required argument missing");
		}

		if (configProvider.getBoolean("itsSRegistrationCompareCanonicalIdPrefix", true)) {
			if (authenticatedCanonicalIdPrefix != null) {
				if (authenticatedCanonicalIdPrefix.length != 8)
					throw new HandlerException("Invalid canonical ID prefix loaded from DB: "
							+ Hex.encodeHexString(authenticatedCanonicalIdPrefix));

				for (int i = 0; i < 8; i++) {
					if (canonicalId[i] != authenticatedCanonicalIdPrefix[i])
						throw new HandlerException("Invalid prefix of canonical ID: given first 8 bytes "
								+ Hex.encodeHexString(canonicalId) + " != permitted prefix bytes "
								+ Hex.encodeHexString(authenticatedCanonicalIdPrefix));
				}

				if (logger.isDebugEnabled())
					logger.debug("Given canonical ID prefix "
							+ Hex.encodeHexString(authenticatedCanonicalIdPrefix) + " is valid");
			}
		} else {
			if (logger.isDebugEnabled())
				logger.debug("Canonical ID prefix " + Hex.encodeHexString(authenticatedCanonicalIdPrefix)
						+ " is not checked as deactived in the configuration");
		}

		// get the CA certificate in order to check permissions
		Certificate caCert;
		try {
			caCert = caInfo.getCaCertificate();
		} catch (ProviderException e) {
			throw new HandlerException(e);
		}
		if (caCert == null)
			throw new HandlerException("No CA certificate available, canceling ITS-S registration!");

		// read abstract permission data into format specific structure
		ArrayList<SubjectAttribute> requestedSubjectAttributes = null;
		IntX[] itsAids = null;
		byte[] itsAidsBytes = null;
		ItsAidSsp[] itsAidSsps = null;
		byte[] itsAidSspsBytes = null;
		ItsAidPriority[] itsAidPriorities = null;
		byte[] itsAidPrioritiesBytes = null;
		ItsAidPrioritySsp[] itsAidPrioritySsps = null;
		byte[] itsAidPrioritySspsBytes = null;
		requestedSubjectAttributes = EtsiPerrmissionUtils
				.psidSetToSubjectAttributeArray(requestedPermissions);
		if (requestedSubjectAttributes == null)
			requestedSubjectAttributes = new ArrayList<SubjectAttribute>(1);
		requestedSubjectAttributes.add(new SubjectAttribute(new SubjectAssurance(assuranceLevel)));

		// check ItsAid-Permissions with LTCA
		EtsiPerrmissionUtils
				.checkSubjectAttributes(requestedSubjectAttributes, caCert.getSubjectAttributes());
		if (logger != null && logger.isDebugEnabled()) {
			logger.debug("ItsAid-check successful: Permissions of the ITS-S comply with the LTCA permissions!");
		}

		if (authenticator != null) {
			// authenticator checks
			EtsiPerrmissionUtils.checkSubjectAttributes(requestedSubjectAttributes, authenticator);
			if (logger != null && logger.isDebugEnabled()) {
				logger.debug("ItsAid-check successful: Permissions of the ITS-S comply with the authenticator!");
			}

			// EtsiPerrmissionUtils.checkSubjectAttributes(requestedAttributes,
			// permittedAttributes)

		} else {
			if (logger.isDebugEnabled())
				logger.debug("No authenticator involved in the registration (i.e. webpage registration or client-server authentication deactivated). Checks are skipped");
		}

		for (SubjectAttribute sa : requestedSubjectAttributes) {
			switch (sa.getType()) {
			case ITS_AID_LIST:
				itsAids = sa.getItsAidList();
				if (itsAids != null && itsAids.length > 0) {
					try {
						ByteArrayOutputStream bos = new ByteArrayOutputStream();
						DataOutputStream dos = new DataOutputStream(bos);
						WaveUtils.writeArrayToStream(dos, itsAids);
						dos.flush();
						itsAidsBytes = bos.toByteArray();
					} catch (IOException e) {
						throw new HandlerException("Unable to encode ItsAidSspList");
					}
				}
				break;
			case ITS_AID_SSP_LIST:
				itsAidSsps = sa.getItsAidSspList();
				if (itsAidSsps != null && itsAidSsps.length > 0) {
					try {
						ByteArrayOutputStream bos = new ByteArrayOutputStream();
						DataOutputStream dos = new DataOutputStream(bos);
						WaveUtils.writeArrayToStream(dos, itsAidSsps);
						dos.flush();
						itsAidSspsBytes = bos.toByteArray();
					} catch (IOException e) {
						throw new HandlerException("Unable to encode ItsAidSspList");
					}
				}
				break;
			case PRIORITY_ITS_AID_LIST:
				itsAidPriorities = sa.getItsAidPriorityList();
				if (itsAidPriorities != null && itsAidPriorities.length > 0) {
					try {
						ByteArrayOutputStream bos = new ByteArrayOutputStream();
						DataOutputStream dos = new DataOutputStream(bos);
						WaveUtils.writeArrayToStream(dos, itsAidPriorities);
						dos.flush();
						itsAidPrioritiesBytes = bos.toByteArray();
					} catch (IOException e) {
						throw new HandlerException("Unable to encode ItsAidPriorityList");
					}
				}
				break;
			case PRIORITY_SSP_LIST:
				itsAidPrioritySsps = sa.getItsAidPrioritySspList();
				if (itsAidPrioritySsps != null && itsAidPrioritySsps.length > 0) {
					try {
						ByteArrayOutputStream bos = new ByteArrayOutputStream();
						DataOutputStream dos = new DataOutputStream(bos);
						WaveUtils.writeArrayToStream(dos, itsAidPrioritySsps);
						dos.flush();
						itsAidPrioritySspsBytes = bos.toByteArray();
					} catch (IOException e) {
						throw new HandlerException("Unable to encode ItsAidPrioritySspList");
					}
				}
				break;
			case ASSURANCE_LEVEL:
				break;
			default:
				throw new HandlerException("Invalid subject attribute found: " + sa.getType()
						+ ". Please provide only AssuranceLevel, AIDs, SSPs, and / or Priorities");
			}
		}

		// read abstract region data into format specific structure
		GeographicRegion geographicRegion = EtsiRegionUtils.regionContainerToGeographicRegion(region);

		// Check CA region
		GeographicRegion caRegion = null;

		for (ValidityRestriction v : caCert.getValidityRestrictions()) {
			if (v.getType() == ValidityRestrictionType.REGION) {
				caRegion = v.getRegion();
				break;
			}
		}

		if (caRegion == null)
			throw new HandlerException(
					"The CA certificate is invalid as it contains no GeographicRegion; canceling ITS-S registration!");
		if (caRegion.getRegionType() != RegionType.NONE) {
			// validate region vs. LTCA certificate
			if (!new GeorgraphicRegionValidator(caRegion).validate(geographicRegion)) {
				throw new HandlerException(
						"RegionValidation failed! The LTCA certificate does not allow the requested region");
			}
			if (logger != null && logger.isDebugEnabled()) {
				logger.debug("Region-check successful: The region of the ITS-S complies with the LTCA region!");
			}
		} else {
			if (logger != null && logger.isDebugEnabled()) {
				logger.debug("Region-check successful! (No region restriction given by the LTCA)");
			}
		}

		// Check Authenticator region
		if (authenticator != null) {
			EtsiPerrmissionUtils.checkRegionRestrictions(geographicRegion, authenticator, geoValidator);
			logger.debug("Region-check successful! The region of the ITS-S complies with the Authenticator region!");
		}

		byte[] circularRegionBytes = null;
		byte[] identifiedRegionBytes = null;
		switch (geographicRegion.getRegionType()) {
		case ID:
			identifiedRegionBytes = WaveUtils.getBytesFromWaveType(geographicRegion.getIdRegion());
			break;
		case CIRCLE:
			circularRegionBytes = WaveUtils.getBytesFromWaveType(geographicRegion.getCircularRegion());
			break;
		case NONE:
			break;
		default:
			throw new HandlerException("Invalid region found: " + geographicRegion.getRegionType()
					+ ". Please provide only IdentifiedRegion or CircularRegion");
		}

		AuthorizedDevice device = new AuthorizedDevice();
		device.setCanonicalId(canonicalId);
		device.setPublicKey(publicKey);
		device.setSubjectAssurance(assuranceLevel);
		device.setItsAidList(itsAidsBytes);
		device.setItsAidSspList(itsAidSspsBytes);
		device.setPriorityItsAidList(itsAidPrioritiesBytes);
		device.setPrioritySspList(itsAidPrioritySspsBytes);
		device.setIdRegion(identifiedRegionBytes);
		device.setCircularRegion(circularRegionBytes);

		return device;
	}
}
