package de.fraunhofer.sit.c2x.pki.etsi_ts103097v1114.impl;

import java.io.DataInputStream;
import java.io.DataOutputStream;
import java.io.IOException;

import de.fraunhofer.sit.c2x.pki.ca.utils.WaveUtils;
import de.fraunhofer.sit.c2x.pki.etsi_ts103097v1114.impl.ValidityRestrictionTypeImpl.ValidityRestrictionType;
import de.fraunhofer.sit.c2x.pki.etsi_ts103097v1114.serializer.Internal;
import de.fraunhofer.sit.c2x.pki.etsi_ts103097v1114.visitor.EtsiVisitor;

public class ValidityRestriction extends WaveElement {

	// ---- fields ----

	@Internal(order = 0)
	private ValidityRestrictionType type;

	@Internal(order = 1)
	private Time32 endValidity;

	@Internal(order = 2)
	private Time32 startValidity;

	@Internal(order = 3)
	private Duration duration;

	@Internal(order = 4)
	private GeographicRegion region;

	@Internal(order = 5)
	private Opaque data;

	// ---- fields ----

	// ---- constructors ----

	public ValidityRestriction() {
	}
	
	public ValidityRestriction(GeographicRegion region) {
		this.type = ValidityRestrictionType.REGION;
		this.region = region;
	}
	
	public ValidityRestriction(Time32 endValidity) {
		this.type = ValidityRestrictionType.TIME_END;
		this.endValidity = endValidity;
	}
	
	public ValidityRestriction(Time32 startValidity, Time32 endValidity) {
		this.type = ValidityRestrictionType.TIME_START_AND_END;
		this.startValidity = startValidity;
		this.endValidity = endValidity;
	}
	
	public ValidityRestriction(Time32 startValidity, Duration duration) {
		this.type = ValidityRestrictionType.TIME_START_AND_DURATION;
		this.startValidity = startValidity;
		this.duration = duration;
	}

	public ValidityRestriction(DataInputStream in) throws IOException {
		type = ValidityRestrictionTypeImpl.getInstance().getEnumType(
				in.readByte());
		if (type == ValidityRestrictionType.TIME_END) {
			endValidity = new Time32(in);
		} else if (type == ValidityRestrictionType.TIME_START_AND_END) {
			startValidity = new Time32(in);
			endValidity = new Time32(in);
		} else if (type == ValidityRestrictionType.TIME_START_AND_DURATION) {
			startValidity = new Time32(in);
			duration = new Duration(in);
		} else if (type == ValidityRestrictionType.REGION) {
			region = new GeographicRegion(in);
		} else {
			data = new Opaque(in);
		}
	}

	// ---- accept ----

	public <T> T accept(EtsiVisitor<T> visitor) {
		return visitor.visit(this);
	}

	// ---- getter ----

	public ValidityRestrictionType getType() {
		return this.type;
	}

	public Time32 getStartValidity() {
		return this.startValidity;
	}

	public Time32 getEndValidity() {
		return this.endValidity;
	}

	public Duration getDuration() {
		return this.duration;
	}

	public GeographicRegion getRegion() {
		return this.region;
	}

	public Opaque getData() {
		return this.data;
	}

	// ---- setter ----

	public void setType(ValidityRestrictionType type) {
		this.type = type;
	}

	public void setStartValidity(Time32 startValidity) {
		this.startValidity = startValidity;
	}

	public void setEndValidity(Time32 endValidity) {
		this.endValidity = endValidity;
	}

	public void setDuration(Duration duration) {
		this.duration = duration;
	}

	public void setRegion(GeographicRegion region) {
		this.region = region;
	}

	public void setData(Opaque data) {
		this.data = data;
	}

	@Override
	public int writeData(DataOutputStream out) throws IOException {
		if (type == null)
			throw new IllegalArgumentException();

		int written = ValidityRestrictionTypeImpl.getInstance().writeData(out,
				type);
		if (type == ValidityRestrictionType.TIME_END) {
			if (endValidity == null) throw new IllegalArgumentException();
			written += WaveUtils.writeWave(out, endValidity);
		} else if (type == ValidityRestrictionType.TIME_START_AND_END) {
			if (endValidity == null || startValidity == null) throw new IllegalArgumentException();
			written += WaveUtils.writeWave(out, startValidity, endValidity);
		} else if (type == ValidityRestrictionType.TIME_START_AND_DURATION) {
			if (duration == null || startValidity == null) throw new IllegalArgumentException();
			written += WaveUtils.writeWave(out, startValidity, duration);
		} else if (type == ValidityRestrictionType.REGION) {
			if (region == null) throw new IllegalArgumentException();
			written += WaveUtils.writeWave(out, region);
		} else {
			if (data == null) throw new IllegalArgumentException();
			written += WaveUtils.writeWave(out, data);
		}
		return written;
	}
}