/**
 * 
 */
package de.fraunhofer.sit.c2x.pki.ca.validator.region.utils;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Set;

import de.fraunhofer.sit.c2x.pki.ca.validator.region.shape.DoubleVector2d;
import de.fraunhofer.sit.c2x.pki.ca.validator.region.shape.LineSegment;
import de.fraunhofer.sit.c2x.pki.ca.validator.region.shape.Polygon;
import de.fraunhofer.sit.c2x.pki.ca.validator.region.utils.LinearMath.StraightLine;
import de.fraunhofer.sit.c2x.pki.ca.validator.region.utils.PolygonPoint.PolygonPointType;

/**
 * @author Daniel Quanz (daniel.quanz@sit.fraunhofer.de)
 * 
 */
public class PolygonUtils {

	public static Set<PolygonPoint> findIntersectionPoints(Polygon polygon1, Polygon polygon2) {
		Set<PolygonPoint> intersectionPoints = new HashSet<PolygonPoint>();
		for (int i = 0; i < polygon1.getEdges().size(); i++) {
			for (int j = 0; j < polygon2.getEdges().size(); j++) {
				DoubleVector2d vector2d = LinearMath.intersection(polygon1.getEdges().get(i), polygon2
						.getEdges().get(j));
				if (vector2d != null) {

					PolygonPoint pp = new PolygonPoint(vector2d, PolygonPointType.INTERSECTION);
					if (!intersectionPoints.contains(pp)) {
						polygon1.insert(pp, i + 1);
						polygon2.insert(new PolygonPoint(vector2d, PolygonPointType.INTERSECTION), j + 1);
						intersectionPoints.add(pp);
						i = 0;
						j = 0;
					}
				}
			}
		}
		return intersectionPoints;
	}

	public static Set<PolygonPoint> markInterestingPoints(Polygon polygon1, Polygon polygon2, boolean union) {

		Set<PolygonPoint> enteringPointsP1 = new HashSet<PolygonPoint>();
		Set<PolygonPoint> intersectionPoints = findIntersectionPoints(polygon1, polygon2);

		// inside outside points
		markInOutsidePoints(polygon1, polygon2);
		markInOutsidePoints(polygon2, polygon1);

		markEnteringExitPoints(polygon1, polygon2, union, intersectionPoints, enteringPointsP1);

		return enteringPointsP1;

	}

	/**
	 * @param polygon1
	 * @param polygon2
	 * @param union
	 * @param intersectionPoints
	 * @param enteringPointsP1
	 */
	private static void markEnteringExitPoints(Polygon polygon1, Polygon polygon2, boolean union,
			Set<PolygonPoint> intersectionPoints, Set<PolygonPoint> enteringPointsP1) {
		for (PolygonPoint polygonPoint : intersectionPoints) {
			PolygonPoint next1 = getNextPoint(polygonPoint, polygon1);
			PolygonPointType nextType = next1.getType();
			int indexP1 = polygon1.getLinkedPoints().indexOf(polygonPoint);
			int indexP2 = polygon2.getLinkedPoints().indexOf(polygonPoint);
			PolygonPointType type1, type2;
			if (nextType == PolygonPointType.OUTSIDE) {
				type1 = (union) ? PolygonPointType.ENTERING : PolygonPointType.EXIT;
				type2 = (union) ? PolygonPointType.EXIT : PolygonPointType.ENTERING;
			} else {
				type1 = (union) ? PolygonPointType.EXIT : PolygonPointType.ENTERING;
				type2 = (union) ? PolygonPointType.ENTERING : PolygonPointType.EXIT;
			}

			polygon1.getLinkedPoints().get(indexP1).setType(type1);
			polygon2.getLinkedPoints().get(indexP2).setType(type2);
			if (type1 == PolygonPointType.ENTERING)
				enteringPointsP1.add(polygonPoint);
		}
	}

	/**
	 * @param polygon1
	 * @param polygon2
	 */
	private static void markInOutsidePoints(Polygon polygon1, Polygon polygon2) {
		for (PolygonPoint pp : polygon1.getLinkedPoints()) {
			if (pp.getType() == PolygonPointType.BOUNDARY) {
				if (PolygonUtils.jordanTest(pp, polygon2, false)) {
					pp.setType(PolygonPointType.INSIDE);
				} else {
					pp.setType(PolygonPointType.OUTSIDE);
				}
			}
		}
	}

	/**
	 * @param pp
	 * @param polygon2
	 * @param includeBorder
	 * @return
	 */
	public static boolean jordanTest(PolygonPoint pp, Polygon polygon2, boolean includeBorder) {

		boolean in = false;

		for (LineSegment segement : polygon2.getEdges()) {
			if (segement.isOn(pp))
				return includeBorder;

			Double ppX = pp.getX();
			Double ppY = pp.getY();
			Double startY = segement.getStart().getY();
			Double endY = segement.getEnd().getY();

			boolean betweenY = (startY > ppY) != (endY > ppY);
			if (betweenY) {
				DoubleVector2d intersection = LinearMath.intersection(new StraightLine(pp,
						new DoubleVector2d(1, 0)), (StraightLine) segement);

				boolean rightCross = intersection != null && ppX < intersection.getX();
				if (rightCross)
					in = !in;
			}
		}
		return in;

	}

	private static Polygon[] clip(Polygon poly1, Polygon poly2, boolean unionFlag) {

		Polygon p = new Polygon(poly1);
		Polygon q = new Polygon(poly2);

		Set<PolygonPoint> enteringp = markInterestingPoints(p, q, unionFlag);

		Iterator<PolygonPoint> it = enteringp.iterator();
		List<Polygon> polygons = new ArrayList<Polygon>();

		List<PolygonPoint> enteringPoints = new ArrayList<PolygonPoint>(enteringp);

		// No intersection
		if (enteringp.isEmpty()) {
			return new Polygon[] { poly1, poly2 };
		}

		for (PolygonPoint polygonPoint : enteringPoints) {

			System.out.println("--> " + polygonPoint);
		}
		// REMOVE this code, if holes are supported
		if (enteringp.size() > 1) {
			throw new IllegalStateException("Polygon holes do not supported yet.");
		}

		for (int i = 0; i < enteringPoints.size(); i++) {
			List<PolygonPoint> union = new ArrayList<PolygonPoint>();
			PolygonPoint first = it.next();
			traverse(union, first, first, p, q, enteringPoints);

			Polygon poly = new Polygon(union);
			polygons.add(poly);

		}

		return polygons.toArray(new Polygon[polygons.size()]);
	}

	public static Polygon[] union(Polygon poly1, Polygon poly2) {

		return clip(poly1, poly2, true);
	}

	public static Polygon[] union(Polygon[] poly1) {

		// Set<Polygon> queue = new HashSet<Polygon>();
		// for (int i = 0; i < poly1.length; i++) {
		// for (int j = i + 1; j < poly1.length; j++) {
		//
		// Polygon[] union = PolygonUtils.union(poly1[i], poly1[j]);
		// if (union.length == 1) {
		// queue.add(union[0]);
		// if (j + 1 < poly1.length) {
		// Polygon[] rest = new Polygon[poly1.length - (j + 1)];
		// System.arraycopy(poly1, j + 1, rest, 0, rest.length);
		// queue.addAll(Arrays.asList(rest));
		// }
		// return union(queue.toArray(new Polygon[queue.size()]));
		// } else {
		// queue.add(poly1[j]);
		// }
		//
		// }
		// queue.add(poly1[i]);
		// }

		Polygon[] copy = new Polygon[poly1.length];
		System.arraycopy(poly1, 0, copy, 0, copy.length);
		
		List<Polygon> notMerged = new ArrayList<Polygon>(Arrays.asList(copy));
		List<Polygon> merged = new ArrayList<Polygon>();
		unionHelper(notMerged, merged);

		return merged.toArray(new Polygon[merged.size()]);
	}

	private static void unionHelper(List<Polygon> notMerged, List<Polygon> merged) {

		if (notMerged.size() == 0)
			return;
		else {
			Polygon current = notMerged.remove(0);
			for (int i = 0; i < notMerged.size(); i++) {
				Polygon[] tmpUnion = PolygonUtils.union(current, notMerged.get(i));
				if (tmpUnion.length == 1) {
					notMerged.set(i, tmpUnion[0]);
					unionHelper(notMerged, merged);
					return;
				}
			}
			merged.add(current);
			unionHelper(notMerged, merged);
			return;

		}
	}

	public static Polygon[] cut(Polygon poly1, Polygon poly2) {

		return clip(poly1, poly2, false);
	}

	public static Polygon[] difference(Polygon poly1, Polygon poly2) {

		Polygon p = new Polygon(poly1);
		Polygon q = new Polygon(poly2);

		Set<PolygonPoint> enteringPoints = markInterestingPoints(p, q, true);

		Iterator<PolygonPoint> it = enteringPoints.iterator();
		List<Polygon> polys = new ArrayList<Polygon>();
		while (it.hasNext()) {
			PolygonPoint entering = it.next();
			List<PolygonPoint> diff = new ArrayList<PolygonPoint>();
			traverseDiff(diff, entering, entering, poly1);
			polys.add(new Polygon(diff));
		}
		return polys.toArray(new Polygon[polys.size()]);
	}

	public static PolygonPoint getNextPoint(PolygonPoint pp, Polygon poly) {
		List<PolygonPoint> lst = poly.getLinkedPoints();
		return lst.get((lst.indexOf(pp) + 1) % lst.size());
	}

	public static boolean equals(Polygon poly1, Polygon poly2) {

		if (poly1 == null && poly2 == null)
			return true;
		if (poly1 == null)
			return false;
		if (poly2 == null)
			return false;
		if (poly1.getClass() != poly2.getClass())
			return false;
		if (poly1.getLinkedPoints().size() != poly2.getLinkedPoints().size())
			return false;
		if (poly1.getLinkedPoints().size() < 3)
			return false;
		if (poly1.getHoles().size() != poly2.getHoles().size())
			return false;

		int toCheck = poly2.getLinkedPoints().indexOf(poly1.getLinkedPoints().get(0));
		for (int i = 0; i < poly1.getLinkedPoints().size(); i++) {
			PolygonPoint pp1 = poly1.getLinkedPoints().get(i);
			PolygonPoint pp2 = poly2.getLinkedPoints().get(toCheck);
			if (!pp1.equals(pp2)) {
				return false;
			}
			toCheck = poly2.getLinkedPoints().indexOf(getNextPoint(pp2, poly2));
		}

		return true;
	}

	private static void traverseDiff(List<PolygonPoint> diff, PolygonPoint first, PolygonPoint pp,
			Polygon poly) {
		diff.add(new PolygonPoint(pp));

		PolygonPoint next;
		while ((next = getNextPoint(pp, poly)).getType() == PolygonPointType.OUTSIDE) {
		}

		if (next.equals(first)) {
			return;
		} else {
			traverseDiff(diff, first, next, poly);
		}
	}

	private static void traverse(List<PolygonPoint> union, PolygonPoint first, PolygonPoint pp, Polygon poly,
			Polygon other, List<PolygonPoint> enteringp) {

		union.add(new PolygonPoint(pp));
		if (enteringp.contains(pp)) {
			enteringp.remove(pp);
		}
		PolygonPoint next = getNextPoint(pp, poly);

		if (next.equals(first)) {
			return;
		}

		if (next.getType() == PolygonPointType.EXIT) {
			traverse(union, first, next, other, poly, enteringp);
		} else {
			traverse(union, first, next, poly, other, enteringp);
		}
	}

}
