/**
 * 
 */
package de.fraunhofer.sit.c2x.pki.ca.validator.region.utils;

import de.fraunhofer.sit.c2x.pki.ca.validator.region.shape.DoubleVector2d;
import de.fraunhofer.sit.c2x.pki.ca.validator.region.shape.LineSegment;
import de.fraunhofer.sit.c2x.pki.ca.validator.region.shape.Matrix;
import de.fraunhofer.sit.c2x.pki.ca.validator.region.shape.Vector2d;

/**
 * @author Daniel Quanz (daniel.quanz@sit.fraunhofer.de)
 * 
 */
public class LinearMath {

	public static class StraightLine {
		private final DoubleVector2d supportVector;
		private final DoubleVector2d directionVector;

		/**
		 * 
		 */
		public StraightLine(DoubleVector2d supportVector, DoubleVector2d directionVector) {
			this.supportVector = supportVector;
			this.directionVector = directionVector;
		}

		/**
		 * @return the supportVector
		 */
		public DoubleVector2d getSupportVector() {
			return supportVector;
		}

		/**
		 * @return the directionVector
		 */
		public DoubleVector2d getDirectionVector() {
			return directionVector;
		}

		/**
		 * @param pointToCheck
		 * @return
		 */
		public boolean isOn(DoubleVector2d pointToCheck) {

			// ax+by+c=0
			DoubleVector2d qr_normale = getDirectionVector().crossProduct();
			Double a = qr_normale.getX();
			Double b = qr_normale.getY();
			Double c = -(a * getSupportVector().getX() + b * getSupportVector().getY());

			// calculates ax+bx+c and rounds on 8 digits
			Double e = Convert.round(a * pointToCheck.getX() + b * pointToCheck.getY() + c, 8);

			return e == 0.0;
		}
	}

	/**
	 * 
	 * @param first
	 * @param second
	 * @return
	 */
	public static DoubleVector2d intersection(StraightLine first, StraightLine second) {

		Matrix matrix = Matrix.createFromValues(first.getDirectionVector().getX(), -second
				.getDirectionVector().getX(), (second.getSupportVector().getX() - first.getSupportVector()
				.getX()), first.getDirectionVector().getY(), -second.getDirectionVector().getY(), (second
				.getSupportVector().getY() - first.getSupportVector().getY()));
		Double[] solution = matrix.solve();
		if (solution == null)
			return null;

		return first.getSupportVector().add(
				new DoubleVector2d(solution[0] * first.getDirectionVector().getX(), solution[0]
						* first.getDirectionVector().getY()));
	}

	public static DoubleVector2d intersection(LineSegment first, LineSegment second) {

		DoubleVector2d solution = intersection((StraightLine) first, (StraightLine) second);
		if (solution == null)
			return null;

		boolean inRange = true;
		inRange &= isInRange(solution.getX(), Math.min(first.getStart().getX(), first.getEnd().getX()),
				Math.max(first.getStart().getX(), first.getEnd().getX()));
		inRange &= isInRange(solution.getY(), Math.min(first.getStart().getY(), first.getEnd().getY()),
				Math.max(first.getStart().getY(), first.getEnd().getY()));
		inRange &= isInRange(solution.getX(), Math.min(second.getStart().getX(), second.getEnd().getX()),
				Math.max(second.getStart().getX(), second.getEnd().getX()));
		inRange &= isInRange(solution.getY(), Math.min(second.getStart().getY(), second.getEnd().getY()),
				Math.max(second.getStart().getY(), second.getEnd().getY()));
		return (!inRange) ? null : solution;

	}

	/**
	 * Checks if a value is between a given range <code>[min, max]</code>
	 * 
	 * @param value
	 *            the value to check
	 * @param min
	 *            the minimal value
	 * @param max
	 *            the maximal value
	 * @return true, if <code>value</code> is within <code>[min, max]</code>
	 */
	public static boolean isInRange(double value, double min, double max) {
		return value >= min && value <= max;
	}

	/**
	 * Returns the distance between a {@link Vector2d} and a
	 * {@link StraightLine}
	 * 
	 * @param vector
	 * @param line
	 * @return the distance as a positive {@link Double}
	 */
	public static double distanceBetween(DoubleVector2d vector, StraightLine line) {
		if (vector == null || line == null) {
			throw new IllegalArgumentException("<vector> and <line> may not be null");
		}

		DoubleVector2d n = line.getDirectionVector().crossProduct();

		Double c = n.dotProduct(line.getSupportVector());
		Double d = (n.dotProduct(vector) - c) / n.length();

		return Math.abs(d);
	}

	/**
	 * Returns the distance between two {@link Vector2d}<code>s</code>
	 * 
	 * @param vector1
	 * @param vector2
	 * @return the distance as a positive {@link Double}
	 */
	public static double distanceBetween(DoubleVector2d vector1, DoubleVector2d vector2) {
		if (vector1 == null || vector2 == null) {
			throw new IllegalArgumentException("<vector1> and <vector2> may not be null");
		}
		return Math.abs(vector1.sub(vector2).length());
	}

	/**
	 * Returns the distance between a {@link Vector2d} and a {@link LineSegment}
	 * 
	 * @param vector
	 * @param segment
	 * 
	 * @see #distanceBetween(DoubleVector2d, StraightLine)
	 * @return the distance as a positive {@link Double}
	 */
	public static double distanceBetween(DoubleVector2d vector, LineSegment segment) {
		if (vector == null || segment == null) {
			throw new IllegalArgumentException("<vector> and <segment> may not be null");
		}
		if (segment.getDirectionVector().dotProduct(vector.sub(segment.getEnd())) > 0)
			return distanceBetween(vector, segment.getEnd());
		if (segment.getDirectionVector().mul(-1.0).dotProduct(vector.sub(segment.getStart())) > 0)
			return distanceBetween(vector, segment.getStart());

		return distanceBetween(vector, (StraightLine) segment);
	}

}
