/**
 * 
 */
package de.fraunhofer.sit.c2x.pki.ca.validator.region.shape;

/**
 * @author Daniel Quanz (daniel.quanz@sit.fraunhofer.de)
 * 
 */
public class DoubleVector2d extends Vector2d<Double> {

	/**
	 * Constructs a 2D-vector (x, y) with double values
	 * 
	 * @param x
	 *            the x value
	 * @param y
	 *            the y value
	 * @throws IllegalArgumentException
	 *             if <code>x</code> and/or <code>y</code> is null
	 */
	public DoubleVector2d(Double x, Double y) {
		super(x, y);
	}

	/**
	 * Constructs a 2D-vector (x, y) with double values
	 * 
	 * @param x
	 *            the x value
	 * @param y
	 *            the y value
	 */
	public DoubleVector2d(int x, int y) {
		this((double) x, (double) y);
	}

	/**
	 * Adds a given given vector to this vector
	 * 
	 * <pre>
	 * Example:
	 * 
	 * (2, 5) + (100, 500) = (102, 505)
	 * </pre>
	 * 
	 * @param vector
	 *            a 2D-vector with {@link Double}-values
	 * @return a 2D-vector with {@link Double}-values with the result after the
	 *         operation
	 * @throws IllegalArgumentException
	 *             if <code>vector</code> is null
	 */
	public DoubleVector2d add(DoubleVector2d vector) {
		if (vector == null) {
			throw new IllegalArgumentException("<vector> may not be null.");
		}
		return new DoubleVector2d(x + vector.x, y + vector.y);
	}

	/**
	 * Subtracts a given vector from this vector
	 * 
	 * <pre>
	 * (2, 5) - (100, 500) = (-98, -495)
	 * </pre>
	 * 
	 * @param vector
	 *            a 2D-vector with {@link Double}-values
	 * @return a 2D-vector with {@link Double}-values with the result after the
	 *         operation
	 * @throws IllegalArgumentException
	 *             if <code>vector</code> is null
	 */
	public DoubleVector2d sub(DoubleVector2d vector) {
		if (vector == null) {
			throw new IllegalArgumentException("<vector> may not be null.");
		}
		return new DoubleVector2d(x - vector.x, y - vector.y);
	}

	/**
	 * Divides the vector through a given vector
	 * 
	 * <pre>
	 * (2, 5) / (100, 500) = (0.02, 0.01)
	 * </pre>
	 * 
	 * @param vector
	 *            a 2D-vector with {@link Double}-values
	 * @return a 2D-vector with {@link Double}-values with the result after the
	 *         operation
	 * @throws IllegalArgumentException
	 *             if <code>vector</code> is null
	 * @throws ArithmeticException
	 *             if the <code>vector.x</code> and/or the <code>vector.y</code>
	 *             is/are zero
	 */
	public DoubleVector2d div(DoubleVector2d vector) {
		if (vector == null) {
			throw new IllegalArgumentException("<vector> may not be null.");
		}
		if (vector.x == 0 || vector.y == 0) {
			throw new ArithmeticException(String.format("Divide-by-zero! [<vector>=%s]", vector));
		}
		return new DoubleVector2d(x / vector.x, y / vector.y);
	}

	/**
	 * Multiplies the vector with a given vector
	 * 
	 * <pre>
	 * (2, 5) * (100, 500) = (200, 2500)
	 * </pre>
	 * 
	 * @param vector
	 *            a 2D-vector with {@link Double}-values
	 * @return a 2D-vector with {@link Double}-values with the result after the
	 *         operation
	 * @throws IllegalArgumentException
	 *             if <code>vector</code> is null
	 */
	public DoubleVector2d mul(DoubleVector2d vector) {
		if (vector == null) {
			throw new IllegalArgumentException("<vector> may not be null.");
		}
		return new DoubleVector2d(x * vector.x, y * vector.y);
	}

	/**
	 * Multiplies the vector with a given factor
	 * 
	 * <pre>
	 * 2.5 * (100, 500) = (250, 1250)
	 * </pre>
	 * 
	 * @param factor
	 *            a {@link Double}-value used to multiply the vector
	 * @return a 2D-vector with {@link Double}-values with the result after the
	 *         operation
	 */
	public DoubleVector2d mul(double factor) {
		return new DoubleVector2d(x * factor, y * factor);
	}

	/**
	 * Returns the cross product of the 2D-vector
	 * 
	 * <pre>
	 * (100, -200) --> (-200, -100)
	 * </pre>
	 * 
	 * @return a 2D-vector with {@link Double}-values with the result after the
	 *         operation
	 */
	public DoubleVector2d crossProduct() {
		return new DoubleVector2d(y, -x);
	}

	/**
	 * Returns the dot product as a {@link Double} value
	 * 
	 * <pre>
	 * (100, 200) dot (5, 20) = 100*5+200*20 = 4500
	 * </pre>
	 * 
	 * @return the dot product of two vectors
	 * @throws IllegalArgumentException
	 *             if <code>vector</code> is null
	 */
	public Double dotProduct(DoubleVector2d vector) {
		if (vector == null) {
			throw new IllegalArgumentException("<vector> may not be null.");
		}
		return x * vector.x + y * vector.y;
	}

	/**
	 * Returns the length of the vector
	 * 
	 * <pre>
	 * (3, 4) --> sqrt(3 * 3 + 4 * 4) = 5
	 * </pre>
	 * 
	 * @return the length of the vector
	 */
	public Double length() {
		return StrictMath.sqrt(dotProduct(this));
	}

	/**
	 * Returns the unit vector
	 * 
	 * <pre>
	 * (3, 4) --> (0.6, 0.8)
	 * </pre>
	 * 
	 * @return the unit vector
	 */
	public DoubleVector2d unitVector() {
		return mul(1 / length());
	}

	/**
	 * Return the angle (arc degree) between a given vector and this vector
	 * 
	 * <pre>
	 *   ^   
	 *   |  
	 *   | 
	 *   | angle
	 *   +--------->
	 * 
	 * (0, 1) angle (1, 0) = 90
	 * </pre>
	 * 
	 * @param vector
	 *            a 2D-vector with {@link Double}-values
	 * @return the angle (arc degree) between two vectors
	 * @throws IllegalArgumentException
	 *             if <code>vector</code> is null
	 */
	public Double angleBetween(DoubleVector2d vector) {
		if (vector == null) {
			throw new IllegalArgumentException("<vector> may not be null.");
		}
		double rad2deg = 57.29577951308232; // simplification of 180.0 /
											// Math.PI;
		return rad2deg * (StrictMath.acos(unitVector().dotProduct(vector.unitVector())));
	}

}
