package de.fraunhofer.sit.c2x.pki.ca.validator.pseudonym;

import java.util.ArrayList;
import java.util.List;
import java.util.Set;

import javax.inject.Inject;

import org.apache.log4j.Logger;

import de.fraunhofer.sit.c2x.pki.ca.core.exceptions.HandlerException;
import de.fraunhofer.sit.c2x.pki.ca.core.logging.InjectLogger;
import de.fraunhofer.sit.c2x.pki.ca.provider.ProviderException;
import de.fraunhofer.sit.c2x.pki.ca.provider.entities.DBInterval;
import de.fraunhofer.sit.c2x.pki.ca.provider.entities.DBIntervalId;
import de.fraunhofer.sit.c2x.pki.ca.provider.interfaces.ConfigProvider;
import de.fraunhofer.sit.c2x.pki.ca.provider.interfaces.PseudonymPolicyEnforcementProvider;
import de.fraunhofer.sit.c2x.pki.ca.utils.ByteUtils;
import de.fraunhofer.sit.c2x.pki.ca.validator.pseudonym.PPEUtils.IntervalDataPair;
import de.fraunhofer.sit.c2x.pki.ca.validator.pseudonym.intervaltree.Interval;

/**
 * @author Daniel Quanz (daniel.quanz@sit.fraunhofer.de)
 */
public class PseudonymPolicyEnforcement {

	@InjectLogger
	private Logger logger;

	private final PseudonymPolicyEnforcementProvider provider;
	private final ConfigProvider config;

	@Inject
	public PseudonymPolicyEnforcement(PseudonymPolicyEnforcementProvider provider, ConfigProvider config) {
		this.provider = provider;
		this.config = config;
	}

	/**
	 * 
	 * @param object
	 * @return affected slots
	 * @throws ProviderException
	 */
	public PPEResult validate(AVRContainer object) throws ProviderException {

		DBInterval[] fromDB = provider.getSlotsByTimeRange(object.getCertId(), object.getRegion(),
				object.getStartTime(), object.getEndTime());
		Interval<Integer> reqeuestedInterval = new Interval<Integer>(object.getStartTime(), object
				.getEndTime(), (int) object.getPseudonyms());

		if (logger != null && logger.isDebugEnabled()) {
			logger.debug("Try to fill intervals for certId = " + ByteUtils.getHex(object.getCertId()));
		}

		int ppn = config.getInt("maxPseudonymParallel", 10);
		long plt = config.getLong("maxPseudonymLifetime", 24 * 3600) * 1000;
		long ppt = config.getLong("maxPseudonymPreloadingTime", 365 * 24 * 3600) * 1000;
		List<Interval<IntervalDataPair>> lst = make(fromDB);
		PPEResult result = PPEUtils.computeIntervals(ppn, plt, ppt, lst, reqeuestedInterval, logger);
		if (logger != null && logger.isDebugEnabled()) {

			if (result.getNumber() > 0) {
				logger.debug("Insert " + result.getSet().size() + " new intervals with " + result.getNumber()
						+ " certificates into DB");
				for (Interval<Integer> interval : result.getSet()) {
					logger.debug("DB: " + interval.getData() + " " + interval.getStartTime() + "-"
							+ interval.getEndTime());
				}
			} else {
				logger.debug("All time slots for requested time " + object.getStartTime() + "-"
						+ object.getEndTime() + ", certId=" + ByteUtils.getHex(object.getCertId())
						+ " are full!");
			}

		}
		DBInterval[] slots = intToDBInt(result.getSet(), object);
		provider.save(slots);

		return result;

	}

	private DBInterval[] intToDBInt(Set<Interval<Integer>> set, AVRContainer avr) {
		List<DBInterval> ret = new ArrayList<>();
		for (Interval<Integer> interval : set) {
			DBInterval dbint = new DBInterval();
			DBIntervalId id = new DBIntervalId(avr.getCertId(), avr.getRegion(), interval.getStart(),
					interval.getEnd());
			dbint.setId(id);
			dbint.setActivatedCerts(0l);
			dbint.setReservedCerts(interval.getData());
			dbint.setRequestHash(avr.getRequestHash());
			ret.add(dbint);
		}

		return ret.toArray(new DBInterval[ret.size()]);
	}

	private List<Interval<IntervalDataPair>> make(DBInterval[] fromDB) {
		List<Interval<IntervalDataPair>> ret = new ArrayList<>();
		for (DBInterval interval : fromDB) {
			ret.add(new Interval<PPEUtils.IntervalDataPair>(interval.getId().getStartTime(), interval
					.getId().getEndTime(), new IntervalDataPair(interval.getActivatedCerts(), interval
					.getReservedCerts())));
		}
		return ret;
	}

	// DBInterval[] dbSlots = provider.getSlotsByTimeRange(object.getCertId(),
	// object.getRegion(),
	// object.getStartTime(), object.getEndTime());
	//
	// DBInterval input = new DBInterval(object.getStartTime(),
	// object.getEndTime(), object.getPseudonyms());
	// input.setCertId(object.getCertId());
	// input.setRegion(object.isCircularRegion(), object.getRegion());
	// DBInterval[] result = PeriodValidatorUtils.merge(dbSlots, input,
	// config.getInt("maxPseudonymParallel", 10),
	// config.getInt("maxPseudonymLifetime", 1209600) * 1000,
	// config.getLong("maxPseudonymPreloadingTime", 63072000) * 1000l);
	//
	// for (DBInterval slot : result) {
	// logger.debug(slot);
	// }
	//
	// provider.save(result);
	//
	// List<DBInterval> tmp = new ArrayList<>();
	// for (DBInterval slot : result) {
	// if (slot.getReservedCerts() > 0) {
	// tmp.add(slot);
	// }
	// }
	// DBInterval[] affected = tmp.toArray(new DBInterval[tmp.size()]);
	// requests.put(ByteUtils.getHex(object.getRequestHash()), affected);
	// if (logger != null && logger.isDebugEnabled())
	// logger.debug(affected.length +
	// " timeslots affected and linkted to requestHash="
	// + ByteUtils.getHex(object.getRequestHash()));
	// return affected;

	public int activate(byte[] requestHash) throws HandlerException {
		if (logger != null && logger.isDebugEnabled())
			logger.debug("Try to activate time slots linked to requestHash=" + ByteUtils.getHex(requestHash));
		try {
			return provider.activate(requestHash);
		} catch (ProviderException e) {
			logger.error(e);
			throw new HandlerException("Unable to activate authorized pseudonym certs: ",e);
		}
	}

}
