package de.fraunhofer.sit.c2x.pki.ca.provider.jdbc;

import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Timestamp;
import java.util.ArrayList;
import java.util.Calendar;
import java.util.List;
import java.util.TimeZone;

import javax.inject.Inject;

import de.fraunhofer.sit.c2x.pki.ca.provider.ProviderException;
import de.fraunhofer.sit.c2x.pki.ca.provider.entities.Constants;
import de.fraunhofer.sit.c2x.pki.ca.provider.entities.DBInterval;
import de.fraunhofer.sit.c2x.pki.ca.provider.entities.DBIntervalId;
import de.fraunhofer.sit.c2x.pki.ca.provider.interfaces.ConfigProvider;
import de.fraunhofer.sit.c2x.pki.ca.provider.interfaces.PseudonymPolicyEnforcementProvider;

/**
 * @author Daniel Quanz (daniel.quanz@sit.fraunhofer.de)
 */
public class JDBCPseudonymPolicyEnforcementProvider extends AbstractMysqlConnection implements
		PseudonymPolicyEnforcementProvider {

	@Inject
	private ConfigProvider configProvider;

	public DBInterval[] getSlotsByTimeRange(byte[] certId, byte[] region, long start, long end)
			throws ProviderException {
		String sql = String
				.format("SELECT * FROM %s WHERE %s = ? AND ((%s BETWEEN ? AND ?) OR (%s BETWEEN ? AND ?) OR (%s <= ? AND %s >= ?)) ORDER BY %s ASC",
						Constants.SLOT__TABLE, Constants.SLOT__ID__CERT_ID, Constants.SLOT__ID__START_TIME,
						Constants.SLOT__ID__END_TIME, Constants.SLOT__ID__START_TIME,
						Constants.SLOT__ID__END_TIME, Constants.SLOT__ID__START_TIME);
		Connection con = getConnection();
		PreparedStatement st = null;
		ResultSet result = null;
		try {
			st = con.prepareStatement(sql);
			st.setBytes(1, certId);
			st.setLong(2, start);
			st.setLong(3, end);
			st.setLong(4, start);
			st.setLong(5, end);
			st.setLong(6, start);
			st.setLong(7, end);

			// System.out.println(st);
			// System.out.println(sql);

			st.execute();
			result = st.getResultSet();

			List<DBInterval> tmp = new ArrayList<>();
			while (result.next()) {
				DBInterval info = new DBInterval();
				DBIntervalId id = new DBIntervalId(result.getBytes(Constants.SLOT__ID__CERT_ID),
						result.getBytes(Constants.SLOT__ID__REGION),
						result.getLong(Constants.SLOT__ID__START_TIME),
						result.getLong(Constants.SLOT__ID__END_TIME));
				info.setId(id);
				info.setRequestHash(result.getBytes(Constants.SLOT__COL__REQUEST_HASH));
				info.setCreationTime(result.getTimestamp(Constants.SLOT__COL__CREATION_TIME));
				info.setActivatedCerts(result.getInt(Constants.SLOT__COL__ACTIVATED_CERTS));
				info.setReservedCerts(result.getInt(Constants.SLOT__COL__RESERVED));
				tmp.add(info);
			}
			return tmp.toArray(new DBInterval[tmp.size()]);
		} catch (SQLException e) {
			e.printStackTrace();
			throw new ProviderException("Can not get all time slots from DB", e);
		} finally {
			closeStatement(st);
			closeConnection(con);
			closeResultSet(result);
		}
	}

	public int activate(byte[] requestHash) throws ProviderException {
		String sql = String.format("SELECT SUM(%s) AS total, startTime, end_time FROM %s WHERE %s = ?;",
				Constants.SLOT__COL__RESERVED, Constants.SLOT__TABLE, Constants.SLOT__COL__REQUEST_HASH);
		Connection con = getConnection();
		PreparedStatement st = null;

		String sql2 = String.format("UPDATE %s SET %s = %s + %s, %s = 0 WHERE %s = ?", Constants.SLOT__TABLE,
				Constants.SLOT__COL__ACTIVATED_CERTS, Constants.SLOT__COL__ACTIVATED_CERTS,
				Constants.SLOT__COL__RESERVED, Constants.SLOT__COL__RESERVED,
				Constants.SLOT__COL__REQUEST_HASH);

		ResultSet result = null;
		try {
			st = con.prepareStatement(sql);
			st.setBytes(1, requestHash);
			st.execute();
			result = st.getResultSet();
			int ret = 0;
			if (result.first()) {
				ret = result.getInt("total");
				long plt = configProvider.getLong("maxPseudonymLifetime", 24 * 3600);
				long mul = (result.getLong(Constants.SLOT__ID__END_TIME)
						- result.getLong(Constants.SLOT__ID__START_TIME) + 1000)
						/ (plt * 1000);
				ret *= mul;
			} else {
				throw new ProviderException();
			}
			st = con.prepareStatement(sql2);
			st.setBytes(1, requestHash);
			st.execute();
			return ret;

		} catch (SQLException e) {
			throw new ProviderException("Can not handlde SQL query " + st, e);
		} finally {
			closeStatement(st);
			closeConnection(con);
			closeResultSet(result);
		}
	}

	public void save(DBInterval[] slots) throws ProviderException {
		// ON DUPLICATE KEY UPDATE %s=VALUES(%s), %s=VALUES(%s), %s=VALUES(%s);
		for (DBInterval slot : slots) {

			Connection con = getConnection();
			PreparedStatement st = null;
			String sql = "";
			try {
				sql = String.format(
						"INSERT INTO %s (%s, %s,%s, %s, %s, %s, %s, %s) VALUES (?,?,?,?,?,?,?,?);",
						Constants.SLOT__TABLE, Constants.SLOT__ID__CERT_ID, Constants.SLOT__ID__REGION,
						Constants.SLOT__ID__START_TIME, Constants.SLOT__ID__END_TIME,
						Constants.SLOT__COL__REQUEST_HASH, Constants.SLOT__COL__CREATION_TIME,
						Constants.SLOT__COL__ACTIVATED_CERTS, Constants.SLOT__COL__RESERVED);
				st = con.prepareStatement(sql);
				st.setBytes(1, slot.getId().getCertId());
				st.setBytes(2, slot.getId().getRegion());
				st.setLong(3, slot.getId().getStartTime());
				st.setLong(4, slot.getId().getEndTime());
				st.setBytes(5, slot.getRequestHash());
				st.setTimestamp(6, new Timestamp((System.currentTimeMillis() / 1000) * 1000),
						Calendar.getInstance(TimeZone.getTimeZone("UTC")));
				st.setLong(7, slot.getActivatedCerts());
				st.setLong(8, slot.getReservedCerts());

				// } else if (slot.getType() == Type.UPDATE_ALL) {
				// System.out.println("UP");
				// sql =
				// String.format("UPDATE %s SET %s = ?, %s = ? WHERE %s = ? AND %s = ? AND %s = ?;",
				// Constants.SLOT__TABLE, Constants.SLOT__COL__END_TIME,
				// Constants.SLOT__COL__RESERVED, Constants.SLOT__ID__CERT_ID,
				// Constants.SLOT__ID__REGION, Constants.SLOT__ID__START_TIME);
				// st = con.prepareStatement(sql);
				// st.setTimestamp(1, slot.getEndTime());
				// st.setLong(2, slot.getReservedCerts());
				// st.setBytes(3, slot.getId().getCertId());
				// st.setBytes(4, slot.getId().getRegion());
				// st.setTimestamp(5, slot.getId().getStartTime());
				// } else if (slot.getType() == Type.ACTIVATE) {
				// System.out.println("ACTIVATE");
				// sql =
				// String.format("UPDATE %s SET %s = ?, %s = ? WHERE %s = ? AND %s = ? AND %s = ?;",
				// Constants.SLOT__TABLE, Constants.SLOT__COL__ACTIVATED_CERTS,
				// Constants.SLOT__COL__RESERVED, Constants.SLOT__ID__CERT_ID,
				// Constants.SLOT__ID__REGION, Constants.SLOT__ID__START_TIME);
				// st = con.prepareStatement(sql);
				// st.setLong(1, slot.getActivatedCerts() +
				// slot.getReservedCerts());
				// st.setInt(2, 0);
				// st.setBytes(3, slot.getId().getCertId());
				// st.setBytes(4, slot.getId().getRegion());
				// st.setTimestamp(5, slot.getId().getStartTime());
				// }

				st.execute();
				// System.out.println("----");
				// System.out.println(st);
				// System.out.println("Execute");
				// System.out.println("----");
			} catch (SQLException e) {
				try {
					sql = String.format("UPDATE %s SET %s=%s+? WHERE %s=? AND %s=? AND %s=? AND %s=?;",
							Constants.SLOT__TABLE, Constants.SLOT__COL__RESERVED,
							Constants.SLOT__COL__RESERVED, Constants.SLOT__ID__CERT_ID,
							Constants.SLOT__ID__REGION, Constants.SLOT__ID__START_TIME,
							Constants.SLOT__ID__END_TIME);
					st = con.prepareStatement(sql);
					st.setLong(1, slot.getReservedCerts());
					st.setBytes(2, slot.getId().getCertId());
					st.setBytes(3, slot.getId().getRegion());
					st.setLong(4, slot.getId().getStartTime());
					st.setLong(5, slot.getId().getEndTime());
					st.execute();
				} catch (SQLException e1) {
					throw new ProviderException("");
				}
			} finally {
				closeStatement(st);
				closeConnection(con);
			}

		}

	}

	@Override
	public String getName() {
		return "Pseudonym Policy Enforcement Provider";
	}

}
