package de.fraunhofer.sit.c2x.pki.ca.provider.jdbc;

import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.ArrayList;

import org.apache.commons.codec.binary.Hex;

import com.mysql.jdbc.Statement;

import de.fraunhofer.sit.c2x.pki.ca.provider.ProviderException;
import de.fraunhofer.sit.c2x.pki.ca.provider.entities.CaCertificate;
import de.fraunhofer.sit.c2x.pki.ca.provider.entities.Constants;
import de.fraunhofer.sit.c2x.pki.ca.provider.interfaces.CaInfoProvider;

public class JDBCCaInfoProvider extends AbstractMysqlConnection implements CaInfoProvider<CaCertificate> {

	private long uptime;
	private ArrayList<CaCertificate> certs;

	@Override
	public void startService() {
		super.startService();
		uptime = System.currentTimeMillis();
	}

	@Override
	public String getName() {
		return "CA info provider";
	}

	@Override
	public long getUptime() {
		return System.currentTimeMillis() - uptime;
	}

	@Override
	public CaCertificate getCaCertificate() throws ProviderException {
		if (certs == null || certs.size() < 1)
			getAll(0, 1000);

		if (certs.size() > 0) {
			return certs.get(0);
		} else
			throw new ProviderException("No certificate found");
	}

	@Override
	public CaCertificate getCaCertificate(byte[] certId) throws ProviderException {
		if (certs == null || certs.size() < 1)
			getAll(0, 1000);

		for (CaCertificate cert : this.certs) {
			if (Hex.encodeHexString(certId).equals(Hex.encodeHexString(cert.getCertId()))) {
				return cert;
			}

		}
		throw new ProviderException("CA cert with ID " + Hex.encodeHexString(certId) + " not found in DB");
	}

	@Override
	public void setCaCertificate(CaCertificate cert) throws ProviderException {
		if (cert.getUsageOrder() == 0) {
			String sql = String.format("SELECT MAX(%s) AS max_usage_order FROM %s;",
					Constants.CA_CERTIFICATE__COL__USAGE_ORDER, Constants.CA_CERTIFICATE__TABLE);
			Connection con = getConnection();
			PreparedStatement st = null;
			ResultSet result = null;
			try {
				st = con.prepareStatement(sql);
				st.execute();
				result = st.getResultSet();

				if (!result.first()) {
				} else {
					int maxUsageOrder = result.getInt("max_usage_order");
					cert.setUsageOrder(maxUsageOrder + 1);
				}
			} catch (SQLException e) {
				logger.debug("invalid sql statement" + sql, e);
			}
		}

		String sql = String.format(
				"INSERT INTO %s (%s, %s, %s, %s, %s, %s, %s, %s) VALUES(?,?,?,?,?,?,?,?);",
				Constants.CA_CERTIFICATE__TABLE, Constants.CA_CERTIFICATE__COL__ID,
				Constants.CA_CERTIFICATE__COL__CERTIFICATE, Constants.CA_CERTIFICATE__COL__CREATION_TIME,
				Constants.CA_CERTIFICATE__COL__EXPIRATION_TIME,
				Constants.CA_CERTIFICATE__COL__SIGNER_CERT_ID, Constants.CA_CERTIFICATE__COL__START_TIME,
				Constants.CA_CERTIFICATE__COL__SUBJECT_NAME, Constants.CA_CERTIFICATE__COL__USAGE_ORDER);
		Connection con = getConnection();
		PreparedStatement st = null;
		try {

			st = con.prepareStatement(sql, Statement.NO_GENERATED_KEYS);
			st.setBytes(1, cert.getCertId());
			st.setBytes(2, cert.getCertificate());
			st.setTimestamp(3, cert.getCreationTime());
			st.setTimestamp(4, cert.getExpirationTime());
			st.setBytes(5, cert.getSignerCertId());
			st.setTimestamp(6, cert.getStartTime());
			st.setString(7, cert.getSubjectName());
			st.setInt(8, cert.getUsageOrder());
			st.execute();

			certs = null;
			sort();
		} catch (SQLException e) {
			logger.error("Setting CA Certificate failed! Cannot prepare sql statement", e);
		} finally {
			closeStatement(st);
			closeConnection(con);
		}

	}

	@Override
	public CaCertificate[] getAll(int offset, int limit) throws ProviderException {
		if (certs == null || certs.size() < 1)
			certs = new ArrayList<>();
		else
			return certs.toArray(new CaCertificate[certs.size()]);

		// TODO Select the right CA Certificate if multiple CA Certificates are
		// available! Delete revoked and outdated/invalid certificates from DB

		String sql = String.format("SELECT * FROM %s ORDER BY %s ASC LIMIT %d, %d",
				Constants.CA_CERTIFICATE__TABLE, Constants.CA_CERTIFICATE__COL__USAGE_ORDER, offset, limit);
		Connection con = getConnection();
		PreparedStatement st = null;
		ResultSet result = null;
		try {
			st = con.prepareStatement(sql);
			st.execute();
			result = st.getResultSet();

			while (result.next()) {

				CaCertificate cert = new CaCertificate();
				cert.setCertId(result.getBytes(Constants.CA_CERTIFICATE__COL__ID));
				cert.setCertificate(result.getBytes(Constants.CA_CERTIFICATE__COL__CERTIFICATE));
				cert.setCreationTime(result.getTimestamp(Constants.CA_CERTIFICATE__COL__CREATION_TIME));
				cert.setExpirationTime(result.getTimestamp(Constants.CA_CERTIFICATE__COL__EXPIRATION_TIME));
				cert.setSignerCertId(result.getBytes(Constants.CA_CERTIFICATE__COL__SIGNER_CERT_ID));
				cert.setStartTime(result.getTimestamp(Constants.CA_CERTIFICATE__COL__START_TIME));
				cert.setSubjectName(result.getString(Constants.CA_CERTIFICATE__COL__SUBJECT_NAME));
				cert.setUsageOrder(result.getInt(Constants.CA_CERTIFICATE__COL__USAGE_ORDER));

				certs.add(cert);
			}
			return certs.toArray(new CaCertificate[certs.size()]);
		} catch (SQLException e) {
			logger.error(e);
			throw new ProviderException(e);
		} finally {
			closeStatement(st);
			closeConnection(con);
			closeResultSet(result);
		}
	}

	@Override
	public void sortUp(byte[] certID) throws ProviderException {
		switchOrder(certID, true, false);
	}

	@Override
	public void sortDown(byte[] certID) throws ProviderException {
		switchOrder(certID, false, true);
	}

	private void sort() throws ProviderException {
		String sql = String.format("SELECT * FROM %s ORDER BY %s ASC", Constants.CA_CERTIFICATE__TABLE,
				Constants.CA_CERTIFICATE__COL__USAGE_ORDER);
		Connection con = getConnection();
		PreparedStatement st = null;
		PreparedStatement stUpdate = null;
		ResultSet result = null;
		try {
			st = con.prepareStatement(sql);
			st.execute();
			result = st.getResultSet();
			int order = 0;
			while (result.next()) {
				byte[] certId = result.getBytes(Constants.CA_CERTIFICATE__COL__ID);
				String sqlUpdate = String.format("UPDATE %s SET %s = ? WHERE %s = ?",
						Constants.CA_CERTIFICATE__TABLE, Constants.CA_CERTIFICATE__COL__USAGE_ORDER,
						Constants.CA_CERTIFICATE__COL__ID);
				stUpdate = con.prepareStatement(sqlUpdate);
				stUpdate.setInt(1, order++);
				stUpdate.setBytes(2, certId);
				stUpdate.execute();
			}
			certs = null;
		} catch (SQLException e) {
			logger.error(e);
			throw new ProviderException(e);
		} finally {
			closeStatement(st);
			closeStatement(stUpdate);
			closeConnection(con);
			closeResultSet(result);
		}
	}

	private void switchOrder(byte[] certID, boolean up, boolean down) throws ProviderException {
		String sql = String.format("SELECT %s, %s FROM %s WHERE %s = ?;", Constants.CA_CERTIFICATE__COL__ID,
				Constants.CA_CERTIFICATE__COL__USAGE_ORDER, Constants.CA_CERTIFICATE__TABLE,
				Constants.CA_CERTIFICATE__COL__ID);
		Connection con = getConnection();
		PreparedStatement st = null;
		PreparedStatement stToChange = null;
		PreparedStatement stUpdate = null;
		ResultSet result = null;
		ResultSet resultToChange = null;
		try {
			st = con.prepareStatement(sql);
			st.setBytes(1, certID);
			st.execute();
			result = st.getResultSet();

			if (!result.first()) {
				// do nothing
			} else {
				int oldOrder = result.getInt(Constants.CA_CERTIFICATE__COL__USAGE_ORDER);
				Integer newOrder = null;
				if (up && !down)
					newOrder = oldOrder - 1;
				else if (down && !up)
					newOrder = oldOrder + 1;
				else
					throw new ProviderException("Invalid order operation");

				String sqlToChange = String.format("SELECT %s, %s FROM %s WHERE %s = ?",
						Constants.CA_CERTIFICATE__COL__ID, Constants.CA_CERTIFICATE__COL__USAGE_ORDER,
						Constants.CA_CERTIFICATE__TABLE, Constants.CA_CERTIFICATE__COL__USAGE_ORDER);
				stToChange = con.prepareStatement(sqlToChange);
				stToChange.setInt(1, newOrder);
				stToChange.execute();
				resultToChange = stToChange.getResultSet();

				if (!resultToChange.first()) {
					sort();
				} else {
					byte[] certIdToChange = resultToChange.getBytes(Constants.CA_CERTIFICATE__COL__ID);
					// update old entry
					String sqlUpdate = String.format("UPDATE %s SET %s = ? WHERE %s = ?",
							Constants.CA_CERTIFICATE__TABLE, Constants.CA_CERTIFICATE__COL__USAGE_ORDER,
							Constants.CA_CERTIFICATE__COL__ID);
					stUpdate = con.prepareStatement(sqlUpdate);
					stUpdate.setInt(1, oldOrder);
					stUpdate.setBytes(2, certIdToChange);
					stUpdate.execute();

					// update new entry
					sqlUpdate = String.format("UPDATE %s SET %s = ? WHERE %s = ?",
							Constants.CA_CERTIFICATE__TABLE, Constants.CA_CERTIFICATE__COL__USAGE_ORDER,
							Constants.CA_CERTIFICATE__COL__ID);
					stUpdate = con.prepareStatement(sqlUpdate);
					stUpdate.setInt(1, newOrder);
					stUpdate.setBytes(2, certID);
					stUpdate.execute();

					sort();
				}
			}
		} catch (SQLException e) {
			logger.debug("invalid sql statement");
			throw new ProviderException("Failed to sort entries: " + e);
		} finally {
			closeStatement(st);
			closeConnection(con);
			closeResultSet(result);
		}
	}
}
