package de.fraunhofer.sit.c2x.pki.ca.provider.jdbc;

import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Timestamp;
import java.util.ArrayList;
import java.util.List;

import org.apache.commons.codec.binary.Hex;

import com.google.inject.Inject;

import de.fraunhofer.sit.c2x.pki.ca.provider.ProviderException;
import de.fraunhofer.sit.c2x.pki.ca.provider.entities.AuthorizedDevice;
import de.fraunhofer.sit.c2x.pki.ca.provider.entities.Constants;
import de.fraunhofer.sit.c2x.pki.ca.provider.entities.PublicKey;
import de.fraunhofer.sit.c2x.pki.ca.provider.interfaces.AuthorizedDeviceProvider;

/**
 * @author Daniel Quanz (daniel.quanz@sit.fraunhofer.de)
 */
public class JDBCAuthorizedDeviceProvider extends AbstractMysqlConnection implements AuthorizedDeviceProvider {

	@Inject
	public JDBCAuthorizedDeviceProvider() {
	}

	@Override
	public String getName() {
		return "Authorized device provider";
	}

	@Override
	public boolean save(AuthorizedDevice device) throws ProviderException {
		String sql = String
				.format("INSERT INTO %s (%s, %s, %s, %s, %s, %s, %s, %s, %s, %s, %s) VALUES (?,?,?,?,?,?,?,?,?,?,?);",
						Constants.AUTHORIZED_DEVICES__TABLE, Constants.AUTHORIZED_DEVICES__COL__ID,
						Constants.AUTHORIZED_DEVICES__COL__CREATION_TIME,
						Constants.PUBLICKEY__COL__ALGORITHM, Constants.PUBLICKEY__COL__KEY,
						Constants.AUTHORIZED_DEVICES__COL__ITS_AID_LIST,
						Constants.AUTHORIZED_DEVICES__COL__ITS_AID_SSP_LIST,
						Constants.AUTHORIZED_DEVICES__COL__PRIORITY_ITS_AID_LIST,
						Constants.AUTHORIZED_DEVICES__COL__PRIORITY_SSP_LIST,
						Constants.AUTHORIZED_DEVICES__COL__SUBJECT_ASSURANCE,
						Constants.AUTHORIZED_DEVICES__COL__IDENTIFIED_REGION,
						Constants.AUTHORIZED_DEVICES__COL__CIRCULAR_REGION);
		Connection con = getConnection();
		PreparedStatement st = null;
		try {
			st = con.prepareStatement(sql);
			st.setBytes(1, device.getCanonicalId());
			st.setTimestamp(2, new Timestamp(System.currentTimeMillis()));
			st.setString(3, device.getPublicKey().getPublicKeyAlgorithm());
			st.setBytes(4, device.getPublicKey().getPublicKey());
			st.setBytes(5, device.getItsAidList());
			st.setBytes(6, device.getItsAidSspList());
			st.setBytes(7, device.getPriorityItsAidList());
			st.setBytes(8, device.getPrioritySspList());
			st.setBytes(9, device.getSubjectAssurance());
			st.setBytes(10, device.getIdRegion());
			st.setBytes(11, device.getCircularRegion());
			st.execute();
			return true;
		} catch (SQLException e) {
			logger.error("AuthorizedDevice not stored: " + e.getMessage());
			throw new ProviderException(e);
		} finally {
			closeStatement(st);
			closeConnection(con);
		}
	}

	@Override
	public AuthorizedDevice get(byte[] canonicalId) throws ProviderException {

		String sql = String.format("SELECT * FROM %s WHERE %s = ?;", Constants.AUTHORIZED_DEVICES__TABLE,
				Constants.AUTHORIZED_DEVICES__COL__ID);
		Connection con = getConnection();
		PreparedStatement st = null;
		ResultSet result = null;
		try {
			st = con.prepareStatement(sql);
			st.setBytes(1, canonicalId);
			st.execute();
			result = st.getResultSet();

			if (!result.first()) {
				logger.warn("Entry not found");
			} else {
				PublicKey publicKey = new PublicKey();
				publicKey.setPublicKey(result.getBytes(Constants.PUBLICKEY__COL__KEY));
				publicKey.setPublicKeyAlgorithm(result.getString(Constants.PUBLICKEY__COL__ALGORITHM));

				AuthorizedDevice info = new AuthorizedDevice();
				info.setCanonicalId(result.getBytes(Constants.AUTHORIZED_DEVICES__COL__ID));
				info.setCreationTime(result.getTimestamp(Constants.AUTHORIZED_DEVICES__COL__CREATION_TIME));
				info.setPublicKey(publicKey);

				info.setItsAidList(result.getBytes(Constants.AUTHORIZED_DEVICES__COL__ITS_AID_LIST));
				info.setItsAidSspList(result.getBytes(Constants.AUTHORIZED_DEVICES__COL__ITS_AID_SSP_LIST));
				info.setPriorityItsAidList(result
						.getBytes(Constants.AUTHORIZED_DEVICES__COL__PRIORITY_ITS_AID_LIST));
				info.setPrioritySspList(result.getBytes(Constants.AUTHORIZED_DEVICES__COL__PRIORITY_SSP_LIST));
				info.setSubjectAssurance(result
						.getBytes(Constants.AUTHORIZED_DEVICES__COL__SUBJECT_ASSURANCE));
				info.setIdRegion(result.getBytes(Constants.AUTHORIZED_DEVICES__COL__IDENTIFIED_REGION));
				info.setCircularRegion(result.getBytes(Constants.AUTHORIZED_DEVICES__COL__CIRCULAR_REGION));
				info.setDeactivated(result.getBoolean(Constants.AUTHORIZED_DEVICES__COL__DEACTIVATED));

				return info;
			}
		} catch (SQLException e) {
			logger.error(e);
			throw new ProviderException(e);
		} finally {
			closeStatement(st);
			closeConnection(con);
			closeResultSet(result);
		}
		return null;
	}

	@Override
	public AuthorizedDevice[] getAll(int offset, int limit) throws ProviderException {
		String sql = String.format("SELECT * FROM %s ORDER BY %s DESC LIMIT %d, %d",
				Constants.AUTHORIZED_DEVICES__TABLE, Constants.AUTHORIZED_DEVICES__COL__CREATION_TIME,
				offset, limit);
		Connection con = getConnection();
		PreparedStatement st = null;
		ResultSet result = null;
		try {
			st = con.prepareStatement(sql);
			st.execute();
			result = st.getResultSet();

			List<AuthorizedDevice> tmp = new ArrayList<>();
			while (result.next()) {

				PublicKey publicKey = new PublicKey();
				publicKey.setPublicKey(result.getBytes(Constants.PUBLICKEY__COL__KEY));
				publicKey.setPublicKeyAlgorithm(result.getString(Constants.PUBLICKEY__COL__ALGORITHM));

				AuthorizedDevice info = new AuthorizedDevice();
				info.setCanonicalId(result.getBytes(Constants.AUTHORIZED_DEVICES__COL__ID));
				info.setCreationTime(result.getTimestamp(Constants.AUTHORIZED_DEVICES__COL__CREATION_TIME));
				info.setPublicKey(publicKey);
				info.setDeactivated(result.getInt(Constants.AUTHORIZED_DEVICES__COL__DEACTIVATED));

				info.setItsAidList(result.getBytes(Constants.AUTHORIZED_DEVICES__COL__ITS_AID_LIST));
				info.setItsAidSspList(result.getBytes(Constants.AUTHORIZED_DEVICES__COL__ITS_AID_SSP_LIST));
				info.setPriorityItsAidList(result
						.getBytes(Constants.AUTHORIZED_DEVICES__COL__PRIORITY_ITS_AID_LIST));
				info.setPrioritySspList(result.getBytes(Constants.AUTHORIZED_DEVICES__COL__PRIORITY_SSP_LIST));
				info.setSubjectAssurance(result
						.getBytes(Constants.AUTHORIZED_DEVICES__COL__SUBJECT_ASSURANCE));
				info.setIdRegion(result.getBytes(Constants.AUTHORIZED_DEVICES__COL__IDENTIFIED_REGION));
				info.setCircularRegion(result.getBytes(Constants.AUTHORIZED_DEVICES__COL__CIRCULAR_REGION));

				tmp.add(info);
			}
			return tmp.toArray(new AuthorizedDevice[tmp.size()]);
		} catch (SQLException e) {
			logger.error(e);
			throw new ProviderException(e);
		} finally {
			closeStatement(st);
			closeConnection(con);
			closeResultSet(result);
		}
		// return new AuthorizedDevice[0];
	}

	@Override
	public boolean deactivateDevice(byte[] canonicalId) throws ProviderException {
		String sql = String.format("UPDATE %s SET %s = ? WHERE %s = ?;", Constants.AUTHORIZED_DEVICES__TABLE,
				Constants.AUTHORIZED_DEVICES__COL__DEACTIVATED, Constants.AUTHORIZED_DEVICES__COL__ID);
		Connection con = getConnection();
		PreparedStatement st = null;
		ResultSet result = null;
		try {
			st = con.prepareStatement(sql);
			st.setInt(1, 1);
			st.setBytes(2, canonicalId);
			int numberOfAffectedRows = st.executeUpdate();

			if (numberOfAffectedRows <= 0) {
				logger.warn("Deactivation failed! No DB row affected.");
				throw new ProviderException("Deactivation failed!");
			} else if (numberOfAffectedRows == 1) {
				logger.debug("Device " + new String(Hex.encodeHex(canonicalId))
						+ " successfully deactivated!");
				return true;
			} else {
				logger.error("Deactivation error! More than one row affected.");
				return true;
			}
		} catch (SQLException e) {
			logger.error(e);
			throw new ProviderException(e);
		} finally {
			closeStatement(st);
			closeConnection(con);
			closeResultSet(result);
		}
	}

	@Override
	public boolean reactivateDevice(byte[] canonicalId) throws ProviderException {
		String sql = String.format("UPDATE %s SET %s = ? WHERE %s = ?;", Constants.AUTHORIZED_DEVICES__TABLE,
				Constants.AUTHORIZED_DEVICES__COL__DEACTIVATED, Constants.AUTHORIZED_DEVICES__COL__ID);
		Connection con = getConnection();
		PreparedStatement st = null;
		ResultSet result = null;
		try {
			st = con.prepareStatement(sql);
			st.setInt(1, 0);
			st.setBytes(2, canonicalId);
			int numberOfAffectedRows = st.executeUpdate();

			if (numberOfAffectedRows <= 0) {
				logger.warn("Re-activation failed! No DB row affected.");
				throw new ProviderException("Re-activation failed!");
			} else if (numberOfAffectedRows == 1) {
				logger.debug("Device " + new String(Hex.encodeHex(canonicalId))
						+ " successfully re-activated!");
				return true;
			} else {
				logger.error("Re-activation error! More than one row affected.");
				return true;
			}
		} catch (SQLException e) {
			logger.error(e);
			throw new ProviderException(e);
		} finally {
			closeStatement(st);
			closeConnection(con);
			closeResultSet(result);
		}
	}

}
