package de.fraunhofer.sit.c2x.pki.ca.provider.jdbc;

import java.sql.Connection;
import java.sql.Driver;
import java.sql.DriverManager;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Statement;

import org.apache.commons.dbcp.ConnectionFactory;
import org.apache.commons.dbcp.PoolableConnectionFactory;
import org.apache.commons.dbcp.PoolingDriver;
import org.apache.commons.pool.impl.GenericObjectPool;
import org.apache.log4j.Logger;

import com.google.inject.Inject;
import com.google.inject.name.Named;

import de.fraunhofer.sit.c2x.pki.ca.core.interfaces.Service;
import de.fraunhofer.sit.c2x.pki.ca.core.logging.InjectLogger;

public abstract class AbstractMysqlConnection implements Service {

	// public static final String DRIVER_POOL_NAME = "PRESERVE_LTCA_Database";
	// public static final String DRIVER_POOL_URL = "jdbc:apache:commons:dbcp:"
	// + DRIVER_POOL_NAME;
	// private static final String DRIVER = "com.mysql.jdbc.Driver";
	// private static final String BASE_URL = "jdbc:mysql://localhost/";
	// private static final String DATABASE_NAME = "preserve_ltca";
	// private static final String URL = BASE_URL + DATABASE_NAME;

	@InjectLogger
	protected Logger logger;

	private boolean started;

	private String driver;

	private String databaseUrl;

	private String database_user;

	private String database_password;

	private GenericObjectPool connectionPool;

	public AbstractMysqlConnection() {
	}

	@Override
	public void startService() {
		Class<?> mysqlDriverClass;
		try {
			mysqlDriverClass = Class.forName(driver);
			Driver mysqlDriver = (Driver) mysqlDriverClass.newInstance();
			DriverManager.registerDriver(mysqlDriver);
		} catch (Exception e) {
			logger.error(e);
		}
		logger.debug("MySQL driver loaded and registered");

		connectionPool = new GenericObjectPool(null);
		connectionPool.setTestOnBorrow(true);

		ConnectionFactory connectionFactory = new MySQLConnectionFactory(
				databaseUrl, database_user, database_password);
		boolean autoCommit = true;
		PoolableConnectionFactory poolableConnectionFactory = new PoolableConnectionFactory(
				connectionFactory, connectionPool, null, "SELECT 1;", false,
				autoCommit);
		connectionPool.setFactory(poolableConnectionFactory);
		PoolingDriver driver = new PoolingDriver();
		driver.registerPool(
				databaseUrl.substring(databaseUrl.lastIndexOf("/") + 1),
				connectionPool);
		logger.debug("Register ConnectionPool: "
				+ databaseUrl.substring(databaseUrl.lastIndexOf("/") + 1));
		started = true;
	}

	@Override
	public void stopService() {
		try {
			connectionPool.close();
			started = false;
		} catch (Exception e) {
			logger.error(e);
		}

	}

	@Override
	public boolean isStarted() {
		return started;
	}

	public Connection getConnection() {
		Connection con = null;
		try {
			con = DriverManager.getConnection("jdbc:apache:commons:dbcp:"
					+ databaseUrl.substring(databaseUrl.lastIndexOf("/") + 1));
		} catch (SQLException e) {
			logger.error("Connection fails", e);
		}
		;
		return con;
	}

	protected void closeResultSet(ResultSet rs) {
		try {
			if (rs != null)
				rs.close();
		} catch (SQLException e) {
			logger.error("Failed to close ResultSet", e);
		}
	}

	protected void closeStatement(Statement st) {
		try {
			if (st != null)
				st.close();
		} catch (Exception e) {
			logger.error("Failed to close Statement", e);
		}
	}

	protected void closeConnection(Connection conn) {
		try {
			if (conn != null)
				conn.close();
		} catch (Exception e) {
			logger.error("Failed to close connection", e);
		}
	}

	/**
	 * Required because of some class-loader troubles when using MySQL together
	 * with DBCP (each in it's own bundle).
	 */
	private static class MySQLConnectionFactory implements ConnectionFactory {

		final String url, user, password;

		public MySQLConnectionFactory(String url, String user, String password) {
			super();
			this.url = url;
			this.user = user;
			this.password = password;
		}

		@Override
		public Connection createConnection() throws SQLException {
			return DriverManager.getConnection(url, user, password);
		}
	}

	public String getDriver() {
		return driver;
	}

	@Inject
	public void setDriver(@Named(value = "db_driver") String driver) {
		this.driver = driver;
	}

	public String getDatabaseUrl() {
		return databaseUrl;
	}

	@Inject
	public void setDatabaseUrl(@Named(value = "db_url") String databaseUrl) {
		this.databaseUrl = databaseUrl;
	}

	public String getDatabaseUser() {
		return database_user;
	}

	@Inject
	public void setDatabaseUser(@Named(value = "db_user") String database_user) {
		this.database_user = database_user;
	}

	public String getDatabasePassword() {
		return database_password;
	}

	@Inject
	public void setDatabasePassword(
			@Named(value = "db_password") String database_password) {
		this.database_password = database_password;
	}

}
