package de.fraunhofer.sit.c2x.pki.ca.module.webserver.servlets;

import java.io.IOException;
import java.security.cert.X509Certificate;
import java.security.interfaces.RSAPrivateKey;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

import org.apache.commons.codec.DecoderException;
import org.apache.commons.codec.binary.Hex;
import org.apache.log4j.Logger;
import org.eclipse.jetty.security.authentication.SessionAuthentication;

import com.google.inject.Inject;

import de.fraunhofer.sit.c2x.pki.ca.core.logging.InjectLogger;
import de.fraunhofer.sit.c2x.pki.ca.module.webserver.interfaces.HtmlProvider;
import de.fraunhofer.sit.c2x.pki.ca.provider.ProviderException;
import de.fraunhofer.sit.c2x.pki.ca.provider.entities.KnownCA;
import de.fraunhofer.sit.c2x.pki.ca.provider.interfaces.KnownCaProvider;

/**
 * A simple servlet that outputs the certificate content of issued certificates.
 * Following arguments are allowed per GET
 * 
 * <ul>
 * <li>CertId8 or CertId10: <b>/cacert?cert=HEX or Base64</b></li>
 * </ul>
 * 
 * @author Norbert Bissmeyer (norbert.bissmeyer@sit.fraunhofer.de)
 * 
 */
public class ShowKnownCaServlet extends AbstractServlet {

	private static final long serialVersionUID = 1L;
	protected static final String URL = "/showKnwonCAs";
	protected static final String TITLE = "Show and delete known CAs";
	public static final String DATE_PATTERN_LOG = "yyyy-MM-dd HH:mm:ss";

	@InjectLogger
	private Logger logger;
	private KnownCaProvider knownCaProvider;

	@Inject
	public ShowKnownCaServlet(HtmlProvider htmlProvider, KnownCaProvider knownCaProvider) {
		super(htmlProvider);
		this.knownCaProvider = knownCaProvider;
	}

	@Override
	public String getUrl() {
		return URL;
	}

	@Override
	public String getTitle() {
		return TITLE;
	}

	@Override
	protected String contentHtml(HttpServletRequest req, HttpServletResponse resp) {
		String userInfo = "";
		StringBuilder sb = new StringBuilder();

		// delete known CA
		if (req.getParameter("deleteCertId") != null && req.getParameter("deleteCertId").length() > 0) {

			try {
				String certIdString = req.getParameter("deleteCertId");
				byte[] certId = new byte[0];
				if (certIdString != null && certIdString.length() > 0)
					certId = Hex.decodeHex(certIdString.toCharArray());

				// get user from http session
				SessionAuthentication user = (SessionAuthentication) req.getSession().getAttribute(
						SessionAuthentication.__J_AUTHENTICATED);
				logger.info("USER: User " + user.getUserIdentity().getUserPrincipal().getName().toString()
						+ " deleted known CA with certID: " + Hex.encodeHexString(certId));

				knownCaProvider.removeKnownCA(certId);

				userInfo = "<div id=\"ackbox\">CA with certId " + Hex.encodeHexString(certId)
						+ " removed from database</div>";

			} catch (DecoderException e) {
				userInfo = "<div id=\"errorbox\">Unable to decode known CA certID</div>";
			} catch (ProviderException e) {
				userInfo = "<div id=\"errorbox\">Unable to delete known CA certID</div>";
			}
		}

		KnownCA[] knownCAs = new KnownCA[0];
		try {
			knownCAs = knownCaProvider.getAll(0, 1000);
		} catch (ProviderException e) {
			logger.error("Unable to known CAs from DB");
			userInfo = "<div id=\"errorbox\">Unable to known CAs from DB</div>";
		}

		if (!userInfo.isEmpty()) {
			sb.append("<br/><br/>" + userInfo);
		}

		// output list
		sb.append("<table>\n");
		for (KnownCA info : knownCAs) {

			sb.append("<tr><form action=\"\" onsubmit=\"return askBeforeDelete('Do you really want delete the known CA: "
					+ Hex.encodeHexString(info.getCertId())
					+ "');\" method=\"get\">\n<td style=\"width:450px;\"><ul>\n");
			sb.append("<li>CA type: " + info.getCaType() + "</li>\n");
			sb.append("<li>Cert ID: " + Hex.encodeHexString(info.getCertId()) + "</li>\n");
			//sb.append("<li>Host IP address: " + info.getHostInfo().getAdress() + "</li>\n");
			//sb.append("<li>HTTP/S port: " + info.getHostInfo().getHttpPort() + "</li>\n");
			//sb.append("<li>UDP port: " + info.getHostInfo().getUdpPort() + "</li>\n");
			sb.append("<li>Webservice WSDL URL: <a href=\"" + info.getHostInfo().getWsdlUrl() + "\">"
					+ info.getHostInfo().getWsdlUrl() + "</a></li>\n");
			if (!info.getX509ClientCertString().isEmpty()) {
				try {
					X509Certificate x509Cert = info.getX509ClientCert();
					sb.append("<li>X.509 client cert: " + x509Cert.getSubjectDN().toString() + "</li>\n");
				} catch (IOException e) {
					sb.append("<li>X.509 client cert: now available</li>\n");
				}
				try {
					RSAPrivateKey key = info.getRsaClientKey();
					if (key != null)
						sb.append("<li>Client private key algorithm and format: " + key.getAlgorithm() +", "+ key.getFormat()+ "</li>\n");
					else
						sb.append("<li>RSA client private key: now available</li>\n");
				} catch (IOException e) {
					sb.append("<li>RSA client private key: now available</li>\n");
				}
			}

			sb.append("</ul></td><td style=\"width:150px; vertical-align:middle;\"><input type=\"hidden\" name=\"deleteCertId\" value=\""
					+ Hex.encodeHexString(info.getCertId())
					+ "\"/><input type=\"submit\" value=\"Delete\" /></td><td style=\"width:150px;\"></td>\n</form>\n</tr>\n");

		}
		sb.append("</table>\n");

		return sb.toString();
	}

	@Override
	public boolean isProtected() {
		return true;
	}

	@Override
	public String[] getAllowedRoles() {
		return new String[] { UserRole.ADMINISTRATOR.toString() };
	}

	@Override
	public Category getCategory() {
		return Category.SETUP;
	}
}
