package de.fraunhofer.sit.c2x.pki.ca.module.webserver.servlets;

import java.text.SimpleDateFormat;
import java.util.Date;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

import org.apache.commons.codec.binary.Hex;
import org.apache.commons.lang3.StringEscapeUtils;
import org.apache.log4j.Logger;
import org.eclipse.jetty.security.authentication.SessionAuthentication;

import com.google.inject.Inject;

import de.fraunhofer.sit.c2x.pki.ca.core.logging.InjectLogger;
import de.fraunhofer.sit.c2x.pki.ca.module.webserver.interfaces.HtmlProvider;
import de.fraunhofer.sit.c2x.pki.ca.provider.ProviderException;
import de.fraunhofer.sit.c2x.pki.ca.provider.entities.Authenticator;
import de.fraunhofer.sit.c2x.pki.ca.provider.entities.HttpUser;
import de.fraunhofer.sit.c2x.pki.ca.provider.interfaces.AuthenticatorProvider;
import de.fraunhofer.sit.c2x.pki.ca.provider.interfaces.HttpUserProvider;

/**
 * A simple servlet that outputs the certificate content of issued certificates.
 * Following arguments are allowed per GET
 * 
 * <ul>
 * <li>CertId8 or CertId10: <b>/cacert?cert=HEX or Base64</b></li>
 * </ul>
 * 
 * @author Norbert Bissmeyer (norbert.bissmeyer@sit.fraunhofer.de)
 * 
 */
public class RegisteredAuthenticatorServlet extends AbstractServlet {

	private static final long serialVersionUID = 1L;
	protected static final String URL = "/showRegisteredAuthenticators";
	protected static final String TITLE = "Show registered Authenticators";
	public static final String DATE_PATTERN_LOG = "yyyy-MM-dd HH:mm:ss";

	@InjectLogger
	private Logger logger;
	private AuthenticatorProvider authenticatorProvider;
	private HttpUserProvider httpUserProvider;

	@Inject
	public RegisteredAuthenticatorServlet(HtmlProvider htmlProvider,
			AuthenticatorProvider authenticatorProvider, HttpUserProvider httpUserProvider) {
		super(htmlProvider);
		this.authenticatorProvider = authenticatorProvider;
		this.httpUserProvider = httpUserProvider;
	}

	@Override
	public String getUrl() {
		return URL;
	}

	@Override
	public String getTitle() {
		return TITLE;
	}

	@Override
	protected String contentHtml(HttpServletRequest req, HttpServletResponse resp) {
		String userInfo = "";
		StringBuilder sb = new StringBuilder();
		Authenticator[] registeredAuthenticators = new Authenticator[0];

		// get user from http session
		SessionAuthentication user = (SessionAuthentication) req.getSession().getAttribute(
				SessionAuthentication.__J_AUTHENTICATED);
		HttpUser httpUser = httpUserProvider.getUser(user.getUserIdentity().getUserPrincipal().getName()
				.toString());

		try {
			// if administrator logged in then all authenticators are shown. Otherwise
			// only the authenticators that are added by the logged in http-user
			if (httpUser.getRole().equals(UserRole.ADMINISTRATOR.toString()))
				registeredAuthenticators = authenticatorProvider.getAll(0, Integer.MAX_VALUE);
			else
				registeredAuthenticators = authenticatorProvider.getRegisteredByUser(user.getUserIdentity()
						.getUserPrincipal().getName().toString());
		} catch (ProviderException e) {
			logger.warn(e);
			userInfo = "<div id=\"errorbox\">Unable to get registered authenticators from DB</div>";
		}
		if (userInfo != "") {
			sb.append("<br/><br/>" + userInfo);
		} else {
			// output list of registered authenticators
			sb.append("<table>\n");
			for (Authenticator info : registeredAuthenticators) {
				sb.append("<tr>\n<td style=\"width:450px;\">\n");
				sb.append("<ul><li>Company: " + StringEscapeUtils.escapeHtml4(info.getCompany()) + "</li>\n");
				sb.append("<li>Name: " + StringEscapeUtils.escapeHtml4(info.getName()) + "</li>\n");
				sb.append("<li id=\"email_" + StringEscapeUtils.escapeHtml4(info.getEmail()) + "\">Email: "
						+ StringEscapeUtils.escapeHtml4(info.getEmail()) + "</li>\n");
				sb.append("<li>Creator (http user): "
						+ StringEscapeUtils.escapeHtml4(info.getHttpUserId()) + "</li>\n");
				sb.append("<li>Telephone: " + StringEscapeUtils.escapeHtml4(info.getTelephone()) + "</li>\n");
				sb.append("<li>Address: " + StringEscapeUtils.escapeHtml4(info.getAddress()) + "</li>\n");
				sb.append("<li>Assurance level: " + Hex.encodeHexString(info.getSubjectAssurance())
						+ "</li>\n");
				sb.append("<li>Creation time: " + formatedDate(info.getCreationTime()) + "</li>\n");
				sb.append("</ul></td><td><form action=\"" + AuthenticatorDeletionServlet.URL
						+ "\" method=\"get\"><input type=\"hidden\" name=\"email\" value=\""
						+ info.getEmail() + "\"/><input type=\"submit\" value=\"Delete\" />\n");
				sb.append("</form></td></tr>\n");
			}
			sb.append("</table>\n");
		}

		return sb.toString();
	}

	private String formatedDate(Date date) {
		return new SimpleDateFormat(DATE_PATTERN_LOG).format(date);
	}

	@Override
	public boolean isProtected() {
		return true;
	}

	@Override
	public String[] getAllowedRoles() {
		return new String[] { UserRole.ADMINISTRATOR.toString(), UserRole.REGISTRATION_USER.toString() };
	}

	@Override
	public Category getCategory() {
		return Category.AUTHENTICATOR;
	}
}
