package de.fraunhofer.sit.c2x.pki.ca.module.webserver.servlets;

import java.io.BufferedReader;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStreamReader;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.Calendar;
import java.util.Date;
import java.util.TimeZone;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

import org.apache.commons.lang3.StringEscapeUtils;
import org.apache.log4j.Logger;

import com.google.inject.Inject;
import com.google.inject.name.Named;

import de.fraunhofer.sit.c2x.pki.ca.core.logging.InjectLogger;
import de.fraunhofer.sit.c2x.pki.ca.module.webserver.interfaces.HtmlProvider;
import de.fraunhofer.sit.c2x.pki.ca.provider.interfaces.ConfigProvider;

/**
 * A simple servlet that outputs the content of the file log/pki.log via HTTP
 * 
 * <ul>
 * <li>10 seconds: <b>/log?period=10s</b></li>
 * <li>10 minutes: <b>/log?period=10m</b></li>
 * <li>10 hours: <b>/log?period=10h</b></li>
 * </ul>
 * 
 * @author Jan Peter Stotz (jan-peter.stotz@sit.fraunhofer.de) Norbert Bissmeyer
 *         (norbert.bissmeyer@sit.fraunhofer.de)
 * 
 */
public class LogServlet extends AbstractServlet {

	/**
	 * 
	 */
	private static final long serialVersionUID = 1L;

	@InjectLogger
	private Logger logger;
	public static final String URL = "/log";
	public static final String TITLE = "Log";
	public static final String DATE_PATTERN_LOG = "yyyy-MM-dd HH:mm:ss,SSS";

	@Inject
	private ConfigProvider configProvider;

	@Inject
	public LogServlet(HtmlProvider htmlProvider) {
		super(htmlProvider);
	}

	@Inject
	@Named(value = "caName")
	private String caName;

	protected String outputLog(String period, TimeZone tz) throws IOException {
		String str = "";
		if (period == null) {
			// set default period if nothing is given
			period = configProvider.get("webpageLogDefaultTimePeriod");
		}
		Date startPeriod = getStartDate(period, tz);

		FileInputStream fin = null;
		BufferedReader br = null;
		try {
			fin = new FileInputStream(configProvider.get("log4jFile"));
			br = new BufferedReader(new InputStreamReader(fin));
			String s = null;
			if (startPeriod == null) {
				while ((s = br.readLine()) != null) {
					if (s.contains("[" + caName))
						str += StringEscapeUtils.escapeHtml4(s) + "\n";
				}
			} else {
				boolean after = false;
				SimpleDateFormat df = new SimpleDateFormat(DATE_PATTERN_LOG);
				df.setTimeZone(tz);
				while ((s = br.readLine()) != null) {
					if ((!after)) {
						boolean dateAfter = false;
						if (s.length() > 23) {
							String date = s.substring(0, 23);
							try {
								Date d = df.parse(date);
								dateAfter = d.after(startPeriod);
							} catch (ParseException e) {
								System.err.println(e);
							}
						}
						if (!dateAfter)
							continue;
						after = true;
					}
					if (s.contains("[" + caName))
						str += StringEscapeUtils.escapeHtml4(s) + "\n";
				}
			}
			return str;
		} finally {
			if (fin != null)
				fin.close();
			if (br != null)
				br.close();
		}
	}

	public static Date getStartDate(String period, TimeZone tz) {
		Date startPeriod = null;
		period = period.trim().toLowerCase();
		String pattern = "(\\d+)([mhs])";
		Pattern p = Pattern.compile(pattern);
		Matcher m = p.matcher(period);
		if (m.matches()) {
			int dur = Integer.parseInt(m.group(1));
			char unit = m.group(2).charAt(0);
			Calendar c = Calendar.getInstance(tz);
			int field = 0;
			switch (unit) {
			case 'm':
				field = Calendar.MINUTE;
				break;
			case 'h':
				field = Calendar.HOUR;
				break;
			case 's':
				field = Calendar.SECOND;
				break;
			}
			c.add(field, -dur);
			startPeriod = c.getTime();
		}
		return startPeriod;
	}

	@Override
	public String getUrl() {
		return URL;
	}

	@Override
	public String getTitle() {
		return TITLE;
	}

	@Override
	protected String contentHtml(HttpServletRequest req, HttpServletResponse resp) {
		String periodValue = configProvider.get("webpageLogDefaultTimePeriod").replaceAll("[a-z]*", "");
		String periodUnit = configProvider.get("webpageLogDefaultTimePeriod").replaceAll("[0-9]*", "");
		String period = periodValue + "" + periodUnit;
		if (req.getParameter("period") != null && req.getParameter("period").length() > 0) {
			period = req.getParameter("period");
		} else if (req.getParameter("periodValue") != null && req.getParameter("periodValue").length() > 0
				&& req.getParameter("periodUnit") != null && req.getParameter("periodUnit").length() > 0) {
			periodValue = req.getParameter("periodValue");
			periodUnit = req.getParameter("periodUnit");
			period = periodValue + "" + periodUnit;
		}
		TimeZone tz = TimeZone.getTimeZone(configProvider.get("webpageLogTimeZone"));
		StringBuilder sb = new StringBuilder();
		String logStr = "";

		// check whether raw output is requested
		if (req.getParameter("raw") != null && req.getParameter("raw").length() > 0
				&& (req.getParameter("raw").equals("true") || req.getParameter("raw").equals("1"))) {
			try {
				logStr = outputLog(period, tz);
				sb.append(logStr.replaceAll("\\n", "<br />"));
			} catch (IOException e) {
				logger.error(e);
				logStr += "<div id=\"errorbox\">Unable to read logfile: " + configProvider.get("log4jFile")
						+ "</div>\n";
			}
		} else {
			sb.append("<form name=\"logForm\" action=\"\">\n"
					+ "<div style=\"margin-bottom:10px;\">\n"
					+ "Limit the log entries of the current day (logs of past days can be requested from the CA operator): <input id=\"periodValue\" name=\"periodValue\" type=\"text\" value=\""
					+ periodValue + "\" style=\"width:50px\" maxlength=\"5\" />&nbsp;"
					+ "<select  id=\"periodUnit\" name=\"periodUnit\"><option value=\"s\"");
			if (periodUnit.equals("s"))
				sb.append(" selected=\"selected\"");
			sb.append(">seconds</option><option value=\"m\"");
			if (periodUnit.equals("m"))
				sb.append(" selected=\"selected\"");
			sb.append(">minutes</option><option value=\"h\"");
			if (periodUnit.equals("h"))
				sb.append(" selected=\"selected\"");

			try {
				logStr = outputLog(period, tz);
			} catch (IOException e) {
				logger.error(e);
				logStr += "<div id=\"errorbox\">Unable to read logfile: " + configProvider.get("log4jFile")
						+ "</div>\n";
			}

			sb.append(">hours</option></select>&nbsp;" + "<input type=\"submit\" value=\"Show log\" />\n");
			sb.append("</div>\n" + "</form>\n");
			// sb.append("Start time: " + getStartDate(period));
			sb.append("<pre>" + logStr.replaceAll("\\n", "<br />") + "</pre><a name=\"logTail\"></a>");
		}
		// str += "<script type=\"text/javascript\">\n"
		// + "location.href = \"" + url + "#logTail\";\n" + "</script>\n";
		// }
		return sb.toString();
	}

	@Override
	public boolean isProtected() {
		return true;
	}

	@Override
	public String[] getAllowedRoles() {
		return new String[] { UserRole.ADMINISTRATOR.toString(), UserRole.REGISTRATION_USER.toString(),
				UserRole.DEVELOPER.toString(), UserRole.MAINTAINER.toString(), UserRole.USER.toString() };
	}

	@Override
	public Category getCategory() {
		return Category.CA_INFO;
	}
}
