package de.fraunhofer.sit.c2x.pki.ca.module.webserver.servlets;

import java.io.IOException;
import java.io.OutputStream;
import java.text.SimpleDateFormat;
import java.util.Date;

import javax.servlet.ServletException;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

import org.apache.commons.codec.DecoderException;
import org.apache.commons.codec.binary.Hex;
import org.apache.commons.lang3.StringEscapeUtils;
import org.apache.log4j.Logger;

import com.google.inject.Inject;

import de.fraunhofer.sit.c2x.pki.ca.core.logging.InjectLogger;
import de.fraunhofer.sit.c2x.pki.ca.module.webserver.interfaces.HtmlProvider;
import de.fraunhofer.sit.c2x.pki.ca.provider.ProviderException;
import de.fraunhofer.sit.c2x.pki.ca.provider.entities.Crl;
import de.fraunhofer.sit.c2x.pki.ca.provider.entities.CrlCertificate;
import de.fraunhofer.sit.c2x.pki.ca.provider.entities.IssuedCertificateInfo;
import de.fraunhofer.sit.c2x.pki.ca.provider.entities.RevokedCertificate;
import de.fraunhofer.sit.c2x.pki.ca.provider.interfaces.IssuedCertsInfoProvider;
import de.fraunhofer.sit.c2x.pki.ca.provider.interfaces.UpdateableCrlProvider;

/**
 * A simple servlet that outputs the certificate content of issued certificates.
 * Following arguments are allowed per GET
 * 
 * <ul>
 * <li>CertId8 or CertId10: <b>/cacert?cert=HEX or Base64</b></li>
 * </ul>
 * 
 * @author Norbert Bissmeyer (norbert.bissmeyer@sit.fraunhofer.de)
 * 
 */
public class IssuedCertificatesServlet extends AbstractServlet {

	private static final long serialVersionUID = 1L;
	protected static final String URL = "/cacert";
	protected static final String TITLE = "Show issued CA certificates";
	public static final String DATE_PATTERN_LOG = "yyyy-MM-dd HH:mm:ss";

	@InjectLogger
	private Logger logger;
	protected IssuedCertsInfoProvider<IssuedCertificateInfo> issuedCertsInfoProvider;

	@Inject
	private UpdateableCrlProvider<Crl, RevokedCertificate, CrlCertificate> crlProvider;

	@Inject
	public IssuedCertificatesServlet(HtmlProvider htmlProvider,
			IssuedCertsInfoProvider<IssuedCertificateInfo> issuedCertsInfoProvider) {
		super(htmlProvider);
		this.issuedCertsInfoProvider = issuedCertsInfoProvider;
	}

	@Override
	public String getUrl() {
		return URL;
	}

	@Override
	public String getTitle() {
		return TITLE;
	}

	@Override
	protected void doGet(HttpServletRequest req, HttpServletResponse resp) throws ServletException,
			IOException {
		if (req.getParameter("submit") != null && !req.getParameter("submit").isEmpty()
				&& req.getParameter("submit").equals("Download") && req.getParameter("cert") != null
				&& !req.getParameter("cert").isEmpty() && req.getParameter("cert").length() == 16) {

			try {

				byte[] certId = Hex.decodeHex(req.getParameter("cert").toCharArray());
				IssuedCertificateInfo certInfo = issuedCertsInfoProvider.getIssuedCertificate(certId);

				resp.setContentType("application/octet-stream");
				resp.setContentLength((int) certInfo.getCertificate().length);
				resp.setHeader("Content-Disposition",
						"attachment; filename=\"ca_cert_" + Hex.encodeHexString(certInfo.getCertId()) + ".crt\"");

				OutputStream os = resp.getOutputStream();
				os.write(certInfo.getCertificate());
				os.flush();
				os.close();
			} catch (DecoderException e) {
				logger.warn("Unable to decode hex values of cert ID " + req.getParameter("cert"), e);
			} catch (ProviderException e) {
				logger.warn("Unable to get certificate for cert ID " + req.getParameter("cert"), e);
			}
		}

		super.doGet(req, resp);
	}

	@Override
	protected String contentHtml(HttpServletRequest req, HttpServletResponse resp) {
		String userInfo = "";
		StringBuilder sb = new StringBuilder();
		sb.append("<form name=\"certForm\" action=\"\" method=\"get\">Search certificate with certificate ID encoded as HEX string: ");
		sb.append("<input type=\"text\" id=\"cert\" name=\"cert\" value=\"\" maxlength=\"20\" style=\"width:200px;\" />&nbsp;<input type=\"submit\" value=\"Find\" /></form>");
		IssuedCertificateInfo[] issuedCerts = new IssuedCertificateInfo[0];

		// get certID8 from GET variable
		byte[] certIDBytes = new byte[0];
		if (req.getParameter("cert") != null && req.getParameter("cert").length() > 0) {
			String certID = req.getParameter("cert");
			certID = certID.trim();
			if (certID.length() > 0) {
				try {
					certIDBytes = Hex.decodeHex(certID.toCharArray());
				} catch (DecoderException e) {
					logger.warn("Unable to decode hex values of cert ID " + certID, e);
					userInfo = "<div id=\"errorbox\">Unable to decode hex values of cert ID " + certID
							+ "</div>";
				}
			}
		}

		try {
			if (certIDBytes.length >= 8) {
				issuedCerts = new IssuedCertificateInfo[1];
				IssuedCertificateInfo issuedCert = issuedCertsInfoProvider.getIssuedCertificate(certIDBytes);
				issuedCerts[0] = issuedCert;
			} else {
				issuedCerts = issuedCertsInfoProvider.getAll(0, 20);
			}
		} catch (ProviderException e) {
			logger.warn(e);
			userInfo = "<div id=\"errorbox\">Unable to get certificate from DB</div>";
		}
		if (userInfo != "") {
			sb.append("<br/><br/>" + userInfo);
		} else {
			// output list of cert infos
			sb.append("<table width=\"550px\" border=\"0\">\n");
			for (IssuedCertificateInfo info : issuedCerts) {
				sb.append("<tr><td valign=\"top\"><ul><li>CertId10: " + Hex.encodeHexString(info.getCertId())
						+ "</li>\n");
				if (info.getSubjectName() != null && info.getSubjectName().isEmpty() == false)
					sb.append("<li>Name: "
							+ StringEscapeUtils.escapeHtml4(info.getSubjectName()).replaceAll(
									"[^a-zA-Z0-9]+", "") + "</li>\n");
				sb.append("<li>Creation time: " + formatedDate(info.getCreationTime()) + "</li>\n");
				sb.append("<li>Expiration time: " + formatedDate(info.getExpirationTime()) + "</li>\n");
				// sb.append("<li>Signer CertId8: " + info.getSignerCertId +
				// "</li>\n");
				sb.append("</ul></td><td>");

				sb.append("<form action=\"\" method=\"get\"><input type=\"hidden\" name=\"cert\" value=\""
						+ Hex.encodeHexString(info.getCertId())
						+ "\"/><input type=\"submit\" name=\"submit\" value=\"Download\" />\n");
				sb.append("</form></td><td>");

				try {
					if (!crlProvider.isRevoked(info.getCertId())) {
						sb.append("<form action=\"/revokeCertificate\" method=\"get\"><input type=\"hidden\" name=\"cert\" value=\""
								+ Hex.encodeHexString(info.getCertId())
								+ "\"/><input type=\"submit\" value=\"Revoke\" />\n");
						sb.append("</form>");
					} else {
						sb.append("certifcate is revoked");
					}
				} catch (ProviderException e) {
					sb.append("Unable to retrieve revocation status");
				}
				sb.append("</td></tr>\n");
			}
			sb.append("</table>\n");
		}

		return sb.toString();
	}

	protected String formatedDate(Date date) {
		return new SimpleDateFormat(DATE_PATTERN_LOG).format(date);
	}

	@Override
	public boolean isProtected() {
		return true;
	}

	@Override
	public String[] getAllowedRoles() {
		return new String[] { UserRole.ADMINISTRATOR.toString(), UserRole.REGISTRATION_USER.toString(),
				UserRole.DEVELOPER.toString() };
	}

	@Override
	public Category getCategory() {
		return Category.CA_INFO;
	}
}
