package de.fraunhofer.sit.c2x.pki.ca.module.webserver;

import java.io.BufferedInputStream;
import java.io.BufferedReader;
import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.io.InputStreamReader;
import java.io.Reader;
import java.io.UnsupportedEncodingException;
import java.net.URL;
import java.security.KeyStore;
import java.security.KeyStoreException;
import java.security.NoSuchAlgorithmException;
import java.security.UnrecoverableKeyException;
import java.security.cert.Certificate;
import java.security.cert.CertificateException;
import java.security.cert.CertificateFactory;
import java.security.cert.X509Certificate;
import java.security.interfaces.RSAPrivateKey;
import java.util.ArrayList;
import java.util.List;

import javax.net.ssl.KeyManagerFactory;
import javax.xml.ws.Endpoint;

import org.apache.log4j.Logger;
import org.eclipse.jetty.http.spi.JettyHttpServer;
import org.eclipse.jetty.http.spi.JettyHttpServerProvider;
import org.eclipse.jetty.http.spi.ThreadPoolExecutorAdapter;
import org.eclipse.jetty.security.ConstraintMapping;
import org.eclipse.jetty.security.ConstraintSecurityHandler;
import org.eclipse.jetty.security.LoginService;
import org.eclipse.jetty.security.SecurityHandler;
import org.eclipse.jetty.security.authentication.FormAuthenticator;
import org.eclipse.jetty.server.Server;
import org.eclipse.jetty.server.handler.ContextHandlerCollection;
import org.eclipse.jetty.server.ssl.SslSelectChannelConnector;
import org.eclipse.jetty.servlet.ServletContextHandler;
import org.eclipse.jetty.servlet.ServletHolder;
import org.eclipse.jetty.util.security.Constraint;
import org.eclipse.jetty.util.ssl.SslContextFactory;

import com.google.inject.Inject;

import de.fraunhofer.sit.c2x.pki.ca.core.interfaces.CallBack;
import de.fraunhofer.sit.c2x.pki.ca.core.interfaces.Service;
import de.fraunhofer.sit.c2x.pki.ca.core.interfaces.ThreadPool;
import de.fraunhofer.sit.c2x.pki.ca.core.logging.InjectLogger;
import de.fraunhofer.sit.c2x.pki.ca.module.webserver.interfaces.Servlet;
import de.fraunhofer.sit.c2x.pki.ca.module.webserver.interfaces.WebService;
import de.fraunhofer.sit.c2x.pki.ca.provider.entities.KnownCA;
import de.fraunhofer.sit.c2x.pki.ca.provider.interfaces.ConfigProvider;

/**
 * @author Daniel Quanz (daniel.quanz@sit.fraunhofer.de)
 */
public class WebServer extends Thread implements Service, CallBack {

	protected class RestartThread extends Thread {
		protected Server jettyServer;
		// needed to allow open processes to finish
		private final static int GRACEFUL_SHUTDOWN_TIMEOUT = 500;

		public RestartThread(Server jettyServer) {
			super();
			this.jettyServer = jettyServer;
		}

		@Override
		public void run() {
			if (jettyServer != null && jettyServer.isRunning()) {
				try {
					// wait X milliseconds before the webserver is stopped.
					jettyServer.setGracefulShutdown(GRACEFUL_SHUTDOWN_TIMEOUT);
					jettyServer.stop();
					if (logger.isDebugEnabled())
						logger.debug("Jetty-HttpServer with client authentication stopped");
				} catch (Exception e) {
					logger.error("Error while stopping Jetty-HttpServer with client authentication", e);
				}
			} else {
				logger.error("No Jetty server found");
			}
			createAndStartJettyServer();
		}
	}

	@InjectLogger
	private Logger logger;

	protected final List<Servlet> servlets;

	protected Server jettyServer;

	protected boolean started;

	protected final WebService service;

	protected final ThreadPool threadpool;

	@Inject
	private LoginService loginService;

	@Inject
	protected ConfigProvider configProvider;

	public WebServer(ThreadPool threadpool) {
		this(new ArrayList<Servlet>(), threadpool, null);
	}

	@Inject
	public WebServer(List<Servlet> servlets, ThreadPool threadpool, WebService service) {
		this.servlets = servlets;
		this.threadpool = threadpool;
		this.service = service;
	}

	protected void createAndStartJettyServer() {
		try {
			for (int timeOut = 0; timeOut < 100; timeOut++) {
				if (configProvider.isStarted())
					break;
				if (logger.isTraceEnabled())
					logger.trace("Sleep until config provider is started.");
				sleep(10);
			}
			boolean httpSslEnabled = configProvider.getBoolean("httpSslEnabled", true);
			SslSelectChannelConnector sslSocket = null;
			try {
				sslSocket = getSslSelectChannelConnector();
			} catch (Exception e) {
				logger.error(e);
				httpSslEnabled = false;
			}
			
			if (httpSslEnabled) {
				jettyServer = new Server();
			} else {
				int httpServerPort = configProvider.getInt("httpServerPort", -1);
				if (httpServerPort == -1)
					throw new IOException(
							"Configuration paramter \"httpServerPort\" not found in DB. Please update the configuration of the server!");
				jettyServer = new Server(httpServerPort);
			}

			ThreadPoolExecutorAdapter pool = new ThreadPoolExecutorAdapter(threadpool.get());
			jettyServer.setThreadPool(pool);

			JettyHttpServerProvider.setServer(jettyServer);

			ContextHandlerCollection handlers = new ContextHandlerCollection();

			if (httpSslEnabled) {
				jettyServer.addConnector(sslSocket);
			}

			ServletContextHandler context = new ServletContextHandler(ServletContextHandler.SESSIONS);
			context.setContextPath("/");
			context.setSecurityHandler(getSecurityHandler());
			for (Servlet sc : servlets) {
				context.addServlet(new ServletHolder(sc), sc.getUrl());
			}
			handlers.addHandler(context);

			jettyServer.setHandler(handlers);

			if (service != null) {
				Endpoint endpoint = Endpoint.create(service);
				JettyHttpServer jettyHttp = new JettyHttpServer(jettyServer, true);
				endpoint.publish(jettyHttp.createContext(service.getPath()));
				logger.info("Web Service is running. " + service.getPath());
			}

			// register call back
			configProvider.registerCallBack("httpServerPort", this);
			configProvider.registerCallBack("httpSslEnabled", this);
			configProvider.registerCallBack("sslServerX509CertFile", this);
			configProvider.registerCallBack("sslServerRsaKeyFile", this);
			configProvider.registerCallBack("sslServerRsaKeyPassword", this);

			if (logger.isInfoEnabled())
				logger.info("Jetty-HttpServer is starting ...");

			jettyServer.start();
			jettyServer.join();

		} catch (Exception e) {
			logger.error(e);
		}
	}

	@Override
	public void run() {
		createAndStartJettyServer();
	}

	protected SslSelectChannelConnector getSslSelectChannelConnector() throws IOException {
		SslContextFactory sslFactory = new SslContextFactory();
		sslFactory.setProtocol("TLSv1");
		try {
			sslFactory.setKeyStore(getKeyStore());
			String password = configProvider.get("sslServerRsaKeyPassword");
			if(password == null || password.isEmpty())
				throw new KeyStoreException("Password \"sslServerRsaKeyPassword\" must not be empty");
			sslFactory.setKeyStorePassword(password);
		} catch (IOException | UnrecoverableKeyException | KeyStoreException | NoSuchAlgorithmException
				| CertificateException e) {
			logger.error(e);
			throw new IOException("Unable to create keystore: "+e.getMessage());
		}
		SslSelectChannelConnector sslSocket = new SslSelectChannelConnector(sslFactory);
		int httpServerPort = configProvider.getInt("httpServerPort", -1);
		if (httpServerPort == -1)
			throw new IOException(
					"Configuration paramter \"httpServerPort\" not found in DB. Please update the configuration of the server!");
		sslSocket.setPort(httpServerPort);
		sslSocket.setConfidentialPort(httpServerPort);
		sslSocket.setForwarded(true);
		return sslSocket;
	}

	protected KeyStore getKeyStore() throws UnsupportedEncodingException, IOException, KeyStoreException,
			NoSuchAlgorithmException, CertificateException, UnrecoverableKeyException {

		String file = configProvider.get("sslServerX509CertFile");
		URL url = getClass().getResource(file);
		if (url == null)
			throw new IOException("X.509 server cert not found within bundle: " + file);
		else if (logger.isDebugEnabled())
			logger.debug("X.509 server cert found within bundle: " + file);

		String x509ClientCertString = readStringFromResource(new InputStreamReader(url.openStream(), "UTF-8"));
		String rsaClientKeyString = "";
		String rsaClientKeyPassword = "";
		if (!x509ClientCertString.isEmpty()) {
			X509Certificate[] x509Certs = createX509CertsFromString(x509ClientCertString);
			if (x509Certs.length < 1)
				throw new IOException("Unexpected number of X.509 certs provided");

			X509Certificate x509Cert = x509Certs[0];
			String x509Name = x509Cert.getSubjectDN().getName();
			if (x509Name.isEmpty())
				x509ClientCertString = "";

			file = configProvider.get("sslServerRsaKeyFile");
			url = getClass().getResource(file);
			if (url == null)
				throw new IOException("RSA private key for server cert not found within bundle: " + file);
			else if (logger.isDebugEnabled())
				logger.debug("RSA private key for server cert found within bundle: " + file);

			rsaClientKeyString = readStringFromResource(new InputStreamReader(url.openStream(), "UTF-8"));
			rsaClientKeyPassword = configProvider.get("sslServerRsaKeyPassword");
			RSAPrivateKey key = KnownCA.createKeyFromString(rsaClientKeyString, rsaClientKeyPassword);

			if (!x509Name.isEmpty() && !rsaClientKeyString.isEmpty()) {
				// try to load key and certificate into keystore
				KeyStore ks = KeyStore.getInstance("JKS");
				ks.load(null, "".toCharArray());
				ks.setKeyEntry(x509Cert.getSubjectDN().getName(), key, rsaClientKeyPassword.toCharArray(),
						x509Certs);

				KeyManagerFactory keyManagerFactory = KeyManagerFactory.getInstance("SunX509");
				keyManagerFactory.init(ks, rsaClientKeyPassword.toCharArray());

				if (logger.isDebugEnabled())
					logger.debug("X.509 cert and RSA private key successfully loaded into key store");

				return ks;
			}
		}
		throw new KeyStoreException("Key store not available");
	}

	protected String readStringFromResource(Reader reader) throws IOException {
		StringBuffer sb = new StringBuffer();
		BufferedReader br = new BufferedReader(reader);
		for (int c = br.read(); c != -1; c = br.read())
			sb.append((char) c);
		return sb.toString();
	}

	public static X509Certificate[] createX509CertsFromString(String certificate) throws IOException {
		if (!certificate.contains("-----BEGIN CERTIFICATE-----")
				&& !certificate.contains("-----END CERTIFICATE-----")) {
			throw new IOException("X.509 certificate has wrong format. Certificate must contain "
					+ "\"-----BEGIN CERTIFICATE-----\" and \"-----END CERTIFICATE-----\"");
		}

		ArrayList<X509Certificate> x509Certs = new ArrayList<>();
		byte[] certificateBytes = certificate.getBytes();
		ByteArrayInputStream bais;
		BufferedInputStream bis;
		try {
			CertificateFactory cf = CertificateFactory.getInstance("X.509");
			bais = new ByteArrayInputStream(certificateBytes);
			bis = new BufferedInputStream(bais);
			while (bis.available() > 0) {
				Certificate cert = cf.generateCertificate(bis);
				if (cert.getType() != "X.509")
					throw new CertificateException("Certificate type " + cert.getType()
							+ " not supported for trust store");
				X509Certificate x509Cert = (X509Certificate) cert;
				x509Cert.checkValidity();
				x509Certs.add(x509Cert);
			}
		} catch (CertificateException e) {
			throw new IOException("Unable to decode X.509 certificate", e);
		}
		bais.close();
		bis.close();
		return x509Certs.toArray(new X509Certificate[x509Certs.size()]);
	}

	@Override
	public void stopService() {
		try {
			jettyServer.stop();
		} catch (Exception e) {
			logger.error(e);
		}
		logger.info("Jetty-HttpServer is stopped");
		started = false;
	}

	@Override
	public void startService() {
		start();
		started = true;
		logger.info("Jetty-HttpServer started");
	}

	@Override
	public boolean isStarted() {
		return started;
	}

	private SecurityHandler getSecurityHandler() {

		// map the security constraint to the root path.

		// create the security handler, set the authentication to Basic
		// and assign the realm.
		ConstraintSecurityHandler csh = new ConstraintSecurityHandler();

		csh.setAuthenticator(new FormAuthenticator("/login", "/login?error=loginfailed", true));
		csh.setRealmName("REALM");

		for (Servlet srv : servlets) {
			if (srv.isProtected()) {
				// add authentication
				String[] roles = srv.getAllowedRoles();
				Constraint constraint = new Constraint();
				constraint.setName(Constraint.__FORM_AUTH);
				constraint.setRoles(roles);
				constraint.setAuthenticate(true);

				ConstraintMapping cm = new ConstraintMapping();
				cm.setConstraint(constraint);
				cm.setPathSpec(srv.getUrl());
				csh.addConstraintMapping(cm);
			}
		}

		// set the login service
		csh.setLoginService(loginService);

		return csh;

	}

	public void restartWebServer() {
		// create new thread that restarts the webserver in order to allow the
		// finishing of open processes. Maybe this restart is triggered from a
		// web page
		RestartThread restartThread = new RestartThread(jettyServer);
		restartThread.start();
		// deregister call back
		configProvider.deregisterCallBack("httpServerPort", this);
		configProvider.deregisterCallBack("httpSslEnabled", this);
		configProvider.deregisterCallBack("sslServerX509CertFile", this);
		configProvider.deregisterCallBack("sslServerRsaKeyFile", this);
		configProvider.deregisterCallBack("sslServerRsaKeyPassword", this);
	}

	@Override
	public void notifyEvent(String info) {
		// restart the webserver if configuration parameter
		// "webserverWithClientAuthentication" has changed
		if (logger.isInfoEnabled())
			logger.info("Restart of Jetty-HttpServer due to change of configuration: " + info);
		restartWebServer();
	}
}
