package de.fraunhofer.sit.c2x.pki.ca.measuring;

import java.util.ArrayList;
import java.util.List;

import org.apache.commons.codec.binary.Hex;
import org.apache.log4j.Logger;

import com.google.inject.Inject;
import com.google.inject.name.Named;

import de.fraunhofer.sit.c2x.pki.ca.core.logging.InjectLogger;
import de.fraunhofer.sit.c2x.pki.ca.provider.ProviderException;
import de.fraunhofer.sit.c2x.pki.ca.provider.entities.Measuring;
import de.fraunhofer.sit.c2x.pki.ca.provider.interfaces.MeasuringWriter;
import de.fraunhofer.sit.c2x.pki.ca.provider.jdbc.JDBCMeasuringProvider;

public class MeasuringWriterThread extends Thread implements MeasuringWriter {
	@InjectLogger
	private Logger logger;
	@Inject
	private JDBCMeasuringProvider db;
	@Inject
	@Named(value = "caName")
	private String caName;

	private List<Measuring> measurings;
	// execution interval in milli seconds
	private final static int EXECUTION_INTERVAL = 100;

	public MeasuringWriterThread() {
		this.measurings = new ArrayList<Measuring>();

		super.start();
	}

	@Override
	public void run() {
		if (logger.isDebugEnabled())
			logger.debug("MeasuringWriterThread started with execution interval " + EXECUTION_INTERVAL
					+ " ms");
		this.setName(caName + " measuring");

		while (isAlive() && !interrupted()) {
			if (db != null && measurings.size() > 0) {
				if (logger.isTraceEnabled())
					logger.trace("MeasuringWriter writes " + measurings.size() + " entries to DB");
				synchronized (measurings) {
					for (Measuring measuring : measurings) {
						try {
							db.save(measuring);

							// write entry into logs
							if (logger.isInfoEnabled())
								logger.info((measuring.getCreationTime().getTime()) + ","
										+ measuring.getStationType() + ","
										+ Hex.encodeHexString(measuring.getStationDigest()) + ","
										+ Hex.encodeHexString(measuring.getRequestHash()) + ",TIMING,"
										+ measuring.getKey() + "," + measuring.getNumberOperations() + ","
										+ measuring.getValue());

						} catch (ProviderException e) {
							logger.error(e);
						}
					}
					measurings.clear();
				}
			}
			try {
				sleep(EXECUTION_INTERVAL);
			} catch (InterruptedException e) {
				logger.error(e);
			}
		}
	}

	public boolean writeMeasuring(Measuring measuring) {
		synchronized (measurings) {
			return this.measurings.add(measuring);
		}
	}
}
