package de.fraunhofer.sit.c2x.pki.ca.crypto.keystore;

import java.io.BufferedReader;
import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.InputStreamReader;
import java.math.BigInteger;
import java.security.Key;
import java.security.KeyStore;
import java.security.KeyStoreException;
import java.security.NoSuchAlgorithmException;
import java.security.PrivateKey;
import java.security.PublicKey;
import java.security.Security;
import java.security.cert.CertificateException;
import java.security.cert.X509Certificate;

import org.bouncycastle.asn1.x500.X500Name;
import org.bouncycastle.asn1.x500.X500NameBuilder;
import org.bouncycastle.asn1.x500.style.BCStyle;
import org.bouncycastle.cert.X509CertificateHolder;
import org.bouncycastle.cert.X509v1CertificateBuilder;
import org.bouncycastle.cert.jcajce.JcaX509CertificateConverter;
import org.bouncycastle.cert.jcajce.JcaX509v1CertificateBuilder;
import org.bouncycastle.crypto.AsymmetricCipherKeyPair;
import org.bouncycastle.crypto.CipherParameters;
import org.bouncycastle.crypto.params.ECDomainParameters;
import org.bouncycastle.crypto.params.ECPrivateKeyParameters;
import org.bouncycastle.crypto.params.ECPublicKeyParameters;
import org.bouncycastle.jce.provider.JCEECPrivateKey;
import org.bouncycastle.jce.provider.JCEECPublicKey;
import org.bouncycastle.jce.spec.ECParameterSpec;
import org.bouncycastle.jce.spec.ECPrivateKeySpec;
import org.bouncycastle.operator.ContentSigner;
import org.bouncycastle.operator.OperatorCreationException;
import org.bouncycastle.operator.jcajce.JcaContentSignerBuilder;
import org.joda.time.DateTime;

import de.fraunhofer.sit.c2x.pki.ca.crypto.CryptoUtils;
import de.fraunhofer.sit.c2x.pki.ca.provider.interfaces.KeystoreProvider.KeyType;

public class KeyStoreImpl {
	private java.security.KeyStore internalKeyStore = null;
	private byte[] keyStoreBytes;
	private final char[] passphrase;

	public KeyStoreImpl() throws KeyStoreException {
		this(null);
	}

	// public KeyStoreImpl(byte[] certId, AsymmetricCipherKeyPair
	// signingkeyPair,
	// AsymmetricCipherKeyPair encryptionKeyPair)
	// throws KeyStoreException, NoSuchAlgorithmException,
	// CertificateException, IOException {
	// this(certId, signingkeyPair, encryptionKeyPair, null);
	// }

	public KeyStoreImpl(AsymmetricCipherKeyPair signingKeyPair,
			AsymmetricCipherKeyPair encryptionKeyPair, String passphrase)
			throws KeyStoreException, NoSuchAlgorithmException,
			CertificateException, IOException {
		if (passphrase != null)
			this.passphrase = passphrase.toCharArray();
		else
			this.passphrase = null;

		Security.addProvider(CryptoUtils.BC);
		this.internalKeyStore = java.security.KeyStore.getInstance("UBER",
				CryptoUtils.BC);

		JCEECPrivateKey privateSigningKey = CryptoUtils
				.convertECECPrivateKeyParametersToJCEECPrivateKey((ECPrivateKeyParameters) signingKeyPair
						.getPrivate());
		JCEECPublicKey publicSigningKey = CryptoUtils
				.convertECECPublicKeyParametersToJCEECPublicKey((ECPublicKeyParameters) signingKeyPair
						.getPublic());
		JCEECPrivateKey privateEncKey = CryptoUtils
				.convertECECPrivateKeyParametersToJCEECPrivateKey((ECPrivateKeyParameters) encryptionKeyPair
						.getPrivate());

		java.security.cert.Certificate[] chain = new java.security.cert.Certificate[1];
		chain[0] = createX509RootCa("", publicSigningKey, privateSigningKey,
				new DateTime(), new DateTime(2099, 1, 1, 0, 0));

		// load key store from bytes
		char[] pass = this.passphrase;
		if (pass == null)
			pass = passphrase();

		internalKeyStore.load(bytesToStreamStream(keyStoreBytes), pass);
		// store signing key pair into key store
		if (signingKeyPair != null) {
			internalKeyStore.setKeyEntry(KeyType.SIGNING_KEY.toString(),
					privateSigningKey, pass, chain);
		}
		// store encryption key pair into key store
		if (encryptionKeyPair != null) {
			internalKeyStore.setKeyEntry(KeyType.ENCRYPTION_KEY.toString(),
					privateEncKey, pass, chain);
		}
		// update bytes of key store
		ByteArrayOutputStream baos = new ByteArrayOutputStream();
		internalKeyStore.store(baos, pass);
		keyStoreBytes = baos.toByteArray();
	}
	
	public KeyStoreImpl(AsymmetricCipherKeyPair signingKeyPair, String passphrase)
			throws KeyStoreException, NoSuchAlgorithmException,
			CertificateException, IOException {
		if (passphrase != null)
			this.passphrase = passphrase.toCharArray();
		else
			this.passphrase = null;

		Security.addProvider(CryptoUtils.BC);
		this.internalKeyStore = java.security.KeyStore.getInstance("UBER",
				CryptoUtils.BC);

		JCEECPrivateKey privateSigningKey = CryptoUtils
				.convertECECPrivateKeyParametersToJCEECPrivateKey((ECPrivateKeyParameters) signingKeyPair
						.getPrivate());
		JCEECPublicKey publicSigningKey = CryptoUtils
				.convertECECPublicKeyParametersToJCEECPublicKey((ECPublicKeyParameters) signingKeyPair
						.getPublic());

		java.security.cert.Certificate[] chain = new java.security.cert.Certificate[1];
		chain[0] = createX509RootCa("", publicSigningKey, privateSigningKey,
				new DateTime(), new DateTime(2099, 1, 1, 0, 0));

		// load key store from bytes
		char[] pass = this.passphrase;
		if (pass == null)
			pass = passphrase();

		internalKeyStore.load(bytesToStreamStream(keyStoreBytes), pass);
		// store signing key pair into key store
		if (signingKeyPair != null) {
			internalKeyStore.setKeyEntry(KeyType.CRL_SIGNING_KEY.toString(),
					privateSigningKey, pass, chain);
		}
		// update bytes of key store
		ByteArrayOutputStream baos = new ByteArrayOutputStream();
		internalKeyStore.store(baos, pass);
		keyStoreBytes = baos.toByteArray();
	}

	public KeyStoreImpl(byte[] keyStoreBytes) throws KeyStoreException {
		this(keyStoreBytes, null);
	}

	public KeyStoreImpl(byte[] keyStoreBytes, String passphrase)
			throws KeyStoreException {
		if (passphrase != null)
			this.passphrase = passphrase.toCharArray();
		else
			this.passphrase = null;

		Security.addProvider(CryptoUtils.BC);
		this.internalKeyStore = java.security.KeyStore.getInstance("UBER",
				CryptoUtils.BC);
		this.keyStoreBytes = keyStoreBytes;
	}

	public CipherParameters getKey(byte[] certId, KeyType type)
			throws Exception {
		char[] pass = this.passphrase;
		if (pass == null)
			pass = passphrase();

		return getKey(certId, String.valueOf(pass), type);
	}

	public CipherParameters getKey(byte[] certId, String passphrase,
			KeyType type) throws Exception {
		internalKeyStore.load(bytesToStreamStream(keyStoreBytes),
				passphrase.toCharArray());
		Key privateKey = internalKeyStore.getKey(type.toString(),
				passphrase.toCharArray());

		// // INFO: delete this log output!!! Used only for testing purposes
		// Logger logger = Logger.getLogger(KeyStoreImpl.class);
		// if(logger.isDebugEnabled()){
		// logger.debug("Private key from type "+type+" extracted from key store: "+privateKey.toString());
		// }

		ECPrivateKeyParameters privateECKey = convertJCEECPrivateKeyToECECPrivateKeyParameters((JCEECPrivateKey) privateKey);
		return privateECKey;
	}

	public ECPrivateKeyParameters getECKey(KeyType type) throws Exception {
		char[] pass = this.passphrase;
		if (pass == null)
			pass = passphrase();

		return getECKey(String.valueOf(pass), type);
	}

	public ECPrivateKeyParameters getECKey(String passphrase, KeyType type)
			throws Exception {
		internalKeyStore.load(bytesToStreamStream(keyStoreBytes),
				passphrase.toCharArray());
		Key privateKey = internalKeyStore.getKey(type.toString(),
				passphrase.toCharArray());
		ECPrivateKeyParameters privateECKey = convertJCEECPrivateKeyToECECPrivateKeyParameters((JCEECPrivateKey) privateKey);
		return privateECKey;
	}

	public int size() throws NoSuchAlgorithmException, CertificateException,
			IOException, KeyStoreException {
		char[] pass = this.passphrase;
		if (pass == null)
			pass = passphrase();
		internalKeyStore.load(bytesToStreamStream(keyStoreBytes), pass);
		return internalKeyStore.size();
	}

	private char[] passphrase() {
		BufferedReader br = new BufferedReader(new InputStreamReader(System.in));
		String line = "";
		System.out.print("Enter passphrase for KeyStore: ");
		try {
			line = br.readLine();
		} catch (IOException e) {
			e.printStackTrace();
		}
		System.out.println();
		return line.toCharArray();
	}

	private ByteArrayInputStream bytesToStreamStream(byte[] bytes) {
		return (bytes == null) ? null : new ByteArrayInputStream(bytes);
	}

	/**
	 * creates a X.509 certificate
	 * 
	 * @param subjectName
	 * @param issuerName
	 * @param subjectsPublicKey
	 * @param issuersPrivateKey
	 * @param startDate
	 * @param endDate
	 * @return
	 */
	public static X509Certificate createX509RootCa(String subjectName,
			String issuerName, java.security.PublicKey subjectsPublicKey,
			PrivateKey issuersPrivateKey, DateTime startDate, DateTime endDate) {

		X500Name subject = buildX500Name(subjectName);
		X500Name issuer = buildX500Name(issuerName);

		ContentSigner sigGen;
		X509Certificate cert = null;
		try {
			sigGen = new JcaContentSignerBuilder("SHA256withECDSA")
					.setProvider(CryptoUtils.BC).build(issuersPrivateKey);
			X509v1CertificateBuilder certBuilder = new JcaX509v1CertificateBuilder(
					issuer, BigInteger.ONE, startDate.toDate(),
					endDate.toDate(), subject, subjectsPublicKey);

			X509CertificateHolder holder = certBuilder.build(sigGen);
			cert = new JcaX509CertificateConverter()
					.setProvider(CryptoUtils.BC).getCertificate(holder);
		} catch (OperatorCreationException e) {
			e.printStackTrace();
		} catch (CertificateException e) {
			e.printStackTrace();
		}

		return cert;
	}

	/**
	 * creates a self signed X.509 certificate
	 * 
	 * @param subjectName
	 * @param keyPair
	 * @param startDate
	 * @param endDate
	 * @return
	 */
	public static X509Certificate createX509RootCa(String subjectName,
			PublicKey publicKey, PrivateKey privateKey, DateTime startDate,
			DateTime endDate) {

		return createX509RootCa(subjectName, subjectName, publicKey,
				privateKey, startDate, endDate);
	}

	private static X500Name buildX500Name(String organisation) {
		return buildX500Name("DE", organisation, "Darmstadt", "Hessen",
				"v2xpilotpki@sit.fraunhofer.de");
	}

	private static X500Name buildX500Name(String country, String organisation,
			String local, String state, String email) {
		X500NameBuilder builder = new X500NameBuilder(BCStyle.INSTANCE);
		builder.addRDN(BCStyle.C, country);
		builder.addRDN(BCStyle.O, organisation);
		builder.addRDN(BCStyle.L, local);
		builder.addRDN(BCStyle.ST, state);
		builder.addRDN(BCStyle.E, email);

		return builder.build();
	}

	public static ECPrivateKeyParameters convertJCEECPrivateKeyToECECPrivateKeyParameters(
			JCEECPrivateKey privateKey) {

		ECParameterSpec spec = privateKey.getParameters();
		ECDomainParameters domain = new ECDomainParameters(spec.getCurve(),
				spec.getG(), spec.getN(), spec.getH(), spec.getSeed());
		return new ECPrivateKeyParameters(privateKey.getD(), domain);
	}

	public static JCEECPrivateKey convertECECPrivateKeyParametersToJCEECPrivateKey(
			ECPrivateKeyParameters privateKey) {
		ECDomainParameters domain = privateKey.getParameters();
		ECParameterSpec spec = new ECParameterSpec(domain.getCurve(),
				domain.getG(), domain.getN(), domain.getH(), domain.getSeed());
		ECPrivateKeySpec keySpec = new ECPrivateKeySpec(privateKey.getD(), spec);
		return new JCEECPrivateKey("EC", keySpec);
	}

	public static JCEECPublicKey convertECECPublicKeyParametersToJCEECPublicKey(
			ECPublicKeyParameters publicKey) {
		ECDomainParameters domain = publicKey.getParameters();
		ECParameterSpec spec = new ECParameterSpec(domain.getCurve(),
				domain.getG(), domain.getN(), domain.getH(), domain.getSeed());
		ECParameterSpec paramECParameterSpec = new ECParameterSpec(
				spec.getCurve(), spec.getG(), spec.getN(), spec.getH(),
				spec.getSeed());
		return new JCEECPublicKey("EC", publicKey, paramECParameterSpec);
	}

	public byte[] getKeyStoreBytes() {
		return keyStoreBytes;
	}
	
	public KeyStore getKeyStore(){
		return internalKeyStore;
	}
}
