package de.fraunhofer.sit.c2x.pki.ca.crypto;

import java.io.IOException;
import java.math.BigInteger;
import java.security.Provider;
import java.util.Arrays;

import org.bouncycastle.crypto.CipherParameters;
import org.bouncycastle.crypto.params.ECDomainParameters;
import org.bouncycastle.crypto.params.ECPrivateKeyParameters;
import org.bouncycastle.crypto.params.ECPublicKeyParameters;
import org.bouncycastle.jce.ECNamedCurveTable;
import org.bouncycastle.jce.provider.BouncyCastleProvider;
import org.bouncycastle.jce.provider.JCEECPrivateKey;
import org.bouncycastle.jce.provider.JCEECPublicKey;
import org.bouncycastle.jce.spec.ECNamedCurveParameterSpec;
import org.bouncycastle.jce.spec.ECParameterSpec;
import org.bouncycastle.jce.spec.ECPrivateKeySpec;
import org.bouncycastle.math.ec.ECPoint;

import de.fraunhofer.sit.c2x.pki.ca.core.interfaces.WaveType;
import de.fraunhofer.sit.c2x.pki.ca.crypto.signer.ECDSA;
import de.fraunhofer.sit.c2x.pki.ca.utils.WaveUtils;

/**
 * @author Daniel Quanz (daniel.quanz@sit.fraunhofer.de)
 * @author Norbert Bissmeyer (norbert.bissmeyer@sit.fraunhofer.de)
 */
public class CryptoUtils {

	public static final Provider BC = new BouncyCastleProvider();

	/**
	 * @param unsignedCsr
	 * @return
	 * @throws CryptoUtilsException
	 */
	public static byte[] sha256(byte[] bytes) {
		return Digest.getInstance().sha256(bytes);
	}

	//
	/**
	 * @param unsignedCsr
	 * @return
	 * @throws IOException
	 * @throws CryptoUtilsException
	 */
	public static byte[] sha256(WaveType dataToHash) {

		return sha256(WaveUtils.getBytesFromWaveType(dataToHash));
	}

	public static ECPublicKeyParameters compressECPublicKeyParameters(CipherParameters pubKey) {
		ECPublicKeyParameters publicKey = (ECPublicKeyParameters) pubKey;

		return new ECPublicKeyParameters(new ECPoint.Fp(publicKey.getQ().getCurve(), publicKey.getQ().getX(),
				publicKey.getQ().getY(), true), publicKey.getParameters());
	}

	public static ECPrivateKeyParameters convertJCEECPrivateKeyToECECPrivateKeyParameters(
			JCEECPrivateKey privateKey) {

		ECParameterSpec spec = privateKey.getParameters();
		ECDomainParameters domain = new ECDomainParameters(spec.getCurve(), spec.getG(), spec.getN(),
				spec.getH(), spec.getSeed());
		return new ECPrivateKeyParameters(privateKey.getD(), domain);
	}

	public static JCEECPrivateKey convertECECPrivateKeyParametersToJCEECPrivateKey(
			ECPrivateKeyParameters privateKey) {
		ECDomainParameters domain = privateKey.getParameters();
		ECParameterSpec spec = new ECParameterSpec(domain.getCurve(), domain.getG(), domain.getN(),
				domain.getH(), domain.getSeed());
		ECPrivateKeySpec keySpec = new ECPrivateKeySpec(privateKey.getD(), spec);
		return new JCEECPrivateKey("EC", keySpec);
	}

	public static JCEECPublicKey convertECECPublicKeyParametersToJCEECPublicKey(
			ECPublicKeyParameters publicKey) {
		ECDomainParameters domain = publicKey.getParameters();
		ECParameterSpec spec = new ECParameterSpec(domain.getCurve(), domain.getG(), domain.getN(),
				domain.getH(), domain.getSeed());
		ECParameterSpec paramECParameterSpec = new ECParameterSpec(spec.getCurve(), spec.getG(), spec.getN(),
				spec.getH(), spec.getSeed());
		return new JCEECPublicKey("EC", publicKey, paramECParameterSpec);
	}

	public static ECPublicKeyParameters bytesToEcPublicKeyParameters(byte[] bytes) {

		ECDomainParameters domain = getDomain(bytes);

		ECPoint q = domain.getCurve().decodePoint(bytes);
		return new ECPublicKeyParameters(q, domain);
	}

	public static ECDomainParameters getDomain(byte[] bytes) {
		ECNamedCurveParameterSpec spec = null;
		if (bytes.length == 57) {
			spec = ECNamedCurveTable.getParameterSpec("P-224");
		} else if (bytes.length == 65) {
			spec = ECNamedCurveTable.getParameterSpec("P-256");
		} else {
			throw new IllegalArgumentException("No ECC algorithm for key size " + bytes.length
					+ " implemented");
		}
		return new ECDomainParameters(spec.getCurve(), spec.getG(), spec.getN(), spec.getH(), spec.getSeed());
	}

	public static ECPrivateKeyParameters bytesToEcPrivateKeyParameters(byte[] bytes) {

		ECNamedCurveParameterSpec spec = ECNamedCurveTable.getParameterSpec("P-256");
		ECDomainParameters domain = new ECDomainParameters(spec.getCurve(), spec.getG(), spec.getN(),
				spec.getH(), spec.getSeed());

		return new ECPrivateKeyParameters(new BigInteger(1, bytes), domain);
	}

	public static byte[] getPublicKeyX(CipherParameters pubKey, int fieldSize) {
		ECPublicKeyParameters publicKey = (ECPublicKeyParameters) pubKey;

		byte[] q = publicKey.getQ().getEncoded();

		byte[] x = new byte[fieldSize];
		x = Arrays.copyOfRange(q, 1, fieldSize + 1);
		return x;
	}

	public static byte[] getPublicKeyY(CipherParameters pubKey, int fieldSize) {
		ECPublicKeyParameters publicKey = (ECPublicKeyParameters) pubKey;

		byte[] q = publicKey.getQ().getEncoded();

		byte[] y = new byte[fieldSize];
		y = Arrays.copyOfRange(q, fieldSize + 1, 2 * fieldSize + 1);
		return y;
	}
	
	public static boolean verifyECDSA(byte[] messageToVerify, BigInteger[] signature,
			ECPublicKeyParameters publicKey) {

		ECDSA engine = new ECDSA();
		engine.init(publicKey);
		boolean signatureIsValid = engine.verifySignature(messageToVerify, signature);

		return signatureIsValid;
	}
}
