package de.fraunhofer.sit.c2x.pki.etsi_ts103097v1114.utils;

import java.io.ByteArrayOutputStream;
import java.io.DataOutputStream;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;

import org.bouncycastle.util.encoders.Base64;

import de.fraunhofer.sit.c2x.pki.ca.module.webservice.datatypes.ItsStationRegistrationRequestData;
import de.fraunhofer.sit.c2x.pki.ca.module.webservice.datatypes.PsidSspPriority;
import de.fraunhofer.sit.c2x.pki.ca.provider.entities.AuthorizedDevice;
import de.fraunhofer.sit.c2x.pki.ca.utils.WaveUtils;
import de.fraunhofer.sit.c2x.pki.etsi_ts103097v1114.impl.IntX;
import de.fraunhofer.sit.c2x.pki.etsi_ts103097v1114.impl.ItsAidPriority;
import de.fraunhofer.sit.c2x.pki.etsi_ts103097v1114.impl.ItsAidPrioritySsp;
import de.fraunhofer.sit.c2x.pki.etsi_ts103097v1114.impl.ItsAidSsp;
import de.fraunhofer.sit.c2x.pki.etsi_ts103097v1114.impl.Opaque;
import de.fraunhofer.sit.c2x.pki.etsi_ts103097v1114.impl.SubjectAttribute;
import de.fraunhofer.sit.c2x.pki.etsi_ts103097v1114.impl.UInt8;

public class EtsiWebserviceUtils {

	public static AuthorizedDevice extractItsAidPermissionsAndSubjectAssurance(
			ItsStationRegistrationRequestData itsStationRegistrationRequestData,
			AuthorizedDevice authorizedDevice) throws Exception {

		ArrayList<IntX> itsAids = new ArrayList<>();
		ArrayList<ItsAidSsp> itsAidSsps = new ArrayList<>();
		ArrayList<ItsAidPriority> itsAidPriorities = new ArrayList<>();
		ArrayList<ItsAidPrioritySsp> itsAidPrioritySsps = new ArrayList<>();

		byte[] itsAidBytes = null;
		byte[] itsAidSspBytes = null;
		byte[] itsAidPriorityBytes = null;
		byte[] itsAidPrioritySspBytes = null;

		try {

			// Set subjectAssurance
			authorizedDevice.setSubjectAssurance(itsStationRegistrationRequestData.getSubjectAssurance());

			// Handle ItsAid Permissions
			List<PsidSspPriority> data = itsStationRegistrationRequestData.getPsidSspPriority();

			if (data == null) {
				return authorizedDevice;
			}

			for (Iterator<PsidSspPriority> iterator = data.iterator(); iterator.hasNext();) {
				PsidSspPriority psidSspPriority = (PsidSspPriority) iterator.next();

				extractItsAidPermissions(itsAids, itsAidSsps, itsAidPriorities, itsAidPrioritySsps,
						psidSspPriority);

			}

			ByteArrayOutputStream bos = new ByteArrayOutputStream();
			DataOutputStream dos = new DataOutputStream(bos);

			if (!itsAids.isEmpty()) {

				WaveUtils.writeArrayToStream(dos, itsAids);
				itsAidBytes = bos.toByteArray();

			}

			if (!itsAidSsps.isEmpty()) {

				bos.reset();

				WaveUtils.writeArrayToStream(dos, itsAidSsps);
				itsAidSspBytes = bos.toByteArray();

			}

			if (!itsAidPriorities.isEmpty()) {

				bos.reset();

				WaveUtils.writeArrayToStream(dos, itsAidPriorities);
				itsAidPriorityBytes = bos.toByteArray();

			}

			if (!itsAidPrioritySsps.isEmpty()) {

				bos.reset();

				WaveUtils.writeArrayToStream(dos, itsAidPrioritySsps);
				itsAidPrioritySspBytes = bos.toByteArray();

			}

			dos.close();
			bos.close();

			authorizedDevice.setItsAidList(itsAidBytes);
			authorizedDevice.setItsAidSspList(itsAidSspBytes);
			authorizedDevice.setPriorityItsAidList(itsAidPriorityBytes);
			authorizedDevice.setPrioritySspList(itsAidPrioritySspBytes);

			return authorizedDevice;

		} catch (Exception e) {
			throw new Exception("Handling of its station " + Base64.encode(authorizedDevice.getCanonicalId())
					+ " failed when extracting permissions! " + e.getMessage(), e);
		}
	}

	/**
	 * @param itsAids
	 * @param itsAidSsps
	 * @param itsAidPriorities
	 * @param itsAidPrioritySsps
	 * @param input
	 * @return
	 * @throws Exception
	 */
	private static void extractItsAidPermissions(ArrayList<IntX> itsAids, ArrayList<ItsAidSsp> itsAidSsps,
			ArrayList<ItsAidPriority> itsAidPriorities, ArrayList<ItsAidPrioritySsp> itsAidPrioritySsps,
			PsidSspPriority input) throws Exception {
		if (input.getPsid() == null) {
			throw new Exception("Missing Psid in PsidSspPriority element!");
		}

		IntX itsAid = new IntX(input.getPsid().longValue());

		if (input.getMaxPriority() != null && input.getServiceSpecificPermissions() != null) {
			ItsAidPrioritySsp x = new ItsAidPrioritySsp();

			x.setItsAid(itsAid);
			x.setMaxPriority(new UInt8(input.getMaxPriority().intValue()));
			x.setServiceSpecificPermissions(new Opaque(input.getServiceSpecificPermissions()));

			itsAidPrioritySsps.add(x);
		} else if (input.getMaxPriority() != null) {
			ItsAidPriority x = new ItsAidPriority();

			x.setItsAid(itsAid);
			x.setMaxPriority(new UInt8(input.getMaxPriority().intValue()));

			itsAidPriorities.add(x);
		} else if (input.getServiceSpecificPermissions() != null) {
			ItsAidSsp x = new ItsAidSsp();

			x.setItsAid(itsAid);
			x.setServiceSpecificPermissions(new Opaque(input.getServiceSpecificPermissions()));

			itsAidSsps.add(x);
		} else {
			itsAids.add(itsAid);
		}
	}

	public static List<SubjectAttribute> extractSubjectAttributes(PsidSspPriority input) throws Exception {
		ArrayList<SubjectAttribute> output = new ArrayList<>();

		ArrayList<IntX> itsAids = new ArrayList<>();
		ArrayList<ItsAidSsp> itsAidSsps = new ArrayList<>();
		ArrayList<ItsAidPriority> itsAidPriorities = new ArrayList<>();
		ArrayList<ItsAidPrioritySsp> itsAidPrioritySsps = new ArrayList<>();

		extractItsAidPermissions(itsAids, itsAidSsps, itsAidPriorities, itsAidPrioritySsps, input);

		if (itsAids != null && itsAids.size() > 0)
			output.add(new SubjectAttribute(itsAids.toArray(new IntX[itsAids.size()])));
		if (itsAidSsps != null && itsAidSsps.size() > 0)
			output.add(new SubjectAttribute(itsAidSsps.toArray(new ItsAidSsp[itsAidSsps.size()])));
		if (itsAidPriorities != null && itsAidPriorities.size() > 0)
			output.add(new SubjectAttribute(itsAidPriorities.toArray(new ItsAidPriority[itsAidPriorities
					.size()])));
		if (itsAidPrioritySsps != null && itsAidPrioritySsps.size() > 0)
			output.add(new SubjectAttribute(itsAidPrioritySsps
					.toArray(new ItsAidPrioritySsp[itsAidPrioritySsps.size()])));

		return output;
	}

	public static List<SubjectAttribute> extractSubjectAttributes(List<PsidSspPriority> input)
			throws Exception {
		ArrayList<SubjectAttribute> output = new ArrayList<>();

		if (input == null)
			return output;

		ArrayList<IntX> itsAids = new ArrayList<>();
		ArrayList<ItsAidSsp> itsAidSsps = new ArrayList<>();
		ArrayList<ItsAidPriority> itsAidPriorities = new ArrayList<>();
		ArrayList<ItsAidPrioritySsp> itsAidPrioritySsps = new ArrayList<>();

		for (PsidSspPriority permissions : input) {
			extractItsAidPermissions(itsAids, itsAidSsps, itsAidPriorities, itsAidPrioritySsps, permissions);
		}
		if (itsAids != null && itsAids.size() > 0)
			output.add(new SubjectAttribute(itsAids.toArray(new IntX[itsAids.size()])));
		if (itsAidSsps != null && itsAidSsps.size() > 0)
			output.add(new SubjectAttribute(itsAidSsps.toArray(new ItsAidSsp[itsAidSsps.size()])));
		if (itsAidPriorities != null && itsAidPriorities.size() > 0)
			output.add(new SubjectAttribute(itsAidPriorities.toArray(new ItsAidPriority[itsAidPriorities
					.size()])));
		if (itsAidPrioritySsps != null && itsAidPrioritySsps.size() > 0)
			output.add(new SubjectAttribute(itsAidPrioritySsps
					.toArray(new ItsAidPrioritySsp[itsAidPrioritySsps.size()])));

		return output;
	}

	public static ItsAidSsp[] extractItsAidSsp(List<PsidSspPriority> input) throws Exception {
		if (input == null)
			return new ItsAidSsp[0];

		ArrayList<IntX> itsAids = new ArrayList<>();
		ArrayList<ItsAidSsp> itsAidSsps = new ArrayList<>();
		ArrayList<ItsAidPriority> itsAidPriorities = new ArrayList<>();
		ArrayList<ItsAidPrioritySsp> itsAidPrioritySsps = new ArrayList<>();

		for (PsidSspPriority permissions : input) {
			extractItsAidPermissions(itsAids, itsAidSsps, itsAidPriorities, itsAidPrioritySsps, permissions);
		}

		if (itsAidSsps != null && itsAidSsps.size() > 0) {
			return itsAidSsps.toArray(new ItsAidSsp[itsAidSsps.size()]);
		}

		return new ItsAidSsp[0];
	}
}
