package de.fraunhofer.sit.c2x.pki.etsi_ts103097v1114.utils;

import java.io.File;
import java.io.FileWriter;
import java.io.IOException;
import java.io.PrintWriter;
import java.math.BigInteger;
import java.security.InvalidKeyException;
import java.security.KeyStore;
import java.security.KeyStoreException;
import java.security.NoSuchAlgorithmException;
import java.security.PrivateKey;
import java.security.Provider;
import java.security.PublicKey;
import java.security.Security;
import java.security.SignatureException;
import java.security.UnrecoverableKeyException;
import java.security.cert.Certificate;
import java.security.cert.CertificateException;
import java.security.cert.X509Certificate;
import java.util.Enumeration;
import java.util.Random;

import org.bouncycastle.asn1.ASN1Object;
import org.bouncycastle.asn1.ASN1Sequence;
import org.bouncycastle.asn1.DERInteger;
import org.bouncycastle.asn1.x509.SubjectPublicKeyInfo;
import org.bouncycastle.crypto.CipherParameters;
import org.bouncycastle.crypto.util.PublicKeyFactory;

import de.fraunhofer.sit.c2x.pki.etsi_ts103097v1114.impl.PublicKeyAlgorithmImpl.PublicKeyAlgorithm;
import de.fraunhofer.sit.c2x.pki.etsi_ts103097v1114.impl.EcdsaSignature;
import de.fraunhofer.sit.c2x.pki.etsi_ts103097v1114.impl.Signature;
import de.fraunhofer.sit.c2x.pki.etsi_ts103097v1114.impl.UInt8;
//import sun.security.pkcs11.SunPKCS11;

/**
 * This Class PKCS11Utils allows communication with an pkcs11 Module. The
 * required certificate should already be on the Token. The HSM is used
 * statically.
 * 
 * If there arouse Problems make sure that the correct Bit version of the
 * pkcs11.dll is used! Or check java.security if there already is an pkcs11
 * provider configured. Comment it out with an "#". You might possibly needed it
 * to use the keytool.
 * 
 * Also remember to Configure the Path to the configuration file.
 * 
 * @author gundlach (tobias.gundlach@sit.fraunhofer.de)
 * @version 28.02.2013
 */
public class EtsiPKCS11Utils {

	// Debug / Error Messages on / off
	private static boolean DEBUG = false;
	private static boolean ERROR = true;


	private static Provider pkcs11Provider;
	private static KeyStore keyStore;
	private static boolean loggedIn = false;

	/**
	 * Generates a configuration File with absolute Path. Remains in File System after use, ignore it.
	 * @param dllPath
	 * @return
	 */
	private String makeConfig(String dllPath) {
		// this is necessary, SunPKCS11 needs an absolute path
		PrintWriter pWriter = null;
		if (!dllPath.contains(":")) {
			
			dllPath = System.getProperty("user.dir") + "\\" +dllPath;
		}
		// this is necessary, SunPKCS11 needs a "real" configuration file
		try {
			File tmp = new File("tmppkcs11.cfg");
			tmp.delete();
			pWriter = new PrintWriter(new FileWriter(tmp));
			pWriter.println("name = ePass2003");
			pWriter.println("library = " + dllPath);
			pWriter.println("slot = 1");
		} catch (IOException ioe) {
			System.err.println("Creating of temporary PKCS11 config failed");
		} finally {
			if (pWriter != null)
				pWriter.flush();
			pWriter.close();
		}

		return "tmppkcs11.cfg";
	}

	/**
	 * Launches the Provider. (If not done already)
	 * @param path relative Path to the dll file, beginning with '/'
	 */
	public void initPKCS11(String path) {
		if (!loggedIn) {
			path = makeConfig(path);
			// FIXME: Requires Java 8
			//pkcs11Provider = new SunPKCS11(path);
			Security.addProvider(pkcs11Provider);
			try {
				keyStore = KeyStore.getInstance("PKCS11", pkcs11Provider);
			} catch (KeyStoreException e) {
				if (ERROR) {
					System.err.println(e.getClass().getName() + ": Could not load PKCS11 KeyStore");
					e.printStackTrace();
				}
			}
		}
	}

	/**
	 * AFTER calling the initPKCS11 function, this builds the Connection to the Token.
	 * @param password
	 *            is the password to log into HSM
	 * @return true if Login was successful
	 */
	public boolean logIntoHSM(char[] password) {
		if (!loggedIn) {
			// login
			try {
				keyStore.load(null, password);
				loggedIn = true;
				if (DEBUG)
					System.out.println("PKCS11 login successful!");
				return true;
			} catch (NoSuchAlgorithmException e) {
				if (ERROR) {
					System.err.println("Could not log into PKCS11 Keystore");
					e.printStackTrace();
				}
			} catch (CertificateException e) {
				if (ERROR) {
					System.err.println("Could not log into PKCS11 Keystore");
					e.printStackTrace();
				}
			} catch (IOException e) {
				if (ERROR) {
					System.err.println("Could not log into PKCS11 Keystore");
					e.printStackTrace();
				}
			}

			return false;
		}

		return true;
	}

	/**
	 * IT IS NOT the private Key itself, its a reference which can be only accessed from the HSM.
	 * @param alias
	 *            of the element on the Token (if only one cert stored, use
	 *            getalias method else name it directly )
	 * @param password
	 *            of the Token
	 * @return PrivateKey object of the alias if existing, else null
	 */
	public PrivateKey getPrivatePKCS11(String alias, char[] password) {
		try {
			PrivateKey privateKey = (PrivateKey) keyStore.getKey(alias, password);
			if (DEBUG)
				System.out.println("Successfully retrieved private Key from PKCS11 keystore");
			return privateKey;
		} catch (UnrecoverableKeyException e) {
			if (ERROR) {
				System.err.println("Failed to retrieve PrivateKey from PKCS11 Keystore");
				e.printStackTrace();
			}
		} catch (KeyStoreException e) {
			if (ERROR) {
				System.err.println("Failed to retrieve PrivateKey from PKCS11 Keystore");
				e.printStackTrace();
			}
		} catch (NoSuchAlgorithmException e) {
			if (ERROR) {
				System.err.println("Failed to retrieve PrivateKey from PKCS11 Keystore");
				e.printStackTrace();
			}
		}
		return null;
	}

	/**
	 * Gets the Publickey of the alias from the Token. And it is not just reference.
	 * @param alias
	 *            of the element on the Token (if only one use getalias method
	 *            or name it directly )
	 * @return PublicKey object of the alias if existing, else null.
	 */
	private PublicKey getPublicPKCS11(String alias) {
		Certificate cert = null;
		try {
			cert = keyStore.getCertificate(alias);
		} catch (KeyStoreException e) {
			if (ERROR) {
				System.err.println("Failed to retrieve PublicKey from PKCS11 Keystore");
				e.printStackTrace();
			}
			return null;
		}
		X509Certificate x509 = (X509Certificate) cert;
		PublicKey publicKey = x509.getPublicKey();
		if (DEBUG)
			System.out.println("Successfully retrieved public Key");
		publicKey.getEncoded();
		return publicKey;
	}

	/**
	 * For external access to the Public Key.
	 * 
	 * @param alias Name of the Cert
	 * @param dllPath Path to the dll File
	 * @param hsmLogin HSM Passwd
	 * @return PublicKey object of the alias if existing, else null.
	 */
	public PublicKey getPublicPKCS11(String alias, String dllPath, char[] hsmLogin) {
		initPKCS11(dllPath);
		logIntoHSM(hsmLogin);
		return getPublicPKCS11(alias);
	}

	/**
	 * Required for Creating the ETSI Signature!
	 * Converts JAVA PubKey to BC CipherParameters
	 * @param publicKey
	 *            java Public Key
	 * @return Bouncy Castle CipherParameters Key
	 */
	public CipherParameters convertKey(PublicKey publicKey) {
		byte[] keyBytes = publicKey.getEncoded();
		SubjectPublicKeyInfo subPkInfo = null;
		try {
			subPkInfo = new SubjectPublicKeyInfo((ASN1Sequence) ASN1Object.fromByteArray(keyBytes));
		} catch (IOException e) {
			if (ERROR)
				e.printStackTrace();
		}
		try {
			CipherParameters cp = (CipherParameters) PublicKeyFactory.createKey(subPkInfo);
			return cp;
		} catch (IOException e) {
			if (ERROR)
				e.printStackTrace();
		}

		return null;
	}

	/**
	 * Just for testing, good to get overview of stored cart on HSM.
	 * Aims for giving alias if only one is stored.
	 * @return If only one alias is on the token its name is returned. If there
	 *         is nothing or more than one null is returned.
	 */
	public String getAliasPKCS11() {
		Enumeration<String> aliases = null;
		try {
			aliases = keyStore.aliases();

		} catch (KeyStoreException e) {
			if (ERROR) {
				System.err.println("Failed to retrieve Aliases from PKCS11 Keystore");
				e.printStackTrace();
			}
			return null;
		}
		// count aliases
		String alias = null;
		int quantity = 0;

		while (aliases.hasMoreElements()) {
			alias = aliases.nextElement();
			if (DEBUG)
				System.out.println("Found Alias: " + alias);
			quantity++;
		}
		if (quantity == 0) {
			if (ERROR)
				System.err.println("No Alias in PKCS11 Keystore found");
			return null;
		} else if (quantity == 1) {
			if (DEBUG)
				System.err.println("Single Alias in PKCS11 Keystore found");
			return alias;
		} else if (quantity > 1) {
			if (ERROR)
				System.err
						.println("More than one Alias in PKCS11 Keystore found, access the alias directly! Dont use this Method!");
			return null;
		} else {
			if (ERROR)
				System.err.println("Sth Strange happened here.");
			return null;
		}

	}

	/**
	 * This is the mom of PKCS11 signing.
	 * 
	 * @param data
	 *            the byte array that should be signed
	 * @param algorithm
	 *            the algorithm that is used for signature
	 * @param privateKey
	 *            reference to the PrivateKey Object in the token
	 * @param alias
	 *            certificate on hsm
	 * @return the signature as ASN.1 encoded byte array if successful, else null.
	 * 
	 */
	public byte[] sign(byte[] data, String algorithm, PrivateKey privateKey) {
		java.security.Signature signer;
		byte[] signature = null;
		try {
			signer = java.security.Signature.getInstance(algorithm, pkcs11Provider);
			if (DEBUG)
				System.out.println("PKCS11 Signature getInstance with pkcs11provider ok");
		} catch (NoSuchAlgorithmException e) {
			if (ERROR) {
				System.err.println("The signature Algorithm " + algorithm + " is not supported by the HSM");
				e.printStackTrace();
			}
			return null;
		}

		try {
			signer.initSign(privateKey);
			if (DEBUG)
				System.out.println("PKCS11 Signature successfully initialized with private Key");
		} catch (InvalidKeyException e) {
			if (ERROR) {
				System.err.println("Signing initialisation with privateKey failed");
				e.printStackTrace();
			}
		}

		try {
			signer.update(data);
			if (DEBUG)
				System.out.println("PKCS11 Signature successfully updated");
		} catch (SignatureException e) {
			if (ERROR) {
				System.err.println("Signing initialisation failed with data");
				e.printStackTrace();
			}
		}

		try {
			signature = signer.sign();
			if (DEBUG)
				System.out.println("PKCS11 Signature successful");
		} catch (SignatureException e) {
			if (ERROR) {
				System.err.println("Signing Operation itself failed!");
				e.printStackTrace();
			}
		}
		return signature;

	}

	/**
	 * Use this function to get the r and s values from the signature the HSM
	 * made. Using Bouncy Castle Crypto API.
	 * 
	 * @param encoded
	 *            DER Encoded ECDSA Signature
	 * @return BigIntger {r, s}
	 */
	public BigInteger[] decodeSignature(byte[] encoded) {

		ASN1Sequence sequence = ASN1Sequence.getInstance(encoded);
		DERInteger r = (DERInteger) sequence.getObjectAt(0);
		DERInteger s = (DERInteger) sequence.getObjectAt(1);
		BigInteger[] tupel = { r.getValue(), s.getValue() };
		return tupel;

	}

	/**
	 * ECDSA ONLY, ETSI
	 * 
	 * @param signature
	 *            in byte format
	 * @return etsi signature
	 */
	public Signature convertByteToEtsiSignature(byte[] signature) {
		BigInteger[] rs = decodeSignature(signature);
		// Hardcoded fieldlength for P-256
		int fieldsize = 32;
		EcdsaSignature ecdsaSignature = new EcdsaSignature(rs[0], rs[1], new UInt8(fieldsize));
		Signature etsiSignature = new Signature(PublicKeyAlgorithm.ECDSA_NISTP256_WITH_SHA256, ecdsaSignature);
		return etsiSignature;
	}

	/**
	 * @param cert
	 *            certifikate to sign
	 * @param algorithm
	 *            Algorithm to be used, actually SHA256withECDSA is supported
	 * @param password
	 *            Token PW
	 * @param alias
	 *            name of stored cert
	 * @param dllPath
	 *            path to the dll file for the Hsm
	 * @return null or new ecdsa Signature
	 */
	public Signature signEtsiSignature(de.fraunhofer.sit.c2x.pki.etsi_ts103097v1114.impl.Certificate cert,
			String algorithm, String alias, char[] password, String dllPath) {

		byte[] data;
		try {

			initPKCS11(dllPath);
			logIntoHSM(password);

			data = cert.getUnsignedBytes();
			PrivateKey privateKey = getPrivatePKCS11(alias, password);
			byte[] signature = sign(data, algorithm, privateKey);
			return convertByteToEtsiSignature(signature);

		} catch (IOException e) {
			if (ERROR)
				System.err.println("Could not read cert unsigned Bytes!");
		}

		return null;
	}

	public boolean testVerification(String alias, char[] hsmLogin, String dllPath) {
		byte[] data = new byte[120];
		Random random = new Random();
		random.nextBytes(data);
		initPKCS11(dllPath);
		logIntoHSM(hsmLogin);
		PrivateKey privateKey = getPrivatePKCS11(alias, hsmLogin);
		byte[] signature = sign(data, "Sha256withECDSA", privateKey);
		return verifyNoPKCS11(getPublicPKCS11(alias), signature, data, "Sha256withECDSA");
	}

	/**
	 * Uses PKCS11 Provider
	 * 
	 * @param publicKey
	 *            of the signature
	 * @param signature
	 *            the signature herself
	 * @param data
	 *            the signed object without signature
	 * @param algorithm
	 *            that was used for signing
	 * @return true if verification was successful
	 */
	public boolean verify(PublicKey publicKey, byte[] signature, byte[] data, String algorithm) {
		java.security.Signature verifier = null;
		try {
			verifier = java.security.Signature.getInstance(algorithm, pkcs11Provider);
		} catch (NoSuchAlgorithmException e) {
			if (ERROR) {
				System.err.println("Verifying Operation failed! Algorithm " + algorithm
						+ " is not supported!");
				e.printStackTrace();
			}
		}
		try {
			verifier.initVerify(publicKey);
		} catch (InvalidKeyException e) {
			if (ERROR) {
				System.err.println("Verifying Operation failed! PublicKey " + publicKey + " is invalid!");
				e.printStackTrace();
			}
		}
		try {
			verifier.update(data);
		} catch (SignatureException e) {
			if (ERROR) {
				System.err.println("Updating the signature bytes failed");
				e.printStackTrace();
			}
		}
		try {
			return verifier.verify(signature);
		} catch (SignatureException e) {
			if (ERROR) {
				System.err.println("Exception during Verification");
				e.printStackTrace();
			}
		}
		return false;

	}

	/**
	 * For test purposes, use ETSI Stuff!
	 * 
	 * @param publicKey
	 * @param signature
	 * @param data
	 * @param algorithm
	 * @return true if verify successful
	 */
	public boolean verifyNoPKCS11(PublicKey publicKey, byte[] signature, byte[] data, String algorithm) {
		java.security.Signature verifier = null;
		try {
			verifier = java.security.Signature.getInstance(algorithm);
		} catch (NoSuchAlgorithmException e) {
			if (ERROR) {
				System.err.println("Verifying Operation failed! Algorithm " + algorithm
						+ " is not supported!");
				e.printStackTrace();
			}
		}
		try {
			verifier.initVerify(publicKey);
		} catch (InvalidKeyException e) {
			if (ERROR) {
				System.err.println("Verifying Operation failed! PublicKey " + publicKey + " is invalid!");
				e.printStackTrace();
			}
		}
		try {
			verifier.update(data);
		} catch (SignatureException e) {
			if (ERROR) {
				System.err.println("Updating the signature bytes failed");
				e.printStackTrace();
			}
		}
		try {
			return verifier.verify(signature);
		} catch (SignatureException e) {
			if (ERROR) {
				System.err.println("Exception during Verification");
				e.printStackTrace();
			}
		}
		return false;

	}

	/**
	 * This verification method uses the public key on the token. ETSI
	 * 
	 * @param data
	 *            sign this
	 * @param etsiSignature
	 * @param alias
	 *            of the certificate stored on the token
	 * @return true if verified correctly
	 */
	public boolean verifyEtsiSignature(byte[] data, Signature etsiSignature, String alias) {
		if (alias != null) {
			return EtsiCryptoUtils.verify(data, etsiSignature, convertKey(getPublicPKCS11(alias)));
		} else {
			System.err.println("More than one alias on Token found. Alias selection not implemented yet.");
			return false;
		}
	}

	// /**
	// * Not implemented. Use java keytool for initialization, Look at
	// documentation
	// */
	// public void initializeToken() {
	// }

	/**
	 * Little helper
	 * 
	 * @param bytearray
	 * @return The bytes as hex-string
	 */
	public static String getHex(byte[] raw) {
		String HEXES = "0123456789ABCDEF";
		if (raw == null) {
			return null;
		}
		final StringBuilder hex = new StringBuilder(2 * raw.length);
		for (final byte b : raw) {
			hex.append(HEXES.charAt((b & 0xF0) >> 4)).append(HEXES.charAt((b & 0x0F)));
		}
		return hex.toString();
	}

}
