package de.fraunhofer.sit.c2x.pki.etsi_ts103097v1114.impl;

import java.io.DataInputStream;
import java.io.DataOutputStream;
import java.io.IOException;

import de.fraunhofer.sit.c2x.pki.etsi_ts103097v1114.impl.PublicKeyAlgorithmImpl.PublicKeyAlgorithm;
import de.fraunhofer.sit.c2x.pki.etsi_ts103097v1114.serializer.Internal;
import de.fraunhofer.sit.c2x.pki.etsi_ts103097v1114.visitor.EtsiVisitor;

/**
 * 
 * @author Daniel Quanz (daniel.quanz@sit.fraunhofer.de)
 * 
 */
public class Signature extends WaveElement {

	// ---- fields ----

	@Internal(order = 0)
	private PublicKeyAlgorithm algorithm;

	@Internal(order = 1)
	private EcdsaSignature ecdsaSignature;

	@Internal(order = 2)
	private Opaque signature;

	// ---- constructors ----

	public Signature() {
	}

	public Signature(PublicKeyAlgorithm algorithm, EcdsaSignature ecdsaSignature) {
		this.algorithm = algorithm;
		this.ecdsaSignature = ecdsaSignature;
	}

	public Signature(DataInputStream in) throws IOException {
		algorithm = PublicKeyAlgorithmImpl.getInstance().getEnumType(
				in.readByte());
		if (algorithm == PublicKeyAlgorithm.ECDSA_NISTP256_WITH_SHA256) {
			ecdsaSignature = new EcdsaSignature(in, new UInt8(32), algorithm);
		} else {
			signature = new Opaque(in);
		}
	}

	// ---- accept ----

	public <T> T accept(EtsiVisitor<T> visitor) {
		return visitor.visit(this);
	}

	// ---- getter ----

	public PublicKeyAlgorithm getAlgorithm() {
		return this.algorithm;
	}

	public EcdsaSignature getEcdsaSignature() {
		return this.ecdsaSignature;
	}

	public Opaque getSignature() {
		return this.signature;
	}

	// ---- setter ----

	public void setAlgorithm(PublicKeyAlgorithm algorithm) {
		this.algorithm = algorithm;
	}

	public void setEcdsaSignature(EcdsaSignature ecdsaSignature) {
		this.ecdsaSignature = ecdsaSignature;
	}

	public void setSignature(Opaque signature) {
		this.signature = signature;
	}

	@Override
	public int writeData(DataOutputStream out) throws IOException {
		if (algorithm == null)
			throw new IllegalArgumentException();

		int written = PublicKeyAlgorithmImpl.getInstance().writeData(out,
				algorithm);
		if (algorithm == PublicKeyAlgorithm.ECDSA_NISTP256_WITH_SHA256) {
			if (ecdsaSignature == null)
				throw new IllegalArgumentException();
			written += ecdsaSignature.writeData(out);
		} else {
			if (signature == null)
				throw new IllegalArgumentException();
			written += signature.writeData(out);
		}
		return written;
	}
}