package de.fraunhofer.sit.c2x.pki.etsi_ts103097v1114.impl;

import java.io.DataInputStream;
import java.io.DataOutputStream;
import java.io.IOException;
import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;
import java.util.Arrays;

import de.fraunhofer.sit.c2x.pki.ca.core.interfaces.WaveType;
import de.fraunhofer.sit.c2x.pki.ca.utils.WaveUtils;
import de.fraunhofer.sit.c2x.pki.etsi_ts103097v1114.serializer.External;
import de.fraunhofer.sit.c2x.pki.etsi_ts103097v1114.serializer.Internal;
import de.fraunhofer.sit.c2x.pki.etsi_ts103097v1114.visitor.EtsiVisitor;

/**
 * 
 * @author Daniel Quanz (daniel.quanz@sit.fraunhofer.de)
 * 
 */
public class Opaque extends WaveElement {

	@Override
	public int hashCode() {
		final int prime = 31;
		int result = 1;
		result = prime * result + (array ? 1231 : 1237);
		result = prime * result + Arrays.hashCode(data);
		return result;
	}

	@Internal(order = 0)
	private byte[] data;

	@External
	private boolean array = false;

	public Opaque() {

	}

	public Opaque(DataInputStream in, int length, boolean array)
			throws IOException {

		if (length < 0)
			throw new IllegalArgumentException(
					"Negative array length as input parameter!");

		this.array = array;

		this.data = new byte[length];
		in.readFully(data);

	}

	public Opaque(DataInputStream in) throws IOException {
		int length = WaveUtils.readVariableLengthField(in);
		this.data = new byte[length];
		in.readFully(data);
		array = false;
	}

	public Opaque(byte[] bs) {
		this(bs, false);
	}

	public Opaque(byte[] bs, boolean array) {
		data = bs;
		this.array = array;

	}

	public Opaque(byte[] bs, int length, boolean b) {
		this.data = new byte[length];
		System.arraycopy(bs, 0, data, 0, length);
		array = b;
	}

	public <T> T accept(EtsiVisitor<T> visitor) {
		return visitor.visit(this);
	}

	public void set(byte[] data) {
		this.data = data;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see de.fraunhofer.sit.c2x.pki.ca.types.etsi.v0_0_4.impl.Opaque#isArray()
	 */
	public boolean isArray() {
		return array;
	}

	public void setArray(boolean array) {
		this.array = array;
	}

	@Override
	public int writeData(DataOutputStream out) throws IOException {

		int written = 0;
		if (!isArray()) {
			written += WaveUtils.writeVariableLengthField(getLength(), out);
		}
		out.write(data);
		written += data.length;
		return written;
	}

	public byte[] get() {
		return data;
	}

	public int getLength() {
		return data.length;
	}
	
	public static Opaque generateHash(WaveType type, int size) {
		try {
			MessageDigest sha256 = MessageDigest.getInstance("SHA-256");
			byte[] digest = sha256.digest(WaveUtils.getBytesFromWaveType(type));
			byte[] hashedID8 = new byte[size];
			System.arraycopy(digest, digest.length - size, hashedID8, 0, hashedID8.length);
			Opaque o = new Opaque(hashedID8, size, true);
			return o;
		} catch (NoSuchAlgorithmException e) {
			throw new RuntimeException(e);
		}
	}

}
