package de.fraunhofer.sit.c2x.pki.etsi_ts103097v1114.impl;

import java.io.DataInputStream;
import java.io.DataOutputStream;
import java.io.IOException;

import de.fraunhofer.sit.c2x.pki.etsi_ts103097v1114.impl.SymmetricAlgorithmImpl.SymmAlgorithm;
import de.fraunhofer.sit.c2x.pki.etsi_ts103097v1114.serializer.Internal;
import de.fraunhofer.sit.c2x.pki.etsi_ts103097v1114.visitor.EtsiVisitor;

/**
 * 
 * @author Daniel Quanz (daniel.quanz@sit.fraunhofer.de)
 * 
 */
public class EncryptionParameters extends WaveElement {

	// ---- fields ----

	@Internal(order = 0)
	private SymmAlgorithm symmAlgorithm;

	@Internal(order = 1)
	private Opaque nonce;

	@Internal(order = 2)
	private Opaque params;

	// ---- fields ----

	// ---- constructors ----

	public EncryptionParameters() {
	}

	public EncryptionParameters(SymmAlgorithm symmAlgorithm, Opaque nonce) {
		super();
		
		if (symmAlgorithm == SymmAlgorithm.AES_128_CCM) {
			if(nonce == null || nonce.getLength() != 12) throw new IllegalArgumentException("In case of AES_128_CCM the nonce length must be 12!");
		}
		
		this.symmAlgorithm = symmAlgorithm;
		this.nonce = nonce;
	}

	public EncryptionParameters(DataInputStream in) throws IOException {
		symmAlgorithm = SymmetricAlgorithmImpl.getInstance().getEnumType(
				in.readByte());
		if (symmAlgorithm == SymmAlgorithm.AES_128_CCM) {
			nonce = new Opaque(in, 12, true);
		} else {
			params = new Opaque(in);
		}
	}

	// ---- accept ----

	public <T> T accept(EtsiVisitor<T> visitor) {
		return visitor.visit(this);
	}

	// ---- getter ----

	public SymmAlgorithm getSymmAlgorithm() {
		return this.symmAlgorithm;
	}

	public Opaque getNonce() {
		return this.nonce;
	}

	public Opaque getParams() {
		return this.params;
	}

	// ---- setter ----

	public void setSymmAlgorithm(SymmAlgorithm symmAlgorithm) {
		this.symmAlgorithm = symmAlgorithm;
	}

	public void setNonce(Opaque nonce) {
		this.nonce = nonce;
	}

	public void setParams(Opaque params) {
		this.params = params;
	}

	@Override
	public int writeData(DataOutputStream out) throws IOException {

		if (symmAlgorithm == null)
			throw new IllegalArgumentException();

		int written = SymmetricAlgorithmImpl.getInstance().writeData(out,
				symmAlgorithm);
		if (symmAlgorithm == SymmAlgorithm.AES_128_CCM) {
			if (nonce == null || !nonce.isArray() || nonce.getLength() != 12)
				throw new IllegalArgumentException(
						"arg may not be null and must be an array of size 12");
			written += nonce.writeData(out);
		} else {
			if (params == null)
				throw new IllegalArgumentException("arg may not be null");
			written += params.writeData(out);
		}

		return written;
	}
}