package de.fraunhofer.sit.c2x.pki.etsi_ts103097v1114.impl;

import java.io.DataInputStream;
import java.io.DataOutputStream;
import java.io.IOException;

import de.fraunhofer.sit.c2x.pki.etsi_ts103097v1114.impl.PublicKeyAlgorithmImpl.PublicKeyAlgorithm;
import de.fraunhofer.sit.c2x.pki.etsi_ts103097v1114.impl.SymmetricAlgorithmImpl.SymmAlgorithm;
import de.fraunhofer.sit.c2x.pki.etsi_ts103097v1114.serializer.External;
import de.fraunhofer.sit.c2x.pki.etsi_ts103097v1114.serializer.Internal;
import de.fraunhofer.sit.c2x.pki.etsi_ts103097v1114.visitor.EtsiVisitor;

/**
 * 
 * @author Daniel Quanz (daniel.quanz@sit.fraunhofer.de)
 * 
 */
public class EciesEncryptedKey extends WaveElement {
	public static final int TAG_LENGTH = 16;

	// ---- fields ----

	@Internal(order = 0)
	private EccPoint v;

	@Internal(order = 1)
	private Opaque c;

	@Internal(order = 2)
	private Opaque t;

	// ---- fields ----

	@External
	private SymmAlgorithm symmAlg;

	@External
	private UInt32 symmKeyLen;

	// ---- constructors ----

	public EciesEncryptedKey() {
	}

	public EciesEncryptedKey(EccPoint v, Opaque c, Opaque t, SymmAlgorithm symmAlg, UInt32 symmKeyLen) {
		this.v = v;
		this.c = c;
		this.t = t;
		this.symmAlg = symmAlg;
		this.symmKeyLen = symmKeyLen;
	}

	public EciesEncryptedKey(DataInputStream in, SymmAlgorithm symmAlg) throws IOException {
		this.symmAlg = symmAlg;
		this.symmKeyLen = new UInt32(symmAlg.getKeyLength() / 8);
		this.v = new EccPoint(in, new UInt8(32), PublicKeyAlgorithm.ECIES_NISTP256);
		this.c = new Opaque(in, (int) symmKeyLen.get(), true);
		this.t = new Opaque(in, 16, true);
	}

	// ---- accept ----

	public <T> T accept(EtsiVisitor<T> visitor) {
		return visitor.visit(this);
	}

	// ---- getter ----

	public EccPoint getV() {
		return this.v;
	}

	public Opaque getC() {
		return this.c;
	}

	public Opaque getT() {
		return this.t;
	}

	public SymmAlgorithm getSymmAlg() {
		return this.symmAlg;
	}

	public UInt32 getSymmKeyLen() {
		return this.symmKeyLen;
	}

	// ---- setter ----

	public void setV(EccPoint v) {
		this.v = v;
	}

	public void setC(Opaque c) {
		this.c = c;
	}

	public void setT(Opaque t) {
		this.t = t;
	}

	public void setSymmAlg(SymmAlgorithm symmAlg) {
		this.symmAlg = symmAlg;
	}

	public void setSymmKeyLen(UInt32 symmKeyLen) {
		this.symmKeyLen = symmKeyLen;
	}

	@Override
	public int writeData(DataOutputStream out) throws IOException {
		int written = 0;
		if (v == null || c == null || t == null)
			throw new IllegalArgumentException();
		written += v.writeData(out);
		if (!c.isArray() || c.getLength() != symmKeyLen.get())
			throw new IllegalArgumentException();
		written += c.writeData(out);
		if (!t.isArray() || t.getLength() != TAG_LENGTH)
			throw new IllegalArgumentException(
					"Byte arrary for agrument t has invalid length: expected length = " + TAG_LENGTH
							+ ", given length = " + t.getLength());
		written += t.writeData(out);
		return written;
	}

}