package de.fraunhofer.sit.c2x.pki.etsi_ts103097v1114.impl;

import java.io.ByteArrayOutputStream;
import java.io.DataInputStream;
import java.io.DataOutputStream;
import java.io.IOException;
import java.math.BigInteger;
import java.util.Collection;
import java.util.NoSuchElementException;
import java.util.SortedMap;
import java.util.TreeMap;

import org.apache.commons.codec.binary.Hex;

import de.fraunhofer.sit.c2x.pki.ca.core.exceptions.HandlerException;
import de.fraunhofer.sit.c2x.pki.ca.crypto.CryptoUtils;
import de.fraunhofer.sit.c2x.pki.ca.crypto.Digest;
import de.fraunhofer.sit.c2x.pki.ca.utils.WaveUtils;
import de.fraunhofer.sit.c2x.pki.etsi_ts103097v1114.impl.EccPointTypeImpl.EccPointType;
import de.fraunhofer.sit.c2x.pki.etsi_ts103097v1114.impl.RegionTypeImpl.RegionType;
import de.fraunhofer.sit.c2x.pki.etsi_ts103097v1114.impl.SignerInfoTypeImpl.SignerInfoType;
import de.fraunhofer.sit.c2x.pki.etsi_ts103097v1114.impl.SubjectAttributeTypeImpl.SubjectAttributeType;

/**
 * 
 * @author Daniel Quanz (daniel.quanz@sit.fraunhofer.de)
 * 
 */
public class Certificate extends WaveElement {
	public static final int VERSION = 2;
	private UInt8 version = new UInt8(VERSION); // Default value: 2
	private SignerInfo signerInfo;
	private SubjectInfo subjectInfo;
	private SortedMap<Integer, SubjectAttribute> subjectAttributes;
	private SortedMap<Integer, ValidityRestriction> validityRestrictions;
	private Signature signature;

	public Certificate() {
		this.subjectAttributes = new TreeMap<Integer, SubjectAttribute>();
		this.validityRestrictions = new TreeMap<Integer, ValidityRestriction>();
	}

	public Certificate(DataInputStream in) throws IOException {
		this.subjectAttributes = new TreeMap<Integer, SubjectAttribute>();
		this.validityRestrictions = new TreeMap<Integer, ValidityRestriction>();

		setVersionAndType(new UInt8(in));
		if (version.get() != VERSION)
			throw new IllegalArgumentException("Invalid certificate version: " + version.get()
					+ ". Expected version = " + VERSION);

		signerInfo = new SignerInfo(in);
		subjectInfo = new SubjectInfo(in);
		SubjectAttribute[] subjectAttributeArray = WaveUtils.getArrayFromStream(in, SubjectAttribute.class);
		for (SubjectAttribute sa : subjectAttributeArray) {
			subjectAttributes.put(SubjectAttributeTypeImpl.getInstance().getCode(sa.getType()), sa);
		}
		ValidityRestriction[] validityRestrictionArray = WaveUtils.getArrayFromStream(in,
				ValidityRestriction.class);
		for (ValidityRestriction va : validityRestrictionArray) {
			validityRestrictions.put(ValidityRestrictionTypeImpl.getInstance().getCode(va.getType()), va);
		}
		signature = new Signature(in);
	}

	public PublicKey getVerificationKey() {
		for (SubjectAttribute sa : subjectAttributes.values()) {
			if (sa.getType() == SubjectAttributeType.VERIFICATION_KEY)
				return sa.getKey();
		}
		return null;
	}

	public PublicKey getEncryptionKey() {
		for (SubjectAttribute sa : subjectAttributes.values()) {
			if (sa.getType() == SubjectAttributeType.ENCRYPTION_KEY)
				return sa.getKey();
		}
		return null;
	}

	public SubjectAssurance getSubjectAssurance() {
		for (SubjectAttribute sa : subjectAttributes.values()) {
			switch (sa.getType()) {
			case ASSURANCE_LEVEL:
				return sa.getAssuranceLevel();
			default:
				break;
			}
		}
		throw new NoSuchElementException("cert contains no subject assurance level");
	}

	public void setVersionAndType(UInt8 version) {
		this.version = version;
		if (version.get() != VERSION)
			throw new IllegalArgumentException("Invalid certificate version: " + version.get()
					+ ". Expected version = " + VERSION);
	}

	public SubjectInfo getSubjectInfo() {
		return subjectInfo;
	}

	public void setSubjectInfo(SubjectInfo subjectInfo) {
		this.subjectInfo = subjectInfo;
	}

	public SubjectAttribute[] getSubjectAttributes() {
		Collection<SubjectAttribute> sac = subjectAttributes.values();
		SubjectAttribute[] sa = new SubjectAttribute[sac.size()];
		return sac.toArray(sa);
	}

	public void setSubjectAttributes(SubjectAttribute[] subjectAttributes) {
		for (SubjectAttribute sa : subjectAttributes) {
			if (this.subjectAttributes.containsKey(SubjectAttributeTypeImpl.getInstance().getCode(
					sa.getType())))
				throw new IllegalArgumentException("SignerInfo " + sa.getType() + " already in list");
			this.subjectAttributes.put(SubjectAttributeTypeImpl.getInstance().getCode(sa.getType()), sa);
		}
	}

	public ValidityRestriction[] getValidityRestrictions() {
		Collection<ValidityRestriction> vrc = validityRestrictions.values();
		ValidityRestriction[] vr = new ValidityRestriction[vrc.size()];
		return vrc.toArray(vr);
	}

	public Time32 getStartTime() throws NoSuchElementException {
		for (ValidityRestriction vr : validityRestrictions.values()) {
			switch (vr.getType()) {
			case TIME_START_AND_DURATION:
			case TIME_START_AND_END:
				return vr.getStartValidity();
			default:
				break;
			}
		}
		throw new NoSuchElementException("cert contains no start time");
	}

	public Time32 getEndTime() throws NoSuchElementException {
		for (ValidityRestriction vr : validityRestrictions.values()) {
			switch (vr.getType()) {
			case TIME_START_AND_DURATION:
				int seconds = vr.getDuration().getDuration().get();
				Time32 expiration = vr.getStartValidity().addSeconds(seconds);
				return expiration;
			case TIME_START_AND_END:
			case TIME_END:
				return vr.getEndValidity();
			default:
				break;
			}
		}
		throw new NoSuchElementException("cert contains no end time");
	}

	public void setValidityRestrictions(ValidityRestriction[] validityRestrictions) {
		for (ValidityRestriction va : validityRestrictions) {
			if (this.validityRestrictions.containsKey(ValidityRestrictionTypeImpl.getInstance().getCode(
					va.getType())))
				throw new IllegalArgumentException("SignerInfo " + va.getType() + " already in list");
			this.validityRestrictions
					.put(ValidityRestrictionTypeImpl.getInstance().getCode(va.getType()), va);
		}
	}

	public RegionType getRegionType() {
		for (ValidityRestriction vr : validityRestrictions.values()) {
			switch (vr.getType()) {
			case REGION:
				return vr.getRegion().getRegionType();
			default:
				break;
			}
		}
		throw new NoSuchElementException("cert contains no region restriction");
	}

	public IdentifiedRegion getIdentifiedRegionRestriction() {
		for (ValidityRestriction vr : validityRestrictions.values()) {
			switch (vr.getType()) {
			case REGION:
				switch (vr.getRegion().getRegionType()) {
				case ID:
					return vr.getRegion().getIdRegion();
				default:
					break;
				}
			default:
				break;
			}
		}
		throw new NoSuchElementException("cert contains no identified region restriction");
	}

	public CircularRegion getCircularRegionRestriction() {
		for (ValidityRestriction vr : validityRestrictions.values()) {
			switch (vr.getType()) {
			case REGION:
				switch (vr.getRegion().getRegionType()) {
				case CIRCLE:
					return vr.getRegion().getCircularRegion();
				default:
					break;
				}
			default:
				break;
			}
		}
		throw new NoSuchElementException("cert contains no circular region restriction");
	}

	public RectangularRegion[] getRectangularRegionRestriction() {
		for (ValidityRestriction vr : validityRestrictions.values()) {
			switch (vr.getType()) {
			case REGION:
				switch (vr.getRegion().getRegionType()) {
				case RECTANGLE:
					return vr.getRegion().getRectangularRegion();
				default:
					break;
				}
			default:
				break;
			}
		}
		throw new NoSuchElementException("cert contains no rectangular region restriction");
	}

	public PolygonalRegion getPolygonalRegionRestriction() {
		for (ValidityRestriction vr : validityRestrictions.values()) {
			switch (vr.getType()) {
			case REGION:
				switch (vr.getRegion().getRegionType()) {
				case POLYGON:
					return vr.getRegion().getPolygonalRegion();
				default:
					break;
				}
			default:
				break;
			}
		}
		throw new NoSuchElementException("cert contains no polygonal region restriction");
	}

	public IntX[] getItsAidList() {
		for (SubjectAttribute sa : subjectAttributes.values()) {
			switch (sa.getType()) {
			case ITS_AID_LIST:
				return sa.getItsAidList();
			default:
				break;
			}
		}
		throw new NoSuchElementException("cert contains no ITS AID list");
	}

	public String getItsAidListString() {
		IntX[] aids = getItsAidList();
		String aidListString = "";
		for (IntX i : aids) {
			if (aidListString.isEmpty())
				aidListString += "AID=" + i.getValue();
			else
				aidListString += ", AID=" + i.getValue();
		}
		return aidListString;
	}

	public ItsAidSsp[] getItsAidSspList() {
		for (SubjectAttribute sa : subjectAttributes.values()) {
			switch (sa.getType()) {
			case ITS_AID_SSP_LIST:
				return sa.getItsAidSspList();
			default:
				break;
			}
		}
		throw new NoSuchElementException("cert contains no ITS AID SSP list");
	}

	public String getItsAidSspListString() {
		ItsAidSsp[] aidSsps = getItsAidSspList();
		String aidListString = "";
		for (ItsAidSsp i : aidSsps) {
			if (aidListString.isEmpty())
				aidListString += "AID=" + i.getItsAid().getValue() + " + SSP=0x"
						+ Hex.encodeHexString(i.getServiceSpecificPermissions().get());
			else
				aidListString += ", AID=" + i.getItsAid().getValue() + " + SSP=0x"
						+ Hex.encodeHexString(i.getServiceSpecificPermissions().get());
		}
		return aidListString;
	}

	public SignerInfo getSignerInfo() {
		return signerInfo;
	}

	public String getIssuerType() {
		if (signerInfo.getType() == null || signerInfo == null)
			throw new IllegalArgumentException("Issuer type not given in certificate");
		return signerInfo.getType().toString();
	}

	public HashedId8 getIssuer() {
		if (signerInfo.getType() == SignerInfoType.CERTIFICATE)
			return signerInfo.getCertificate().getHashedId8();
		if (signerInfo.getType() == SignerInfoType.CERTIFICATE_CHAIN) {
			int size = signerInfo.getCertificates().length;
			return signerInfo.getCertificates()[(size - 1)].getHashedId8();
		}
		if (signerInfo.getType() == SignerInfoType.CERTIFICATE_DIGEST_WITH_ECDSAP256)
			return signerInfo.getDigest();
		if (signerInfo.getType() == SignerInfoType.SELF)
			return getHashedId8();
		else
			throw new IllegalArgumentException("Cert issuer not supported: " + signerInfo.getType());
	}

	public void setSignerInfo(SignerInfo signerInfo) {
		this.signerInfo = signerInfo;
	}

	public void setSignature(Signature signature) {
		this.signature = signature;
	}

	@Override
	public int writeData(DataOutputStream out) throws IOException {
		int written = WaveUtils.writeWave(out, version);
		written += WaveUtils.writeWave(out, signerInfo);
		written += WaveUtils.writeWave(out, subjectInfo);
		written += WaveUtils.writeArrayToStream(out, getSubjectAttributes());
		written += WaveUtils.writeArrayToStream(out, getValidityRestrictions());
		written += WaveUtils.writeWave(out, signature);
		return written;
	}

	public Signature getSignature() {
		return signature;
	}

	public UInt8 getVersionAndType() {
		return version;
	}

	public byte[] getUnsignedBytes() throws IOException {
		ByteArrayOutputStream baos = new ByteArrayOutputStream();
		DataOutputStream out = new DataOutputStream(baos);

		WaveUtils.writeWave(out, version);
		WaveUtils.writeWave(out, signerInfo);
		WaveUtils.writeWave(out, subjectInfo);
		WaveUtils.writeArrayToStream(out, getSubjectAttributes());
		WaveUtils.writeArrayToStream(out, getValidityRestrictions());

		baos.close();
		out.close();

		return baos.toByteArray();
	}

	public HashedId8 getHashedId8() {
		if (signature == null)
			throw new IllegalArgumentException("No Signature availanle in certificate");

		EccPointType originalEccPointType = signature.getEcdsaSignature().getR().getType();
		if (originalEccPointType != EccPointType.X_COORDINATE_ONLY) {
			signature.getEcdsaSignature().getR().setType(EccPointType.X_COORDINATE_ONLY);
		}

		byte[] digest = Digest.getInstance().sha256(WaveUtils.getBytesFromWaveType(this));
		int length = 8;
		byte[] id8 = new byte[length];
		System.arraycopy(digest, digest.length - length, id8, 0, id8.length);
		HashedId8 hashedId8 = new HashedId8();
		Opaque op = new Opaque();
		op.set(id8);
		op.setArray(true);
		hashedId8.set(op);

		if (originalEccPointType != EccPointType.X_COORDINATE_ONLY) {
			signature.getEcdsaSignature().getR().setType(originalEccPointType);
		}

		return hashedId8;
	}

	public byte[] getBytes() throws IOException {
		ByteArrayOutputStream bos = new ByteArrayOutputStream();
		DataOutputStream dos = new DataOutputStream(bos);
		this.writeData(dos);
		dos.flush();
		return bos.toByteArray();
	}

	public boolean verfiySignature() throws IOException, HandlerException {
		Certificate issuerCert = null;

		switch (signerInfo.getType()) {
		case CERTIFICATE:
			issuerCert = signerInfo.getCertificate();
			break;
		case CERTIFICATE_CHAIN:
			throw new HandlerException("Certificate chain as issuer not allowed in certificate.");
		case SELF:
			issuerCert = this;
			break;
		default:
			throw new HandlerException(
					"Unable to verify signature because issuer public key not contained in certificate");
		}

		return verfiySignature(issuerCert);
	}

	public boolean verfiySignature(Certificate issuerCert) throws IOException {
		byte[] messageToVerify = getUnsignedBytes();
		BigInteger[] signature = getSignature().getEcdsaSignature().toBigIntegers();
		return CryptoUtils.verifyECDSA(messageToVerify, signature, issuerCert.getVerificationKey()
				.getPublicKey().toECPublicKeyParameters());
	}
}
