package de.fraunhofer.sit.c2x.pki.etsi_ts103097v1114.handler;

import java.io.ByteArrayOutputStream;
import java.io.DataOutputStream;
import java.io.IOException;
import java.security.cert.X509Certificate;
import java.util.ArrayList;
import java.util.List;

import org.apache.log4j.Logger;

import com.google.inject.Inject;

import de.fraunhofer.sit.c2x.pki.ca.certificates.datacontainers.GeographicRegionDataContainer;
import de.fraunhofer.sit.c2x.pki.ca.certificates.datacontainers.PsidSspPriorityDataContainer;
import de.fraunhofer.sit.c2x.pki.ca.core.exceptions.HandlerException;
import de.fraunhofer.sit.c2x.pki.ca.core.logging.InjectLogger;
import de.fraunhofer.sit.c2x.pki.ca.module.webserver.WebServerWithClientAuth;
import de.fraunhofer.sit.c2x.pki.ca.module.webserver.servlets.interfaces.AuthenticatorWebHandler;
import de.fraunhofer.sit.c2x.pki.ca.provider.ProviderException;
import de.fraunhofer.sit.c2x.pki.ca.provider.entities.Authenticator;
import de.fraunhofer.sit.c2x.pki.ca.provider.interfaces.AuthenticatorProvider;
import de.fraunhofer.sit.c2x.pki.ca.provider.interfaces.CaInfoProvider;
import de.fraunhofer.sit.c2x.pki.ca.provider.interfaces.ConfigProvider;
import de.fraunhofer.sit.c2x.pki.ca.utils.WaveUtils;
import de.fraunhofer.sit.c2x.pki.ca.validator.region.GeorgraphicRegionValidator;
import de.fraunhofer.sit.c2x.pki.etsi_ts103097v1114.impl.Certificate;
import de.fraunhofer.sit.c2x.pki.etsi_ts103097v1114.impl.GeographicRegion;
import de.fraunhofer.sit.c2x.pki.etsi_ts103097v1114.impl.IntX;
import de.fraunhofer.sit.c2x.pki.etsi_ts103097v1114.impl.ItsAidPriority;
import de.fraunhofer.sit.c2x.pki.etsi_ts103097v1114.impl.ItsAidPrioritySsp;
import de.fraunhofer.sit.c2x.pki.etsi_ts103097v1114.impl.ItsAidSsp;
import de.fraunhofer.sit.c2x.pki.etsi_ts103097v1114.impl.RegionTypeImpl.RegionType;
import de.fraunhofer.sit.c2x.pki.etsi_ts103097v1114.impl.SubjectAttribute;
import de.fraunhofer.sit.c2x.pki.etsi_ts103097v1114.impl.ValidityRestriction;
import de.fraunhofer.sit.c2x.pki.etsi_ts103097v1114.impl.ValidityRestrictionTypeImpl.ValidityRestrictionType;
import de.fraunhofer.sit.c2x.pki.etsi_ts103097v1114.utils.EtsiPerrmissionUtils;
import de.fraunhofer.sit.c2x.pki.etsi_ts103097v1114.utils.EtsiRegionUtils;

public class EtsiRegisterAuthenticatorWebHandler implements AuthenticatorWebHandler {

	@Inject
	private AuthenticatorProvider ap;

	@Inject
	private WebServerWithClientAuth webServerWithClientAuth;

	@Inject
	protected CaInfoProvider<Certificate> caInfo;

	@InjectLogger
	private Logger logger;

	@Inject
	private GeorgraphicRegionValidator georgraphicRegionValidator;

	@Inject
	private ConfigProvider configProvider;

	@Override
	public boolean handleAuthenticatorRegistration(String company, String name, String email,
			String telephone, String address, byte[] assuranceLevel, String certificateString,
			List<PsidSspPriorityDataContainer> permissions, GeographicRegionDataContainer region,
			String httpUserId) throws HandlerException {

		// get the CA certificate in order to check permissions
		Certificate caCert;
		try {
			caCert = caInfo.getCaCertificate();
		} catch (ProviderException e) {
			throw new HandlerException(e);
		}
		if (caCert == null)
			throw new HandlerException("No CA certificate available, canceling authenticator registration!");

		// read abstract permission data into format specific structure
		ArrayList<SubjectAttribute> subjectAttributes = null;
		IntX[] itsAids = null;
		byte[] itsAidsBytes = null;
		ItsAidSsp[] itsAidSsps = null;
		byte[] itsAidSspsBytes = null;
		ItsAidPriority[] itsAidPriorities = null;
		byte[] itsAidPrioritiesBytes = null;
		ItsAidPrioritySsp[] itsAidPrioritySsps = null;
		byte[] itsAidPrioritySspsBytes = null;
		if (permissions != null && permissions.size() > 0) {
			subjectAttributes = EtsiPerrmissionUtils.psidSetToSubjectAttributeArray(permissions);

			// check the permissions
			EtsiPerrmissionUtils.checkSubjectAttributes(subjectAttributes, caCert.getSubjectAttributes());
			// TODO check policies
			if (logger != null && logger.isDebugEnabled()) {
				logger.debug("ItsAid-check successful: Permissions of the authenticator comply with the LTCA permissions!");
			}

			for (SubjectAttribute sa : subjectAttributes) {
				switch (sa.getType()) {
				case ITS_AID_LIST:
					itsAids = sa.getItsAidList();
					if (itsAids != null && itsAids.length > 0) {
						try {
							ByteArrayOutputStream bos = new ByteArrayOutputStream();
							DataOutputStream dos = new DataOutputStream(bos);
							WaveUtils.writeArrayToStream(dos, itsAids);
							dos.flush();
							itsAidsBytes = bos.toByteArray();
						} catch (IOException e) {
							throw new HandlerException("Unable to encode ItsAidSspList");
						}
					}
					break;
				case ITS_AID_SSP_LIST:
					itsAidSsps = sa.getItsAidSspList();
					if (itsAidSsps != null && itsAidSsps.length > 0) {
						try {
							ByteArrayOutputStream bos = new ByteArrayOutputStream();
							DataOutputStream dos = new DataOutputStream(bos);
							WaveUtils.writeArrayToStream(dos, itsAidSsps);
							dos.flush();
							itsAidSspsBytes = bos.toByteArray();
						} catch (IOException e) {
							throw new HandlerException("Unable to encode ItsAidSspList");
						}
					}
					break;
				case PRIORITY_ITS_AID_LIST:
					itsAidPriorities = sa.getItsAidPriorityList();
					if (itsAidPriorities != null && itsAidPriorities.length > 0) {
						try {
							ByteArrayOutputStream bos = new ByteArrayOutputStream();
							DataOutputStream dos = new DataOutputStream(bos);
							WaveUtils.writeArrayToStream(dos, itsAidPriorities);
							dos.flush();
							itsAidPrioritiesBytes = bos.toByteArray();
						} catch (IOException e) {
							throw new HandlerException("Unable to encode ItsAidPriorityList");
						}
					}
					break;
				case PRIORITY_SSP_LIST:
					itsAidPrioritySsps = sa.getItsAidPrioritySspList();
					if (itsAidPrioritySsps != null && itsAidPrioritySsps.length > 0) {
						try {
							ByteArrayOutputStream bos = new ByteArrayOutputStream();
							DataOutputStream dos = new DataOutputStream(bos);
							WaveUtils.writeArrayToStream(dos, itsAidPrioritySsps);
							dos.flush();
							itsAidPrioritySspsBytes = bos.toByteArray();
						} catch (IOException e) {
							throw new HandlerException("Unable to encode ItsAidPrioritySspList");
						}
					}
					break;
				default:
					throw new HandlerException("Invalid subject attribute found: " + sa.getType()
							+ ". Please provide only AIDs, SSPs, and / or Priorities");
				}
			}

		}

		// read abstract region data into format specific structure
		GeographicRegion geographicRegion = EtsiRegionUtils.regionContainerToGeographicRegion(region);

		GeographicRegion caRegion = null;

		for (ValidityRestriction v : caCert.getValidityRestrictions()) {
			if (v.getType() == ValidityRestrictionType.REGION) {
				caRegion = v.getRegion();
				break;
			}
		}

		if (caRegion == null)
			throw new HandlerException(
					"The CA certificate is invalid as it contains no GeographicRegion; canceling authenticator registration!");

		// validate region vs. LTCA certificate
		if (geographicRegion == null) {
			geographicRegion = new GeographicRegion();
			geographicRegion.setRegionType(RegionType.NONE);
		}

		georgraphicRegionValidator.setRegion(caRegion);
		georgraphicRegionValidator.validate(geographicRegion);
		if (logger != null && logger.isDebugEnabled()) {
			logger.debug("Region-check successful: The region of the authenticator complies with the LTCA region!");
		}

		byte[] circularRegionBytes = null;
		byte[] identifiedRegionBytes = null;
		switch (geographicRegion.getRegionType()) {
		case ID:
			identifiedRegionBytes = WaveUtils.getBytesFromWaveType(geographicRegion.getIdRegion());
			break;
		case CIRCLE:
			circularRegionBytes = WaveUtils.getBytesFromWaveType(geographicRegion.getCircularRegion());
			break;
		case NONE:
			break;
		default:
			throw new HandlerException("Invalid region found: " + geographicRegion.getRegionType()
					+ ". Please provide only IdentifiedRegion or CircularRegion");
		}

		try {
			if (company != null && name != null && email != null && telephone != null && address != null
					&& assuranceLevel != null && certificateString != null && httpUserId != null
					&& httpUserId.length() > 0) {
				Authenticator authenticator = new Authenticator();
				authenticator.setCompany(company);
				authenticator.setName(name);
				authenticator.setEmail(email);
				authenticator.setTelephone(telephone);
				authenticator.setAddress(address);
				authenticator.setSubjectAssurance(assuranceLevel);
				authenticator.setCertificate(certificateString);
				authenticator.setItsAidList(itsAidsBytes);
				authenticator.setItsAidSspList(itsAidSspsBytes);
				authenticator.setPriorityItsAidList(itsAidPrioritiesBytes);
				authenticator.setPrioritySspList(itsAidPrioritySspsBytes);
				authenticator.setIdRegion(identifiedRegionBytes);
				authenticator.setCircularRegion(circularRegionBytes);
				authenticator.setHttpUserId(httpUserId);

				X509Certificate x509Cert = authenticator.getX509Certificate();
				String x509Name = x509Cert.getSubjectDN().getName();
				if (configProvider.getBoolean("webpageAddAuthenticatorCheckEmailInCert", true)) {
					if (x509Name.contains("EMAILADDRESS=" + email + ",") == false)
						throw new IOException(
								"Email address is not matching with Email address in the given certificate. X.509 certificate DN: "
										+ x509Name);

				}
				if (x509Name.contains("O=" + company) == false)
					throw new IOException(
							"Company is not matching with organization in the given certificate. X.509 certificate DN: "
									+ x509Name);

				boolean result = ap.save(authenticator);
				if (result == true) {
					if (logger.isDebugEnabled()) {
						logger.debug("Authenticator with Email address " + email + " stored in DB.");
					}
					webServerWithClientAuth.restartWebServer();
				}

				return result;
			} else {
				throw new HandlerException("Required argument missing");
			}
		} catch (IOException | ProviderException e) {
			throw new HandlerException(e);
		}
	}
}
