package de.fraunhofer.sit.c2x.pki;

import java.io.IOException;
import java.util.ArrayList;

import org.apache.commons.codec.DecoderException;
import org.apache.commons.codec.binary.Hex;
import org.bouncycastle.crypto.AsymmetricCipherKeyPair;
import org.bouncycastle.crypto.params.ECPrivateKeyParameters;

import com.google.inject.Inject;

import de.fraunhofer.sit.c2x.pki.ca.certificates.CertificateBuilder;
import de.fraunhofer.sit.c2x.pki.ca.certificates.RootCertificateResult;
import de.fraunhofer.sit.c2x.pki.ca.certificates.datacontainers.GeographicRegionDataContainer;
import de.fraunhofer.sit.c2x.pki.ca.certificates.datacontainers.PsidSspPriorityDataContainer;
import de.fraunhofer.sit.c2x.pki.ca.core.exceptions.HandlerException;
import de.fraunhofer.sit.c2x.pki.ca.crypto.CryptoUtils;
import de.fraunhofer.sit.c2x.pki.ca.crypto.Digest;
import de.fraunhofer.sit.c2x.pki.ca.crypto.KeyGenerator;
import de.fraunhofer.sit.c2x.pki.ca.provider.ProviderException;
import de.fraunhofer.sit.c2x.pki.ca.provider.interfaces.Certificate;

public class CertificateGenerator {

	@Inject
	private CertificateBuilder certificateBuilder;

	private String certId;
	private String certificate;
	private AsymmetricCipherKeyPair signingKeyPair;
	private AsymmetricCipherKeyPair encryptionKeyPair;

	private void initLocalVariables() {
		certId = null;
		certificate = null;
		signingKeyPair = null;
		encryptionKeyPair = null;
	}

	public String createRootCert(String startDate, String endDate, String assuranceLevel, String subjectName,
			ArrayList<PsidSspPriorityDataContainer> permissions,
			ArrayList<GeographicRegionDataContainer> regionRestriction) throws HandlerException, IOException {
		initLocalVariables();

		// keys
		RootCertificateResult certResult = certificateBuilder.createNewRootCertificate(startDate, endDate,
				assuranceLevel, subjectName, permissions, regionRestriction);

		signingKeyPair = certResult.getSigningKeyPair();
		encryptionKeyPair = certResult.getEncryptionKeyPair();

		de.fraunhofer.sit.c2x.pki.ca.provider.interfaces.Certificate certContainer = certResult.getCert();

		// out
		certificate = certificateBuilder.getCertificateAsHexString(certContainer);
		System.out.println("Root Certificate: " + certificate);
		if (certContainer.getCertId() != null)
			certId = Hex.encodeHexString(certContainer.getCertId());
		else {
			byte[] digest = Digest.getInstance().sha256(certContainer.getCertificate());
			int length = 8;
			byte[] id8 = new byte[length];
			System.arraycopy(digest, digest.length - length, id8, 0, id8.length);
			certId = Hex.encodeHexString(id8);
		}
		return certificate;
	}

	public String createPCACert(String startDate, String endDate,
			ArrayList<PsidSspPriorityDataContainer> permissions, ECPrivateKeyParameters rootPrivateKey,
			byte[] rootCertificate, String assuranceLevel, String subjectName,
			ArrayList<GeographicRegionDataContainer> regionRestriction) throws HandlerException,
			ProviderException, DecoderException, IOException {
		initLocalVariables();

		// generate signing key pair
		signingKeyPair = KeyGenerator.getInstance().generateECKeyPair("P-256");
		System.out.println("PCA private signing key: "
				+ ((ECPrivateKeyParameters) signingKeyPair.getPrivate()).getD());
		String signingKeyX = Hex.encodeHexString(CryptoUtils.getPublicKeyX(signingKeyPair.getPublic(), 32));
		String signingKeyY = Hex.encodeHexString(CryptoUtils.getPublicKeyY(signingKeyPair.getPublic(), 32));

		// generate encryption key pair
		encryptionKeyPair = KeyGenerator.getInstance().generateECKeyPair("P-256");
		System.out.println("PCA private encrpytion key: "
				+ ((ECPrivateKeyParameters) encryptionKeyPair.getPrivate()).getD());
		String encryptionKeyX = Hex.encodeHexString(CryptoUtils.getPublicKeyX(encryptionKeyPair.getPublic(),
				32));
		String encryptionKeyY = Hex.encodeHexString(CryptoUtils.getPublicKeyX(encryptionKeyPair.getPublic(),
				32));

		String version = "2";
		String algorithm = "ECDSA_NISTP256_WITH_SHA256";
		String symmetricAlgorithm = "ECIES_NISTP256";
		String compression = "false";

		if (signingKeyX == null || signingKeyY == null || encryptionKeyX == null || encryptionKeyY == null
				|| startDate == null || endDate == null || signingKeyX.equals("") || signingKeyY.equals("")
				|| encryptionKeyX.equals("") || encryptionKeyY.equals("") || startDate.equals("")
				|| endDate.equals("")) {
			throw new HandlerException("Request failed! Please fill out all required fields!");
		}

		Certificate cert = certificateBuilder.createNewPseudonymCaCertificate(signingKeyX, signingKeyY,
				encryptionKeyX, encryptionKeyY, version, algorithm, symmetricAlgorithm, compression,
				startDate, endDate, permissions, rootPrivateKey, rootCertificate, true, assuranceLevel,
				subjectName, regionRestriction);
		System.out.println("AA Certificate: " + certificateBuilder.getCertificateAsHexString(cert));
		certificate = certificateBuilder.getCertificateAsHexString(cert);
		if (cert.getCertId() != null)
			certId = Hex.encodeHexString(cert.getCertId());
		else {
			byte[] digest = Digest.getInstance().sha256(cert.getCertificate());
			int length = 8;
			byte[] id8 = new byte[length];
			System.arraycopy(digest, digest.length - length, id8, 0, id8.length);
			certId = Hex.encodeHexString(id8);
		}
		return certificate;
	}

	public String createPseudonymCert(String startDate, String endDate,
			ArrayList<PsidSspPriorityDataContainer> permissions, ECPrivateKeyParameters pcaPrivateKey,
			byte[] pcaCertificate, String assuranceLevel,
			ArrayList<GeographicRegionDataContainer> regionRestriction) throws HandlerException,
			ProviderException, DecoderException, IOException {

		// generate signing key pair
		signingKeyPair = KeyGenerator.getInstance().generateECKeyPair("P-256");
		System.out.println("Pseudonym private signing key: "
				+ ((ECPrivateKeyParameters) signingKeyPair.getPrivate()).getD());
		byte[] signingKeyX = CryptoUtils.getPublicKeyX(signingKeyPair.getPublic(), 32);
		byte[] signingKeyY = CryptoUtils.getPublicKeyY(signingKeyPair.getPublic(), 32);

		// generate encryption key pair
		encryptionKeyPair = KeyGenerator.getInstance().generateECKeyPair("P-256");
		System.out.println("Pseudonym private encryption key: "
				+ ((ECPrivateKeyParameters) encryptionKeyPair.getPrivate()).getD());
		byte[] encryptionKeyX = CryptoUtils.getPublicKeyX(encryptionKeyPair.getPublic(), 32);
		byte[] encryptionKeyY = CryptoUtils.getPublicKeyY(encryptionKeyPair.getPublic(), 32);

		return createPseudonymCert(signingKeyX, signingKeyY, encryptionKeyX, encryptionKeyY, startDate,
				endDate, permissions, pcaPrivateKey, pcaCertificate, assuranceLevel, regionRestriction);
	}

	public String createPseudonymCert(byte[] signingKeyX, byte[] signingKeyY, byte[] encryptionKeyX,
			byte[] encryptionKeyY, String startDate, String endDate,
			ArrayList<PsidSspPriorityDataContainer> permissions, ECPrivateKeyParameters pcaPrivateKey,
			byte[] pcaCertificate, String assuranceLevel,
			ArrayList<GeographicRegionDataContainer> regionRestriction) throws HandlerException,
			ProviderException, DecoderException, IOException {
		initLocalVariables();

		if (signingKeyX == null || signingKeyX.length != 32) {
			throw new HandlerException("Request failed! Invalid public signing key X provided: "
					+ signingKeyX.toString());
		}
		if (signingKeyY == null || signingKeyY.length != 32) {
			throw new HandlerException("Request failed! Invalid public signing key Y provided: "
					+ signingKeyY.toString());
		}

		String encryptionKeyXHexEncoded = null;
		if (encryptionKeyX != null && encryptionKeyX.length != 32) {
			throw new HandlerException("Request failed! Invalid public encrpytion key X provided: "
					+ signingKeyX.toString());
		} else if (encryptionKeyX != null && encryptionKeyX.length == 32)
			encryptionKeyXHexEncoded = Hex.encodeHexString(encryptionKeyX);

		String encryptionKeyYHexEncoded = null;
		if (encryptionKeyY != null && encryptionKeyY.length != 32) {
			throw new HandlerException("Request failed! Invalid public encrpytion key Y provided: "
					+ signingKeyX.toString());
		} else if (encryptionKeyY != null && encryptionKeyY.length == 32)
			encryptionKeyYHexEncoded = Hex.encodeHexString(encryptionKeyY);

		String signingKeyXHexEncoded = Hex.encodeHexString(signingKeyX);
		String signingKeyYHexEncoded = Hex.encodeHexString(signingKeyY);

		String version = "2";
		String algorithm = "ECDSA_NISTP256_WITH_SHA256";
		String symmetricAlgorithm = "ECIES_NISTP256";
		String compression = "false";

		if (signingKeyX == null || signingKeyY == null || startDate == null || endDate == null
				|| signingKeyX.equals("") || signingKeyY.equals("") || startDate.equals("")
				|| endDate.equals("")) {
			throw new HandlerException("Request failed! Please fill out all required fields!");
		}

		Certificate cert = certificateBuilder.createNewPseudonymCertificate(signingKeyXHexEncoded,
				signingKeyYHexEncoded, encryptionKeyXHexEncoded, encryptionKeyYHexEncoded, version,
				algorithm, symmetricAlgorithm, compression, startDate, endDate, permissions, pcaPrivateKey,
				pcaCertificate, assuranceLevel, regionRestriction);
		System.out.println("AT Certificate: " + certificateBuilder.getCertificateAsHexString(cert));
		certificate = certificateBuilder.getCertificateAsHexString(cert);
		if (cert.getCertId() != null)
			certId = Hex.encodeHexString(cert.getCertId());
		else {
			byte[] digest = Digest.getInstance().sha256(cert.getCertificate());
			int length = 8;
			byte[] id8 = new byte[length];
			System.arraycopy(digest, digest.length - length, id8, 0, id8.length);
			certId = Hex.encodeHexString(id8);
		}
		return certificate;
	}

	public String getCertId() {
		return certId;
	}

	public String getCertificate() {
		return certificate;
	}

	public AsymmetricCipherKeyPair getEncryptionKeyPair() {
		return encryptionKeyPair;
	}

	public AsymmetricCipherKeyPair getSigningKeyPair() {
		return signingKeyPair;
	}
}
