package de.fraunhofer.sit.c2x.pki;

import java.io.IOException;
import java.security.Security;
import java.text.DateFormat;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Date;
import java.util.List;

import org.apache.commons.codec.DecoderException;
import org.apache.commons.codec.binary.Hex;
import org.bouncycastle.crypto.AsymmetricCipherKeyPair;
import org.bouncycastle.crypto.params.ECPrivateKeyParameters;
import org.bouncycastle.jce.provider.BouncyCastleProvider;

import com.google.inject.Guice;
import com.google.inject.Inject;
import com.google.inject.Injector;

import de.fraunhofer.sit.c2x.CertChain;
import de.fraunhofer.sit.c2x.RegionRestriction;
import de.fraunhofer.sit.c2x.pki.ca.certificates.datacontainers.GeographicRegionDataContainer;
import de.fraunhofer.sit.c2x.pki.ca.certificates.datacontainers.PsidSspPriorityDataContainer;
import de.fraunhofer.sit.c2x.pki.ca.core.exceptions.HandlerException;
import de.fraunhofer.sit.c2x.pki.ca.provider.ProviderException;

public class CertChainGenerator {
	private static CertChainGenerator instance;
	private static final DateFormat df = new SimpleDateFormat("dd.MM.yyyy");
	private static final String today = df.format(new Date());

	@Inject
	CertificateGenerator caCertificateRequestHandler;

	public static CertChainGenerator getInstance() {
		if (instance == null)
			instance = new CertChainGenerator();
		return instance;
	}

	public CertChainGenerator() {
		Security.addProvider(new BouncyCastleProvider());
		Injector injector = Guice.createInjector(new CertificateBuilderModule());
		caCertificateRequestHandler = injector.getInstance(CertificateGenerator.class);
	}

	/**
	 * @param endDate
	 *            as string with the format: dd.mm.yyyy
	 * @param assuranceLevel
	 *            as value: 0,1,2,3,4,5,6,7
	 * @param aidList
	 *            as array of long values giving the Application IDs according
	 *            to ETSI TS 103 097 and ETSI TR 102 965
	 * @param sspList
	 *            as array of byte arrays giving the Service Specific
	 *            Permissions to every AID according to ETSI TS 103 097
	 * @param List
	 *            of PseudonymSigningKeysX of NIST p256 ECC public key
	 * @param List
	 *            of PseudonymSigningKeysY of NIST p256 ECC public key
	 * @param numberOfAAs
	 *            that should be generated
	 * @param numberOfATsPerAA
	 *            that should be generated
	 * @return CertChain structure that contains the HEX encoded ETSI TS 103 097
	 *         certificates. First certificate is the root certificate, second
	 *         is the Authorization Authority certificate and the third is the
	 *         Authorization Ticket certificate
	 * @throws HandlerException
	 */
	public static CertChain getCertChain(String endDate, int assuranceLevel, long[] aidList,
			byte[][] sspList, List<byte[]> pseudonymSigningKeysX, List<byte[]> pseudonymSigningKeysY,
			int numberOfAAs, int numberOfATsPerAA) throws HandlerException {
		return getInstance()
				._getCertChain(today, endDate, assuranceLevel, aidList, sspList, null, null, null,
						pseudonymSigningKeysX, pseudonymSigningKeysY, null, null, numberOfAAs,
						numberOfATsPerAA);
	}

	/**
	 * @param endDate
	 *            as string with the format: dd.mm.yyyy
	 * @param assuranceLevel
	 *            as value: 0,1,2,3,4,5,6,7
	 * @param aidList
	 *            as array of long values giving the Application IDs according
	 *            to ETSI TS 103 097 and ETSI TR 102 965
	 * @param sspList
	 *            as array of byte arrays giving the Service Specific
	 *            Permissions to every AID according to ETSI TS 103 097
	 * @param List
	 *            of PseudonymSigningKeyX of NIST p256 ECC public key
	 * @param List
	 *            of PseudonymSigningKeyY of NIST p256 ECC public key
	 * @param List
	 *            of PseudonymEncryptionKeyX of NIST p256 ECC public key
	 * @param List
	 *            of PseudonymEncryptionKeyY of NIST p256 ECC public key
	 * @param numberOfAAs
	 *            that should be generated
	 * @param numberOfATsPerAA
	 *            that should be generated
	 * @return CertChain structure that contains the HEX encoded ETSI TS 103 097
	 *         certificates. First certificate is the root certificate, second
	 *         is the Authorization Authority certificate and the third is the
	 *         Authorization Ticket certificate
	 * @throws HandlerException
	 */
	public static CertChain getCertChain(String endDate, int assuranceLevel, long[] aidList,
			byte[][] sspList, List<byte[]> pseudonymSigningKeysX, List<byte[]> pseudonymSigningKeysY,
			List<byte[]> pseudonymEncryptionKeysX, List<byte[]> pseudonymEncryptionKeysY, int numberOfAAs,
			int numberOfATsPerAA) throws HandlerException {
		return getInstance()._getCertChain(today, endDate, assuranceLevel, aidList, sspList, null, null,
				null, pseudonymSigningKeysX, pseudonymSigningKeysY, pseudonymEncryptionKeysX,
				pseudonymEncryptionKeysY, numberOfAAs, numberOfATsPerAA);
	}

	/**
	 * @param endDate
	 *            as string with the format: dd.mm.yyyy
	 * @param assuranceLevel
	 *            as value: 0,1,2,3,4,5,6,7
	 * @param aidList
	 *            as array of long values giving the Application IDs according
	 *            to ETSI TS 103 097 and ETSI TR 102 965
	 * @param sspList
	 *            as array of byte arrays giving the Service Specific
	 *            Permissions to every AID according to ETSI TS 103 097
	 * @param circularRegionRestrictionCenterLatitude
	 *            as WGS 84 encoded latitude value
	 * @param circularRegionRestrictionCenterLongitude
	 *            as WGS 84 encoded longitude value
	 * @param circularRegionRestrictionRadius
	 *            in meters
	 * @param rootHasRegionRestriction
	 *            if true then the root cert contains region restriction, if
	 *            false then root has region restriction = none
	 * @param List
	 *            of PseudonymSigningKeyX of NIST p256 ECC public key
	 * @param List
	 *            of PseudonymSigningKeyY of NIST p256 ECC public key
	 * @param List
	 *            of PseudonymEncryptionKeyX of NIST p256 ECC public key
	 * @param List
	 *            of PseudonymEncryptionKeyY of NIST p256 ECC public key
	 * @param numberOfAAs
	 *            that should be generated
	 * @param numberOfATsPerAA
	 *            that should be generated
	 * @return CertChain structure that contains the HEX encoded ETSI TS 103 097
	 *         certificates. First certificate is the root certificate, second
	 *         is the Authorization Authority certificate and the third is the
	 *         Authorization Ticket certificate
	 * @throws HandlerException
	 */
	public static CertChain getCertChainWithCircularRegionRestriction(String endDate, int assuranceLevel,
			long[] aidList, byte[][] sspList, double circularRegionRestrictionCenterLatitude,
			double circularRegionRestrictionCenterLongitude, int circularRegionRestrictionRadius,
			boolean rootHasRegionRestriction, List<byte[]> pseudonymSigningKeysX,
			List<byte[]> pseudonymSigningKeysY, List<byte[]> pseudonymEncryptionKeysX,
			List<byte[]> pseudonymEncryptionKeysY, int numberOfAAs, int numberOfATsPerAA)
			throws HandlerException {

		RegionRestriction regionRestriction = new RegionRestriction(circularRegionRestrictionCenterLatitude,
				circularRegionRestrictionCenterLongitude, circularRegionRestrictionRadius);
		List<RegionRestriction> aaRegionRestrictions = new ArrayList<>(numberOfAAs);
		for (int i = 0; i < numberOfAAs; i++)
			aaRegionRestrictions.add(regionRestriction);
		List<RegionRestriction> atRegionRestrictions = new ArrayList<>(numberOfAAs);
		for (int i = 0; i < numberOfAAs; i++)
			atRegionRestrictions.add(regionRestriction);

		if (rootHasRegionRestriction)
			return getInstance()._getCertChain(today, endDate, assuranceLevel, aidList, sspList,
					regionRestriction, aaRegionRestrictions, atRegionRestrictions, pseudonymSigningKeysX,
					pseudonymSigningKeysY, pseudonymEncryptionKeysX, pseudonymEncryptionKeysY, numberOfAAs,
					numberOfATsPerAA);
		else
			return getInstance()._getCertChain(today, endDate, assuranceLevel, aidList, sspList, null,
					aaRegionRestrictions, atRegionRestrictions, pseudonymSigningKeysX, pseudonymSigningKeysY,
					pseudonymEncryptionKeysX, pseudonymEncryptionKeysY, numberOfAAs, numberOfATsPerAA);
	}

	/**
	 * @param endDate
	 *            as string with the format: dd.mm.yyyy
	 * @param assuranceLevel
	 *            as value: 0,1,2,3,4,5,6,7
	 * @param aidList
	 *            as array of long values giving the Application IDs according
	 *            to ETSI TS 103 097 and ETSI TR 102 965
	 * @param sspList
	 *            as array of byte arrays giving the Service Specific
	 *            Permissions to every AID according to ETSI TS 103 097
	 * @param rectangularRegionRestrictionNorthWestPointsLatitude
	 *            as WGS 84 encoded latitude value
	 * @param rectangularRegionRestrictionNorthWestPointsLongitude
	 *            as WGS 84 encoded longitude value
	 * @param rectangularRegionRestrictionSouthEastPointsLatitude
	 *            as WGS 84 encoded latitude value
	 * @param rectangularRegionRestrictionSouthEastPointsLongitude
	 *            as WGS 84 encoded longitude value
	 * @param rootHasRegionRestriction
	 *            if true then the root cert contains region restriction, if
	 *            false then root has region restriction = none
	 * @param List
	 *            of PseudonymSigningKeyX of NIST p256 ECC public key
	 * @param List
	 *            of PseudonymSigningKeyY of NIST p256 ECC public key
	 * @param List
	 *            of PseudonymEncryptionKeyX of NIST p256 ECC public key
	 * @param List
	 *            of PseudonymEncryptionKeyY of NIST p256 ECC public key
	 * @param numberOfAAs
	 *            that should be generated
	 * @param numberOfATsPerAA
	 *            that should be generated
	 * @return CertChain structure that contains the HEX encoded ETSI TS 103 097
	 *         certificates. First certificate is the root certificate, second
	 *         is the Authorization Authority certificate and the third is the
	 *         Authorization Ticket certificate
	 * @throws HandlerException
	 */
	public static CertChain getCertChainWithRectangularRegionRestriction(String endDate, int assuranceLevel,
			long[] aidList, byte[][] sspList, Double[] rectangularRegionRestrictionNorthWestPointsLatitude,
			Double[] rectangularRegionRestrictionNorthWestPointsLongitude,
			Double[] rectangularRegionRestrictionSouthEastPointsLatitude,
			Double[] rectangularRegionRestrictionSouthEastPointsLongitude, boolean rootHasRegionRestriction,
			List<byte[]> pseudonymSigningKeysX, List<byte[]> pseudonymSigningKeysY,
			List<byte[]> pseudonymEncryptionKeysX, List<byte[]> pseudonymEncryptionKeysY, int numberOfAAs,
			int numberOfATsPerAA) throws HandlerException {

		RegionRestriction regionRestriction = new RegionRestriction(
				rectangularRegionRestrictionNorthWestPointsLatitude,
				rectangularRegionRestrictionNorthWestPointsLongitude,
				rectangularRegionRestrictionSouthEastPointsLatitude,
				rectangularRegionRestrictionSouthEastPointsLongitude);
		List<RegionRestriction> aaRegionRestrictions = new ArrayList<>(numberOfAAs);
		for (int i = 0; i < numberOfAAs; i++)
			aaRegionRestrictions.add(regionRestriction);
		List<RegionRestriction> atRegionRestrictions = new ArrayList<>(numberOfAAs);
		for (int i = 0; i < numberOfAAs; i++)
			atRegionRestrictions.add(regionRestriction);

		if (rootHasRegionRestriction)
			return getInstance()._getCertChain(today, endDate, assuranceLevel, aidList, sspList,
					regionRestriction, aaRegionRestrictions, atRegionRestrictions, pseudonymSigningKeysX,
					pseudonymSigningKeysY, pseudonymEncryptionKeysX, pseudonymEncryptionKeysY, numberOfAAs,
					numberOfATsPerAA);
		else
			return getInstance()._getCertChain(today, endDate, assuranceLevel, aidList, sspList, null,
					aaRegionRestrictions, atRegionRestrictions, pseudonymSigningKeysX, pseudonymSigningKeysY,
					pseudonymEncryptionKeysX, pseudonymEncryptionKeysY, numberOfAAs, numberOfATsPerAA);
	}

	/**
	 * @param endDate
	 *            as string with the format: dd.mm.yyyy
	 * @param assuranceLevel
	 *            as value: 0,1,2,3,4,5,6,7
	 * @param aidList
	 *            as array of long values giving the Application IDs according
	 *            to ETSI TS 103 097 and ETSI TR 102 965
	 * @param sspList
	 *            as array of byte arrays giving the Service Specific
	 *            Permissions to every AID according to ETSI TS 103 097
	 * @param polygonalRegionRestrictionPointsLatitude
	 *            as WGS 84 encoded latitude value
	 * @param polygonalRegionRestrictionPointsLongitude
	 *            as WGS 84 encoded longitude value
	 * @param rootHasRegionRestriction
	 *            if true then the root cert contains region restriction, if
	 *            false then root has region restriction = none
	 * @param List
	 *            of PseudonymSigningKeyX of NIST p256 ECC public key
	 * @param List
	 *            of PseudonymSigningKeyY of NIST p256 ECC public key
	 * @param List
	 *            of PseudonymEncryptionKeyX of NIST p256 ECC public key
	 * @param List
	 *            of PseudonymEncryptionKeyY of NIST p256 ECC public key
	 * @param numberOfAAs
	 *            that should be generated
	 * @param numberOfATsPerAA
	 *            that should be generated
	 * @return CertChain structure that contains the HEX encoded ETSI TS 103 097
	 *         certificates. First certificate is the root certificate, second
	 *         is the Authorization Authority certificate and the third is the
	 *         Authorization Ticket certificate
	 * @throws HandlerException
	 */
	public static CertChain getCertChainWithPolygonalRegionRestriction(String endDate, int assuranceLevel,
			long[] aidList, byte[][] sspList, Double[] polygonalRegionRestrictionPointsLatitude,
			Double[] polygonalRegionRestrictionPointsLongitude, boolean rootHasRegionRestriction,
			List<byte[]> pseudonymSigningKeysX, List<byte[]> pseudonymSigningKeysY,
			List<byte[]> pseudonymEncryptionKeysX, List<byte[]> pseudonymEncryptionKeysY, int numberOfAAs,
			int numberOfATsPerAA) throws HandlerException {

		RegionRestriction regionRestriction = new RegionRestriction(polygonalRegionRestrictionPointsLatitude,
				polygonalRegionRestrictionPointsLongitude);
		List<RegionRestriction> aaRegionRestrictions = new ArrayList<>(numberOfAAs);
		for (int i = 0; i < numberOfAAs; i++)
			aaRegionRestrictions.add(regionRestriction);
		List<RegionRestriction> atRegionRestrictions = new ArrayList<>(numberOfAAs);
		for (int i = 0; i < numberOfAAs; i++)
			atRegionRestrictions.add(regionRestriction);

		if (rootHasRegionRestriction)
			return getInstance()._getCertChain(today, endDate, assuranceLevel, aidList, sspList,
					regionRestriction, aaRegionRestrictions, atRegionRestrictions, pseudonymSigningKeysX,
					pseudonymSigningKeysY, pseudonymEncryptionKeysX, pseudonymEncryptionKeysY, numberOfAAs,
					numberOfATsPerAA);
		else
			return getInstance()._getCertChain(today, endDate, assuranceLevel, aidList, sspList, null,
					aaRegionRestrictions, atRegionRestrictions, pseudonymSigningKeysX, pseudonymSigningKeysY,
					pseudonymEncryptionKeysX, pseudonymEncryptionKeysY, numberOfAAs, numberOfATsPerAA);
	}

	/**
	 * @param endDate
	 *            as string with the format: dd.mm.yyyy
	 * @param assuranceLevel
	 *            as value: 0,1,2,3,4,5,6,7
	 * @param aidList
	 *            as array of long values giving the Application IDs according
	 *            to ETSI TS 103 097 and ETSI TR 102 965
	 * @param sspList
	 *            as array of byte arrays giving the Service Specific
	 *            Permissions to every AID according to ETSI TS 103 097
	 * @param identifiedRegionRestrictionDictionary
	 * @param identifiedRegionRestrictionIdentifer
	 * @param identifiedRegionRestrictionLocalRegion
	 * @param rootHasRegionRestriction
	 *            if true then the root cert contains region restriction, if
	 *            false then root has region restriction = none
	 * @param List
	 *            of PseudonymSigningKeyX of NIST p256 ECC public key
	 * @param List
	 *            of PseudonymSigningKeyY of NIST p256 ECC public key
	 * @param List
	 *            of PseudonymEncryptionKeyX of NIST p256 ECC public key
	 * @param List
	 *            of PseudonymEncryptionKeyY of NIST p256 ECC public key
	 * @param numberOfAAs
	 *            that should be generated
	 * @param numberOfATsPerAA
	 *            that should be generated
	 * @return CertChain structure that contains the HEX encoded ETSI TS 103 097
	 *         certificates. First certificate is the root certificate, second
	 *         is the Authorization Authority certificate and the third is the
	 *         Authorization Ticket certificate
	 * @throws HandlerException
	 */
	public static CertChain getCertChainWithIdentifiedRegionRestriction(String endDate, int assuranceLevel,
			long[] aidList, byte[][] sspList, String identifiedRegionRestrictionDictionary,
			long identifiedRegionRestrictionIdentifer, long identifiedRegionRestrictionLocalRegion,
			boolean rootHasRegionRestriction, List<byte[]> pseudonymSigningKeysX,
			List<byte[]> pseudonymSigningKeysY, List<byte[]> pseudonymEncryptionKeysX,
			List<byte[]> pseudonymEncryptionKeysY, int numberOfAAs, int numberOfATsPerAA)
			throws HandlerException {

		RegionRestriction regionRestriction = new RegionRestriction(identifiedRegionRestrictionDictionary,
				identifiedRegionRestrictionIdentifer, identifiedRegionRestrictionLocalRegion);
		List<RegionRestriction> aaRegionRestrictions = new ArrayList<>(numberOfAAs);
		for (int i = 0; i < numberOfAAs; i++)
			aaRegionRestrictions.add(regionRestriction);
		List<RegionRestriction> atRegionRestrictions = new ArrayList<>(numberOfAAs);
		for (int i = 0; i < numberOfAAs; i++)
			atRegionRestrictions.add(regionRestriction);

		if (rootHasRegionRestriction)
			return getInstance()._getCertChain(today, endDate, assuranceLevel, aidList, sspList,
					regionRestriction, aaRegionRestrictions, atRegionRestrictions, pseudonymSigningKeysX,
					pseudonymSigningKeysY, pseudonymEncryptionKeysX, pseudonymEncryptionKeysY, numberOfAAs,
					numberOfATsPerAA);
		else
			return getInstance()._getCertChain(today, endDate, assuranceLevel, aidList, sspList, null,
					aaRegionRestrictions, atRegionRestrictions, pseudonymSigningKeysX, pseudonymSigningKeysY,
					pseudonymEncryptionKeysX, pseudonymEncryptionKeysY, numberOfAAs, numberOfATsPerAA);
	}
	
	public static CertChain getCertChain(String endDate, int assuranceLevel, long[] aidList,
			byte[][] sspList, RegionRestriction rootRegionRestriction,
			List<RegionRestriction> aaRegionRestrictions, List<RegionRestriction> atRegionRestrictions,
			List<byte[]> pseudonymSigningKeysX, List<byte[]> pseudonymSigningKeysY,
			List<byte[]> pseudonymEncryptionKeysX, List<byte[]> pseudonymEncryptionKeysY, int numberOfAAs,
			int numberOfATsPerAA) throws HandlerException {
		return getInstance()._getCertChain(today, endDate, assuranceLevel, aidList, sspList, rootRegionRestriction,
				aaRegionRestrictions, atRegionRestrictions, pseudonymSigningKeysX, pseudonymSigningKeysY,
				pseudonymEncryptionKeysX, pseudonymEncryptionKeysY, numberOfAAs, numberOfATsPerAA);
	}

	private CertChain _getCertChain(String startDate, String endDate, int assuranceLevel, long[] aidList,
			byte[][] sspList, RegionRestriction rootRegionRestriction,
			List<RegionRestriction> pcaRegionRestriction, List<RegionRestriction> pcRegionRestriction,
			List<byte[]> pseudonymSigningKeysX, List<byte[]> pseudonymSigningKeysY,
			List<byte[]> pseudonymEncryptionKeysX, List<byte[]> pseudonymEncryptionKeysY, int numberOfPCAs,
			int numberOfPCsPerPCA) throws HandlerException {

		if (pcaRegionRestriction != null && numberOfPCAs != pcaRegionRestriction.size())
			throw new HandlerException(
					"Number of requested AA certificates != size of AA region restriction list");

		if (pcRegionRestriction != null && numberOfPCsPerPCA != pcRegionRestriction.size())
			throw new HandlerException(
					"Number of requested pseudonym certificates != size of pseudonym certificate region restriction list");

		if (pseudonymSigningKeysX.size() != pseudonymSigningKeysY.size())
			throw new HandlerException(
					"Number of pseudonymSigningKeysX and pseudonymSigningKeysY must be equal!");

		if (pseudonymEncryptionKeysX != null
				&& pseudonymEncryptionKeysX.size() != pseudonymEncryptionKeysY.size())
			throw new HandlerException(
					"Number of pseudonymEncryptionKeysX and pseudonymEncryptionKeysY must be equal!");

		CertChain certChain = new CertChain();
		try {
			// copy permissions (only AIDs for CA certificates) into correct
			// format
			ArrayList<PsidSspPriorityDataContainer> permissionsCA = new ArrayList<>();
			for (int i = 0; i < aidList.length; i++) {
				PsidSspPriorityDataContainer c = new PsidSspPriorityDataContainer();
				c.setPsid(new Long(aidList[i]));
				permissionsCA.add(c);
			}

			// ArrayList<GeographicRegionDataContainer> regionRestrictions = new
			// ArrayList<>();
			// if (circularRegionRestrictionCenterLatitude != null
			// && circularRegionRestrictionCenterLongitude != null
			// && circularRegionRestrictionRadius != null) {
			// // circular region restriction
			// GeographicRegionDataContainer r = new
			// GeographicRegionDataContainer(
			// circularRegionRestrictionCenterLatitude,
			// circularRegionRestrictionCenterLongitude,
			// circularRegionRestrictionRadius);
			// regionRestrictions.add(r);
			// } else if (rectangularRegionRestrictionNorthWestPointsLatitude !=
			// null
			// && rectangularRegionRestrictionNorthWestPointsLongitude != null
			// && rectangularRegionRestrictionSouthEastPointsLatitude != null
			// && rectangularRegionRestrictionSouthEastPointsLongitude != null)
			// {
			// // rectangular region restriction
			// GeographicRegionDataContainer r = new
			// GeographicRegionDataContainer(
			// rectangularRegionRestrictionNorthWestPointsLatitude,
			// rectangularRegionRestrictionNorthWestPointsLongitude,
			// rectangularRegionRestrictionSouthEastPointsLatitude,
			// rectangularRegionRestrictionSouthEastPointsLongitude);
			// regionRestrictions.add(r);
			// } else if (polygonalRegionRestrictionPointsLatitude != null
			// && polygonalRegionRestrictionPointsLongitude != null) {
			// // polygonal region restriction
			// GeographicRegionDataContainer r = new
			// GeographicRegionDataContainer(
			// polygonalRegionRestrictionPointsLatitude,
			// polygonalRegionRestrictionPointsLongitude);
			// regionRestrictions.add(r);
			// } else if (identifiedRegionRestrictionDictionary != null
			// && identifiedRegionRestrictionIdentifer != null) {
			// // identified region restriction
			// GeographicRegionDataContainer r = new
			// GeographicRegionDataContainer(
			// identifiedRegionRestrictionDictionary,
			// identifiedRegionRestrictionIdentifer,
			// identifiedRegionRestrictionLocalRegion);
			// regionRestrictions.add(r);
			// }

			// create RCA certificate
			String rcaCert = null;
			ArrayList<GeographicRegionDataContainer> regionRestrictions = null;
			if (rootRegionRestriction != null
					&& rootRegionRestriction.getGeographicRegionDataContainer() != null) {
				regionRestrictions = new ArrayList<>(1);
				regionRestrictions.add(rootRegionRestriction.getGeographicRegionDataContainer());
			}
			rcaCert = caCertificateRequestHandler.createRootCert(startDate, endDate, new Integer(
					assuranceLevel).toString(), "ETSI_Plugtest_Root", permissionsCA, regionRestrictions);

			certChain.setRootCert(rcaCert);
			AsymmetricCipherKeyPair rcaSigningKeyPair = caCertificateRequestHandler.getSigningKeyPair();

			for (int pcaI = 0; pcaI < numberOfPCAs; pcaI++) {
				// get region restrictions
				if (pcaRegionRestriction != null && pcaRegionRestriction.size() > pcaI
						&& pcaRegionRestriction.get(pcaI).getGeographicRegionDataContainer() != null) {
					regionRestrictions = new ArrayList<>(1);
					regionRestrictions.add(pcaRegionRestriction.get(pcaI).getGeographicRegionDataContainer());
				}
				// create PCA certificate
				String pcaCert = caCertificateRequestHandler.createPCACert(startDate, endDate, permissionsCA,
						(ECPrivateKeyParameters) rcaSigningKeyPair.getPrivate(),
						Hex.decodeHex(rcaCert.toCharArray()), new Integer(assuranceLevel).toString(),
						"ETSI_Plugtest_AA", regionRestrictions);
				AsymmetricCipherKeyPair pcaSigningKeyPair = caCertificateRequestHandler.getSigningKeyPair();
				byte[] pcaCertBytes = Hex.decodeHex(pcaCert.toCharArray());

				// copy permissions into correct format for pseudonym certs
				ArrayList<PsidSspPriorityDataContainer> permissionsPC = new ArrayList<>();
				for (int i = 0; i < aidList.length; i++) {
					PsidSspPriorityDataContainer c = new PsidSspPriorityDataContainer();
					c.setPsid(new Long(aidList[i]));
					if (sspList.length > i) {
						c.setServiceSpecificPermissions(sspList[i]);
					}
					permissionsPC.add(c);
				}

				List<String> pcCerts = new ArrayList<>(numberOfPCsPerPCA);
				for (int pcI = 0; pcI < numberOfPCsPerPCA; pcI++) {
					// get region restrictions
					if (pcRegionRestriction != null && pcRegionRestriction.size() > pcaI
							&& pcRegionRestriction.get(pcI).getGeographicRegionDataContainer() != null) {
						regionRestrictions = new ArrayList<>(1);
						regionRestrictions.add(pcRegionRestriction.get(pcI)
								.getGeographicRegionDataContainer());
					}
					// create pseudonym certificate
					byte[] pseudonymSigningKeyX = null;
					if (pseudonymSigningKeysX != null && pseudonymSigningKeysX.size() > pcI)
						pseudonymSigningKeyX = pseudonymSigningKeysX.get(pcI);
					byte[] pseudonymSigningKeyY = null;
					if (pseudonymSigningKeysY != null && pseudonymSigningKeysY.size() > pcI)
						pseudonymSigningKeyY = pseudonymSigningKeysY.get(pcI);
					byte[] pseudonymEncryptionKeyX = null;
					if (pseudonymEncryptionKeysX != null && pseudonymEncryptionKeysX.size() > pcI)
						pseudonymEncryptionKeyX = pseudonymEncryptionKeysX.get(pcI);
					byte[] pseudonymEncryptionKeyY = null;
					if (pseudonymEncryptionKeysY != null && pseudonymEncryptionKeysY.size() > pcI)
						pseudonymEncryptionKeyY = pseudonymEncryptionKeysY.get(pcI);

					String pcCert = caCertificateRequestHandler.createPseudonymCert(pseudonymSigningKeyX,
							pseudonymSigningKeyY, pseudonymEncryptionKeyX, pseudonymEncryptionKeyY,
							startDate, endDate, permissionsPC,
							(ECPrivateKeyParameters) pcaSigningKeyPair.getPrivate(), pcaCertBytes,
							new Integer(assuranceLevel).toString(), regionRestrictions);
					pcCerts.add(pcCert);
				}
				certChain.putAuthorizationTicketCerts(pcaCert, pcCerts);
			}

		} catch (HandlerException | IOException | ProviderException | DecoderException e) {
			e.printStackTrace();
			throw new HandlerException(e);
		}
		return certChain;
	}
}
