#include "LibItsSecurity_Functions.hh"

#include "sha256.hh"
#include "sha384.hh"
#include "ec_keys.hh"

#include "security_services.hh"

#include <openssl/ec.h>
#include <openssl/ecdsa.h>

#include "loggers.hh"

namespace LibItsSecurity__Functions 
{

  // FIXME Unify code with security_services
  
  /**
   * @desc    Produces a 256-bit (32-byte) hash value
   * @param   p_toBeHashedData Data to be used to calculate the hash value
   * @return  The hash value
   fx_hashWithSha256(in octetstring p_toBeHashedData) return Oct32;
  */
  OCTETSTRING fx__hashWithSha256(
                                 const OCTETSTRING& p__toBeHashedData
                                 ) {
    sha256 hash;
    std::vector<unsigned char> tbh(static_cast<const unsigned char *>(p__toBeHashedData), p__toBeHashedData.lengthof() + static_cast<const unsigned char *>(p__toBeHashedData));
    std::vector<unsigned char> hashData;
    hash.generate(tbh, hashData);
    return OCTETSTRING(hashData.size(), hashData.data());
  } // End of function fx__hashWithSha256

  /**
   * @desc    Produces a 384-bit (48-byte) hash value
   * @param   p_toBeHashedData Data to be used to calculate the hash value
   * @return  The hash value
   fx_hashWithSha384(in octetstring p_toBeHashedData) return Oct48;
  */
  OCTETSTRING fx__hashWithSha384(
                                 const OCTETSTRING& p__toBeHashedData
                                 ) {
    sha384 hash;
    std::vector<unsigned char> tbh(static_cast<const unsigned char *>(p__toBeHashedData), p__toBeHashedData.lengthof() + static_cast<const unsigned char *>(p__toBeHashedData));
    std::vector<unsigned char> hashData;
    hash.generate(tbh, hashData);
    return OCTETSTRING(hashData.size(), hashData.data());
  } // End of function fx__hashWithSha384

  /**
   * @desc    Produces a Elliptic Curve Digital Signature Algorithm (ECDSA) signaturee
   * @param   p_toBeSignedSecuredMessage    The data to be signed
   * @param   p_privateKey        The private key
   * @return  The signature value
   fx_signWithEcdsaNistp256WithSha256(in octetstring p_toBeSignedSecuredMessage, in octetstring<UInt64> p_privateKey) return octetstring;
  */
  OCTETSTRING fx__signWithEcdsaNistp256WithSha256(
                                                  const OCTETSTRING& p__toBeSignedSecuredMessage,
                                                  const OCTETSTRING& p__privateKey
                                                  ) {
    // Calculate the SHA256 of the data
    sha256 hash;
    std::vector<unsigned char> hashData;
    // TODO Create SHX interface and add generate method with std::vector
    std::vector<unsigned char> tbs(static_cast<const unsigned char *>(p__toBeSignedSecuredMessage), p__toBeSignedSecuredMessage.lengthof() + static_cast<const unsigned char *>(p__toBeSignedSecuredMessage));
    hash.generate(tbs, hashData);
    // Calculate the signature
    std::vector<unsigned char> p_key(static_cast<const unsigned char *>(p__privateKey), static_cast<const unsigned char *>(p__privateKey) + p__privateKey.lengthof());
    ec_keys k(ec_elliptic_curves::nist_p_256, p_key);
    std::vector<unsigned char> r_sig;
    std::vector<unsigned char> s_sig;
    if (k.sign(hashData, r_sig, s_sig) == 0) {
      OCTETSTRING os(r_sig.size(), r_sig.data());
      // loggers::get_instance().log_to_hexa("r_sig= ", os); 
      // loggers::get_instance().log_to_hexa("s_sig= ", OCTETSTRING(s_sig.size(), s_sig.data()); 
      os += OCTETSTRING(s_sig.size(), s_sig.data());
      // loggers::get_instance().log_to_hexa("sig= ", os); 
      return os;
    }

    return OCTETSTRING();
  }

  /**
   * @desc    Produces a Elliptic Curve Digital Signature Algorithm (ECDSA) signaturee
   * @param   p_toBeSignedSecuredMessage    The data to be signed
   * @param   p_privateKey        The private key
   * @return  The signature value
   fx_signWithEcdsaBrainpoolp256WithSha256(in octetstring p_toBeSignedSecuredMessage, in octetstring<UInt64> p_privateKey) return octetstring;
  */
  OCTETSTRING fx__signWithEcdsaBrainpoolp256WithSha256(
                                                       const OCTETSTRING& p__toBeSignedSecuredMessage,
                                                       const OCTETSTRING& p__privateKey
                                                       ) {
    // Calculate the SHA256 of the data
    sha256 hash;
    std::vector<unsigned char> hashData;
    // TODO Create SHX interface and add generate method with std::vector
    std::vector<unsigned char> tbs(static_cast<const unsigned char *>(p__toBeSignedSecuredMessage), p__toBeSignedSecuredMessage.lengthof() + static_cast<const unsigned char *>(p__toBeSignedSecuredMessage));
    hash.generate(tbs, hashData);
    // Calculate the signature
    std::vector<unsigned char> p_key(static_cast<const unsigned char *>(p__privateKey), static_cast<const unsigned char *>(p__privateKey) + p__privateKey.lengthof());
    ec_keys k(ec_elliptic_curves::brainpool_p_256_r1, p_key);
    std::vector<unsigned char> r_sig;
    std::vector<unsigned char> s_sig;
    if (k.sign(hashData, r_sig, s_sig) == 0) {
      OCTETSTRING os(r_sig.size(), r_sig.data());
      // loggers::get_instance().log_to_hexa("r_sig= ", os); 
      // loggers::get_instance().log_to_hexa("s_sig= ", OCTETSTRING(s_sig.size(), s_sig.data()); 
      os += OCTETSTRING(s_sig.size(), s_sig.data());
      // loggers::get_instance().log_to_hexa("sig= ", os); 
      return os;
    }

    return OCTETSTRING();
  }

  /**
   * @desc    Produces a Elliptic Curve Digital Signature Algorithm (ECDSA) signaturee
   * @param   p_toBeSignedSecuredMessage    The data to be signed
   * @param   p_privateKey        The private key
   * @return  The signature value
   fx_signWithEcdsaBrainpoolp384WithSha384(in octetstring p_toBeSignedSecuredMessage, in octetstring<UInt64> p_privateKey) return octetstring;
  */
  OCTETSTRING fx__signWithEcdsaBrainpoolp384WithSha384(
                                                       const OCTETSTRING& p__toBeSignedSecuredMessage,
                                                       const OCTETSTRING& p__privateKey
                                                       ) {
    // Calculate the SHA384 of the data
    sha384 hash;
    std::vector<unsigned char> hashData;
    // TODO Create SHX interface and add generate method with std::vector
    std::vector<unsigned char> tbs(static_cast<const unsigned char *>(p__toBeSignedSecuredMessage), p__toBeSignedSecuredMessage.lengthof() + static_cast<const unsigned char *>(p__toBeSignedSecuredMessage));
    hash.generate(tbs, hashData);
    // Calculate the signature
    std::vector<unsigned char> p_key(static_cast<const unsigned char *>(p__privateKey), static_cast<const unsigned char *>(p__privateKey) + p__privateKey.lengthof());
    ec_keys k(ec_elliptic_curves::brainpool_p_384_r1, p_key);
    std::vector<unsigned char> r_sig;
    std::vector<unsigned char> s_sig;
    if (k.sign(hashData, r_sig, s_sig) == 0) {
      OCTETSTRING os(r_sig.size(), r_sig.data());
      //loggers::get_instance().log_to_hexa("fx__signWithEcdsaBrainpoolp384WithSha384: r_sig= ", os); 
      //loggers::get_instance().log_to_hexa("fx__signWithEcdsaBrainpoolp384WithSha384: s_sig= ", OCTETSTRING(s_sig.size(), s_sig.data())); 
      os += OCTETSTRING(s_sig.size(), s_sig.data());
      //loggers::get_instance().log_to_hexa("fx__signWithEcdsaBrainpoolp384WithSha384: sig= ", os); 
      return os;
    }

    return OCTETSTRING();
  }

  /**
   * @desc    Verify the signature of the specified data
   * @param   p_toBeVerifiedData          The data to be verified
   * @param   p_signature                 The signature
   * @param   p_ecdsaNistp256PublicKeyX   The public key (x coordinate)
   * @param   p_ecdsaNistp256PublicKeyY   The public key (y coordinate)
   * @return  true on success, false otherwise
   fx_verifyWithEcdsaNistp256WithSha256(in octetstring p_toBeVerifiedData, in octetstring p_signature, in octetstring p_ecdsaNistp256PublicKeyX, in octetstring p_ecdsaNistp256PublicKeyY) return boolean;
  */
  BOOLEAN fx__verifyWithEcdsaNistp256WithSha256(
                                                const OCTETSTRING& p__toBeVerifiedData,
                                                const OCTETSTRING& p__signature,
                                                const OCTETSTRING& p__ecdsaNistp256PublicKeyX,
                                                const OCTETSTRING& p__ecdsaNistp256PublicKeyY
                                                ) {
    // Calculate the hash
    sha256 hash;
    std::vector<unsigned char> hashData;
    // TODO Create SHX interface and add generate method with std::vector
    std::vector<unsigned char> tbh(static_cast<const unsigned char *>(p__toBeVerifiedData), static_cast<const unsigned char *>(p__toBeVerifiedData) + p__toBeVerifiedData.lengthof());
    hash.generate(tbh, hashData);
    // Check the signature
    std::vector<unsigned char> signature(static_cast<const unsigned char *>(p__signature), static_cast<const unsigned char *>(p__signature) + p__signature.lengthof());
    std::vector<unsigned char> pub_key_x(static_cast<const unsigned char *>(p__ecdsaNistp256PublicKeyX), static_cast<const unsigned char *>(p__ecdsaNistp256PublicKeyX) + p__ecdsaNistp256PublicKeyX.lengthof());
    std::vector<unsigned char> pub_key_y(static_cast<const unsigned char *>(p__ecdsaNistp256PublicKeyY), static_cast<const unsigned char *>(p__ecdsaNistp256PublicKeyY) + p__ecdsaNistp256PublicKeyY.lengthof());
    ec_keys k(ec_elliptic_curves::nist_p_256, pub_key_x, pub_key_y);
    if (k.sign_verif(hashData, signature) == 0) {
      return TRUE;
    }

    return FALSE;
  }

  /**
   * @desc    Verify the signature of the specified data
   * @param   p_toBeVerifiedData          The data to be verified
   * @param   p_signature                 The signature
   * @param   p_ecdsaBrainpoolp256PublicKeyX   The public key (x coordinate)
   * @param   p_ecdsaBrainpoolp256PublicKeyY   The public key (y coordinate)
   * @return  true on success, false otherwise
   fx_verifyWithEcdsaBrainpoolp256WithSha256(in octetstring p_toBeVerifiedData, in octetstring p_signature, in octetstring p_ecdsaBrainpoolp256PublicKeyX, in octetstring p_ecdsaBrainpoolp256PublicKeyY) return boolean;
  */
  BOOLEAN fx__verifyWithEcdsaBrainpoolp256WithSha256(
                                                     const OCTETSTRING& p__toBeVerifiedData,
                                                     const OCTETSTRING& p__signature,
                                                     const OCTETSTRING& p__ecdsaBrainpoolp256PublicKeyX,
                                                     const OCTETSTRING& p__ecdsaBrainpoolp256PublicKeyY
                                                     ) {
    // Calculate the hash
    sha256 hash;
    std::vector<unsigned char> hashData;
    // TODO Create SHX interface and add generate method with std::vector
    std::vector<unsigned char> tbh(static_cast<const unsigned char *>(p__toBeVerifiedData), static_cast<const unsigned char *>(p__toBeVerifiedData) + p__toBeVerifiedData.lengthof());
    hash.generate(tbh, hashData);
    // Check the signature
    std::vector<unsigned char> signature(static_cast<const unsigned char *>(p__signature), static_cast<const unsigned char *>(p__signature) + p__signature.lengthof());
    std::vector<unsigned char> pub_key_x(static_cast<const unsigned char *>(p__ecdsaBrainpoolp256PublicKeyX), static_cast<const unsigned char *>(p__ecdsaBrainpoolp256PublicKeyX) + p__ecdsaBrainpoolp256PublicKeyX.lengthof());
    std::vector<unsigned char> pub_key_y(static_cast<const unsigned char *>(p__ecdsaBrainpoolp256PublicKeyY), static_cast<const unsigned char *>(p__ecdsaBrainpoolp256PublicKeyY) + p__ecdsaBrainpoolp256PublicKeyY.lengthof());
    ec_keys k(ec_elliptic_curves::brainpool_p_256_r1, pub_key_x, pub_key_y);
    if (k.sign_verif(hashData, signature) == 0) {
      return TRUE;
    }

    return FALSE;
  }

  /**
   * @desc    Verify the signature of the specified data
   * @param   p_toBeVerifiedData          The data to be verified
   * @param   p_signature                 The signature
   * @param   p_ecdsaBrainpoolp384PublicKeyX   The public key (x coordinate)
   * @param   p_ecdsaBrainpoolp384PublicKeyY   The public key (y coordinate)
   * @return  true on success, false otherwise
   fx_verifyWithEcdsaBrainpoolp384WithSha384(in octetstring p_toBeVerifiedData, in octetstring p_signature, in octetstring p_ecdsaBrainpoolp384PublicKeyX, in octetstring p_ecdsaBrainpoolp384PublicKeyY) return boolean;
  */
  BOOLEAN fx__verifyWithEcdsaBrainpoolp384WithSha384(
                                                     const OCTETSTRING& p__toBeVerifiedData,
                                                     const OCTETSTRING& p__signature,
                                                     const OCTETSTRING& p__ecdsaBrainpoolp384PublicKeyX,
                                                     const OCTETSTRING& p__ecdsaBrainpoolp384PublicKeyY
                                                     ) {
    // Calculate the hash
    sha384 hash;
    std::vector<unsigned char> hashData;
    // TODO Create SHX interface and add generate method with std::vector
    std::vector<unsigned char> tbh(static_cast<const unsigned char *>(p__toBeVerifiedData), static_cast<const unsigned char *>(p__toBeVerifiedData) + p__toBeVerifiedData.lengthof());
    hash.generate(tbh, hashData);
    // Check the signature
    std::vector<unsigned char> signature(static_cast<const unsigned char *>(p__signature), static_cast<const unsigned char *>(p__signature) + p__signature.lengthof());
    std::vector<unsigned char> pub_key_x(static_cast<const unsigned char *>(p__ecdsaBrainpoolp384PublicKeyX), static_cast<const unsigned char *>(p__ecdsaBrainpoolp384PublicKeyX) + p__ecdsaBrainpoolp384PublicKeyX.lengthof());
    std::vector<unsigned char> pub_key_y(static_cast<const unsigned char *>(p__ecdsaBrainpoolp384PublicKeyY), static_cast<const unsigned char *>(p__ecdsaBrainpoolp384PublicKeyY) + p__ecdsaBrainpoolp384PublicKeyY.lengthof());
    ec_keys k(ec_elliptic_curves::brainpool_p_384_r1, pub_key_x, pub_key_y);
    if (k.sign_verif(hashData, signature) == 0) {
      return TRUE;
    }

    return FALSE;
  }

  /**
   * @desc    Produce a new public/private key pair based on Elliptic Curve Digital Signature Algorithm (ECDSA) algorithm.
   *          This function should not be used by the ATS
   * @param   p_privateKey    The new private key value
   * @param   p_publicKeyX    The new public key value (x coordinate)
   * @param   p_publicKeyX    The new public key value (y coordinate)
   * @return  true on success, false otherwise
   fx_generateKeyPair_nistp256(out octetstring<UInt64> p_privateKey, out octetstring p_publicKeyX, out octetstring p_publicKeyY) return boolean;
  */
  BOOLEAN fx__generateKeyPair__nistp256(
                                        OCTETSTRING& p__privateKey,
                                        OCTETSTRING& p__publicKeyX,
                                        OCTETSTRING& p__publicKeyY
                                        ) {
    ec_keys k(ec_elliptic_curves::nist_p_256);
    if (k.generate() != 0) {
      p__privateKey = OCTETSTRING();
      p__publicKeyX = OCTETSTRING();
      p__publicKeyY = OCTETSTRING();
      return FALSE;
    }

    p__privateKey = OCTETSTRING(k.private_key().size(), k.private_key().data());
    p__publicKeyX = OCTETSTRING(k.public_key_x().size(), k.public_key_x().data());
    p__publicKeyY = OCTETSTRING(k.public_key_y().size(), k.public_key_y().data());
    return TRUE;
  }

  /**
   * @desc    Produce a new public/private key pair based on Elliptic Curve Digital Signature Algorithm (ECDSA) algorithm.
   *          This function should not be used by the ATS
   * @param   p_privateKey    The new private key value
   * @param   p_publicKeyX    The new public key value (x coordinate)
   * @param   p_publicKeyX    The new public key value (y coordinate)
   * @return  true on success, false otherwise
   fx_generateKeyPair_nistp256(out octetstring<UInt64> p_privateKey, out octetstring p_publicKeyX, out octetstring p_publicKeyY) return boolean;
  */
  BOOLEAN fx__generateKeyPair__brainpoolp256(
                                             OCTETSTRING& p__privateKey,
                                             OCTETSTRING& p__publicKeyX,
                                             OCTETSTRING& p__publicKeyY
                                             ) {
    ec_keys k(ec_elliptic_curves::brainpool_p_256_r1);
    if (k.generate() != 0) {
      p__privateKey = OCTETSTRING();
      p__publicKeyX = OCTETSTRING();
      p__publicKeyY = OCTETSTRING();
      return FALSE;
    }

    p__privateKey = OCTETSTRING(k.private_key().size(), k.private_key().data());
    p__publicKeyX = OCTETSTRING(k.public_key_x().size(), k.public_key_x().data());
    p__publicKeyY = OCTETSTRING(k.public_key_y().size(), k.public_key_y().data());
    return TRUE;
  }

  /**
   * @desc    Produce a new public/private key pair based on Elliptic Curve Digital Signature Algorithm (ECDSA) algorithm.
   *          This function should not be used by the ATS
   * @param   p_privateKey    The new private key value
   * @param   p_publicKeyX    The new public key value (x coordinate)
   * @param   p_publicKeyX    The new public key value (y coordinate)
   * @return  true on success, false otherwise
   fx_generateKeyPair_nistp256(out octetstring<UInt64> p_privateKey, out octetstring p_publicKeyX, out octetstring p_publicKeyY) return boolean;
  */
  BOOLEAN fx__generateKeyPair__brainpoolp384(
                                             OCTETSTRING& p__privateKey,
                                             OCTETSTRING& p__publicKeyX,
                                             OCTETSTRING& p__publicKeyY
                                             ) {
    ec_keys k(ec_elliptic_curves::brainpool_p_384_r1);
    if (k.generate() != 0) {
      p__privateKey = OCTETSTRING();
      p__publicKeyX = OCTETSTRING();
      p__publicKeyY = OCTETSTRING();
      return FALSE;
    }

    p__privateKey = OCTETSTRING(k.private_key().size(), k.private_key().data());
    p__publicKeyX = OCTETSTRING(k.public_key_x().size(), k.public_key_x().data());
    p__publicKeyY = OCTETSTRING(k.public_key_y().size(), k.public_key_y().data());
    return TRUE;
  }

  //        group encryption

  //        group certificatesLoader

  /* 
   * @desc    Load in memory cache the certificates available in the specified directory
   * @param   p_rootDirectory Root directory to access to the certificates identified by the certificate ID
   * @param   p_configId      A configuration identifier
   * @remark  This method SHALL be call before any usage of certificates
   * @return  true on success, false otherwise
   fx_loadCertificates(in charstring p_rootDirectory, in charstring p_configId) return boolean;
  */
  BOOLEAN fx__loadCertificates(
                               const CHARSTRING& p__rootDirectory,
                               const CHARSTRING& p__configId
                               ) {
    loggers::get_instance().log("fx__loadCertificates: '%s', '%s'", p__rootDirectory, p__configId);
    Params params;
    params.insert(std::pair<std::string, std::string>(std::string("sec_db_path"), std::string(static_cast<const char*>(p__rootDirectory))));
    if (security_services::get_instance().setup(params) == -1) {
      return FALSE;
    }
    
    return TRUE;
  }

  BOOLEAN fx__store__certificate(const CHARSTRING& p__cert__id, const OCTETSTRING& p__cert, const OCTETSTRING& p__private__key, const OCTETSTRING& p__public__key__x, const OCTETSTRING& p__public__key__y, const OCTETSTRING& p__hashid8, const OCTETSTRING& p__issuer) {
    if (security_services::get_instance().store_certificate(p__cert__id, p__cert, p__private__key, p__public__key__x, p__public__key__y, p__hashid8, p__issuer) == -1) {
      return FALSE;
    }
    
    return TRUE;
  }
  
  /*
   * @desc    Unload from memory cache the certificates
   * @return  true on success, false otherwise
   fx_unloadCertificates() return boolean;
  */
  BOOLEAN fx__unloadCertificates(
) {
    return TRUE;
  }

  /*          * @desc    Read the specified certificate
   * @param   p_certificateId the certificate identifier
   * @param   p_certificate   the expected certificate
   * @return  true on success, false otherwise
   fx_readCertificate(in charstring p_certificateId, out octetstring p_certificate) return boolean;
  */
  BOOLEAN fx__readCertificate(
                              const CHARSTRING& p__certificateId,
                              OCTETSTRING& p__certificate
                              ) {
    return TRUE;
  }


  /*          * @desc    Read the specified certificate digest
   * @param   p_certificateId the certificate identifier
   * @param   p_digest   the expected certificate
   * @return  true on success, false otherwise
   fx_readCertificateDigest(in charstring p_certificateId, out HashedId8 p_digest) return boolean;
  */
  BOOLEAN fx__readCertificateDigest(
                                    const CHARSTRING& p__certificateId,
                                    OCTETSTRING& p__digest
                                    ) {
    return TRUE;
  }

  /*          * @desc    Read the private keys for the specified certificate
   * @param   p_keysId            the keys identifier
   * @param   p_signingPrivateKey the signing private key
   * @return  true on success, false otherwise
   fx_readSigningKey(in charstring p_keysId, out Oct32 p_signingPrivateKey) return boolean;
  */
  BOOLEAN fx__readSigningKey(
                             const CHARSTRING& p__keysId,
                             OCTETSTRING& p__signingPrivateKey
                             ) {
    return TRUE;
  }

  /*          * @desc    Read the private keys for the specified certificate
   * @param   p_keysId            the keys identifier
   * @param   p_encryptPrivateKey the encrypt private key
   * @return  true on success, false otherwise
   fx_readEncryptingKey(in charstring p_keysId, out Oct32 p_encryptingPrivateKey) return boolean;
  */
  BOOLEAN fx__readEncryptingKey(
                                const CHARSTRING& p__keysId,
                                OCTETSTRING& p__encryptingPrivateKey
                                ) {
    return TRUE;
  }

  //        group geodesic 

  /*          * @desc    Check that given polygon doesn't have neither self-intersections nor holes.
   * @param   p_region   Polygonal Region
   * @return  true on success, false otherwise
   * @verdict Unchanged
   fx_isValidPolygonalRegion(in PolygonalRegion p_region) return boolean;
  */
  BOOLEAN fx__isValidPolygonalRegion(
                                     const IEEE1609dot2BaseTypes::PolygonalRegion& p__region
                                     ) {
    return TRUE;
  }

  /*          * @desc Check if a polygonal region is inside another one
   * @param p_parent  The main polygonal region
   * @param p_region  The polygonal region to be included
   * @return true on success, false otherwise
   * @verdict Unchanged
   fx_isPolygonalRegionInside(in PolygonalRegion p_parent, in PolygonalRegion p_region) return boolean;
  */
  BOOLEAN fx__isPolygonalRegionInside(
                                      const IEEE1609dot2BaseTypes::PolygonalRegion& p__parent,
                                      const IEEE1609dot2BaseTypes::PolygonalRegion& p__region
                                      ) {
    return TRUE;
  }

  /*          * @desc Check that the location is inside a circular region
   * @param p_region      The circular region to consider
   * @param p_location    The device location
   * @return true on success, false otherwise
   * @verdict Unchanged
   fx_isLocationInsideCircularRegion(in CircularRegion p_region, in ThreeDLocation p_location) return boolean;
  */
  BOOLEAN fx__isLocationInsideCircularRegion(
                                             const IEEE1609dot2BaseTypes::CircularRegion& p__region,
                                             const IEEE1609dot2BaseTypes::ThreeDLocation& p__location
                                             ) {
    return TRUE;
  }

  /*          * @desc Check that the location is inside a rectangular region
   * @param p_region      The rectangular region to consider
   * @param p_location    The device location
   * @return true on success, false otherwise
   * @verdict Unchanged
   fx_isLocationInsideRectangularRegion(in SequenceOfRectangularRegion p_region, in ThreeDLocation p_location) return boolean;
  */
  BOOLEAN fx__isLocationInsideRectangularRegion(
                                                const IEEE1609dot2BaseTypes::SequenceOfRectangularRegion& p__region,
                                                const IEEE1609dot2BaseTypes::ThreeDLocation& p__location
                                                ) {
    return TRUE;
  }

  /*          * @desc Check that the location is inside a polygonal region
   * @param p_region      The polygonal region to consider
   * @param p_location    The device location
   * @return true on success, false otherwise
   * @verdict Unchanged
   fx_isLocationInsidePolygonalRegion(in PolygonalRegion p_region, in ThreeDLocation p_location) return boolean;
  */
  BOOLEAN fx__isLocationInsidePolygonalRegion(
                                              const IEEE1609dot2BaseTypes::PolygonalRegion& p__region,
                                              const IEEE1609dot2BaseTypes::ThreeDLocation& p__location
                                              ) {
    return TRUE;
  }


  /*          * @desc Check if the location is inside an identified region
   * @param p_region      The identified region to consider
   * @param p_location    The device location
   * @return true on success, false otherwise
   * @verdict Unchanged
   fx_isLocationInsideIdentifiedRegion(in IdentifiedRegion p_region, in ThreeDLocation p_location) return boolean;
  */
  BOOLEAN fx__isLocationInsideIdentifiedRegion(
                                               const IEEE1609dot2BaseTypes::IdentifiedRegion& p__region,
                                               const IEEE1609dot2BaseTypes::ThreeDLocation& p__location
                                               ) {
    return TRUE;
  }

  /*          * @desc Check if the location is inside an undefined region
   * @param p_region      The identified region to consider
   * @param p_location    The device location
   * @return true on success, false otherwise
   * @verdict Unchanged
   fx_isLocationInsideOtherRegion(in octetstring p_region, in ThreeDLocation p_location) return boolean; 
  */
  BOOLEAN fx__isLocationInsideOtherRegion(
                                          const OCTETSTRING& p_region,
                                          const IEEE1609dot2BaseTypes::ThreeDLocation& p_location
                                          ) {
    return TRUE;
  } 

  /*           * @desc    Check that p_circular_region_1 circular region is included into p_circular_region_2 circular region
   * @param   p_circular_region_1    Circular region 1
   * @param   p_circular_region_2    Circular region 2
   * @return  true on success, false otherwise
   fx_areCirclesInside(in CircularRegion p_circular_region_1, in CircularRegion p_circular_region_2) return boolean;
  */
  BOOLEAN fx__areCirclesInside(
                               const IEEE1609dot2BaseTypes::CircularRegion& p_circular_region_1,
                               const IEEE1609dot2BaseTypes::CircularRegion& p_circular_region_2
                               ) {
    return TRUE;
  }

  /*           * @desc    Check that p_rectanglar_region_1 rectangular region is included into p_rectanglar_region_2 rectangular region
   * @param   p_rectanglar_region_1    Rectangular region 1
   * @param   p_rectanglar_region_2    Rectangular region 2
   * @return  true on success, false otherwise
   fx_areRectanglesInside(in SequenceOfRectangularRegion p_rectanglar_region_1, in SequenceOfRectangularRegion p_rectanglar_region_2) return boolean;
  */
  BOOLEAN fx__areRectanglesInside(
                                  const IEEE1609dot2BaseTypes::SequenceOfRectangularRegion& p_rectanglar_region_1,
                                  const IEEE1609dot2BaseTypes::SequenceOfRectangularRegion& p_rectanglar_region_2
                                  ) {
    return TRUE;
  }

  /*           * @desc    Check that p_polygonal_region_1 polygonal region is included into p_polygonal_region_2 polygonal region
   * @param   p_polygonal_region_1    Polygonal region 1
   * @param   p_polygonal_region_2    Polygonal region 2
   * @return  true on success, false otherwise
   fx_arePolygonsInside(in PolygonalRegion p_polygonal_region_1, in PolygonalRegion p_polygonal_region_2) return boolean;
  */
  BOOLEAN fx__arePolygonsInside(
                                const IEEE1609dot2BaseTypes::PolygonalRegion& p_polygonal_region_1,
                                const IEEE1609dot2BaseTypes::PolygonalRegion& p_polygonal_region_2
                                ) {
    return TRUE;
  }

  /*          * @desc Convert a spacial coordinate from DMS to Dms
   * @param p_degrees The degrees (D)
   * @param p_minutes The minutes (M)
   * @param p_seconds The seconds (S)
   * @param p_latlon  The latitude/longitude: (N|S|E|W)
   * @return The decimal coordinate on success, 0.0, otherwise
   * @verdict Unchanged
   fx_dms2dd(in Int p_degrees, in Int p_minutes, in float p_seconds, in Oct1 p_latlon) return float;
  */
  FLOAT fx__dms2dd(
                   const INTEGER& p__degrees,
                   const INTEGER& p__minutes,
                   const FLOAT& p__seconds,
                   const OCTETSTRING& p__latlon
                   ) {
    return 0.0;
  }

} // end of namespace
