package de.fraunhofer.sit.c2x.pki.ca.validator.pseudonym.intervaltree;

import java.util.ArrayList;
import java.util.List;

/**
 * @author Daniel Quanz (daniel.quanz@sit.fraunhofer.de)
 */
public class IntervalTree<DataType> {
	private Node<DataType> head;
	private List<Interval<DataType>> intervalList;
	private boolean inSync;
	private int size;

	/**
	 * Instantiate a new interval tree with no intervals
	 */
	public IntervalTree() {
		this.head = new Node<DataType>();
		this.intervalList = new ArrayList<Interval<DataType>>();
		this.inSync = true;
		this.size = 0;
	}

	/**
	 * Instantiate and build an interval tree with a preset list of intervals
	 * 
	 * @param intervalList
	 *            the list of intervals to use
	 */
	public IntervalTree(List<Interval<DataType>> intervalList) {
		this.head = new Node<DataType>(intervalList);
		this.intervalList = new ArrayList<Interval<DataType>>();
		this.intervalList.addAll(intervalList);
		this.inSync = true;
		this.size = intervalList.size();
	}

	/**
	 * Perform a stabbing query, returning the associated data Will rebuild the
	 * tree if out of sync
	 * 
	 * @param time
	 *            the time to stab
	 * @return the data associated with all intervals that contain time
	 */
	public List<DataType> get(long time) {
		List<Interval<DataType>> intervals = getIntervals(time);
		List<DataType> result = new ArrayList<DataType>();
		for (Interval<DataType> interval : intervals)
			result.add(interval.getData());
		return result;
	}

	/**
	 * Perform a stabbing query, returning the interval objects Will rebuild the
	 * tree if out of sync
	 * 
	 * @param time
	 *            the time to stab
	 * @return all intervals that contain time
	 */
	public List<Interval<DataType>> getIntervals(long time) {
		build();
		return head.stab(time);
	}

	/**
	 * Perform an interval query, returning the associated data Will rebuild the
	 * tree if out of sync
	 * 
	 * @param start
	 *            the start of the interval to check
	 * @param end
	 *            the end of the interval to check
	 * @return the data associated with all intervals that intersect target
	 */
	public List<DataType> get(long start, long end) {
		List<Interval<DataType>> intervals = getIntervals(start, end);
		List<DataType> result = new ArrayList<DataType>();
		for (Interval<DataType> interval : intervals)
			result.add(interval.getData());
		return result;
	}

	/**
	 * Perform an interval query, returning the interval objects Will rebuild
	 * the tree if out of sync
	 * 
	 * @param start
	 *            the start of the interval to check
	 * @param end
	 *            the end of the interval to check
	 * @return all intervals that intersect target
	 */
	public List<Interval<DataType>> getIntervals(long start, long end) {
		build();
		return head.query(new Interval<DataType>(start, end, null));
	}

	/**
	 * Add an interval object to the interval tree's list Will not rebuild the
	 * tree until the next query or call to build
	 * 
	 * @param interval
	 *            the interval object to add
	 */
	public void addInterval(Interval<DataType> interval) {
		intervalList.add(interval);
		inSync = false;
	}
	
	public void addAll(List<Interval<DataType>> lst) {
		for (Interval<DataType> interval : lst) {
			addInterval(interval);
		}
	}

	/**
	 * Determine whether this interval tree is currently a reflection of all
	 * intervals in the interval list
	 * 
	 * @return true if no changes have been made since the last build
	 */
	public boolean inSync() {
		return inSync;
	}

	/**
	 * Build the interval tree to reflect the list of intervals, Will not run if
	 * this is currently in sync
	 */
	public void build() {
		if (!inSync) {
			head = new Node<DataType>(intervalList);
			inSync = true;
			size = intervalList.size();
		}
	}

	/**
	 * @return the number of entries in the currently built interval tree
	 */
	public int currentSize() {
		return size;
	}

	/**
	 * @return the number of entries in the interval list, equal to .size() if
	 *         inSync()
	 */
	public int listSize() {
		return intervalList.size();
	}

	@Override
	public String toString() {
		return nodeString(head, 0);
	}

	private String nodeString(Node<DataType> node, int level) {
		if (node == null)
			return "";

		StringBuffer sb = new StringBuffer();
		for (int i = 0; i < level; i++)
			sb.append("\t");
		sb.append(node + "\n");
		sb.append(nodeString(node.getLeft(), level + 1));
		sb.append(nodeString(node.getRight(), level + 1));
		return sb.toString();
	}
}
