package de.fraunhofer.sit.c2x.pki.ca.utils;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.DataInputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.math.BigInteger;
import java.nio.charset.Charset;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashSet;
import java.util.Random;

import org.apache.commons.codec.binary.Hex;
import org.apache.log4j.xml.DOMConfigurator;

/**
 * 
 * @author Jan Peter Stotz (jan-peter.stotz@sit.fraunhofer.de)
 * @author Norbert Bissmeyer (norbert.bissmeyer@sit.fraunhofer.de)
 */
public class ByteUtils {

	/**
	 * <code>2<sup>32</sup>-1 = 4294967295l</code>
	 */
	public static final long MAX_UNSIGNED_INTEGER = 4294967295L;

	/**
	 * <code>2<sup>16</sup>-1 = 65535</code>
	 */
	public static final int MAX_UNSIGNED_SHORT = 65535;

	/**
	 * <code>2<sup>8</sup>-1 = 255</code>
	 */
	public static final int MAX_UNSIGNED_BYTE = 255;

	static final String HEXES = "0123456789ABCDEF";

	private static boolean LOGGING_CONFIGURED = false;

	public static final Charset UTF8 = Charset.forName("UTF-8");

	public static String byteToBitString(byte b) {
		char[] out = new char[8];
		for (int i = 7; i >= 0; i--) {
			out[i] = (b & 0x01) == 0 ? '0' : '1';
			b = (byte) (b >>> 1);
		}
		return new String(out);
	}

	/**
	 * Class <code>close()</code> on the given {@link InputStream} and ignores
	 * all thrown Exceptions.
	 * 
	 * @param stream
	 */
	public static void closeStream(InputStream stream) {
		try {
			stream.close();
		} catch (Exception e) {
			// Ingnore all exceptions
		}
	}

	/**
	 * Class <code>close()</code> on the given {@link OutputStream} and ignores
	 * all thrown Exceptions.
	 * 
	 * @param stream
	 */
	public static void closeStream(OutputStream stream) {
		try {
			stream.close();
		} catch (Exception e) {
			// Ingnore all exceptions
		}
	}

	public static synchronized void configureLogging() {
		// Test if logging has already been configured
		if (ByteUtils.LOGGING_CONFIGURED) {
			return;
		}
		ByteUtils.LOGGING_CONFIGURED = true;
		DOMConfigurator.configureAndWatch("log4j.xml", 60 * 1000);
	}

	public static int convert2ByteArrayToInt(byte[] byteArray) {
		byte[] buffer = new byte[2];
		System.arraycopy(byteArray, byteArray.length - 2, buffer, 0, 2);
		int i = 0;
		int pos = 0;

		i += (byteArray[pos++] & 0xFF) << 8;
		i += (byteArray[pos] & 0xFF) << 0;

		return i;
	}

	public static int convert4ByteArrayToInt(byte[] byteArray) {
		return ByteUtils.convert4ByteArrayToInt(byteArray, 0);
	}

	public static int convert4ByteArrayToInt(byte[] byteArray, int offset) {
		int i = 0;

		i += (byteArray[offset++] & 0xFF) << 24;
		i += (byteArray[offset++] & 0xFF) << 16;
		i += (byteArray[offset++] & 0xFF) << 8;
		i += byteArray[offset] & 0xFF;

		return i;
	}

	public static long convert4ByteArrayToLong(byte[] byteArray) {
		// byte[] buffer = new byte[4];
		// System.arraycopy(byteArray, byteArray.length - 4, buffer, 0, 4);
		// long l = 0;
		// int pos = 0;
		// l |= ((long) buffer[pos++] & 0xFF) << 24;
		// l |= ((long) buffer[pos++] & 0xFF) << 16;
		// l |= ((long) buffer[pos++] & 0xFF) << 8;
		// l |= ((long) buffer[pos++] & 0xFF) << 0;
		// Validator.checkCondition("byteArray. Length(byteArray) shall be lower or equal 4",
		// byteArray.length <= 4);
		return new BigInteger(1, byteArray).longValue();
	}

	public static int convert4ByteArrayToUInt(byte[] byteArray) throws IllegalArgumentException {
		return ByteUtils.convert4ByteArrayToUInt(byteArray, 0);
	}

	public static int convert4ByteArrayToUInt(byte[] byteArray, int offset) throws IllegalArgumentException {
		int i = ByteUtils.convert4ByteArrayToInt(byteArray, offset);
		if (i >= 0) {
			return i;
		}
		throw new IllegalArgumentException("Number out of range (larger than 2^31)");
	}

	public static byte[] convertBigIntTo4ByteArray(BigInteger val) {

		return ByteUtils.convertBigIntToArray(val, 4);
	}

	public static byte[] convertBigIntTo8ByteArray(BigInteger val) {

		return ByteUtils.convertBigIntToArray(val, 8);
	}

	public static byte[] convertBigIntToArray(BigInteger val, int numberOfBytes) {
		// byte[] buffer = new byte[9]; // first byte = 0x00 in order to get an
		// unsigned value
		// System.arraycopy(byteArray, byteArray.length - 8, buffer, 1, 8);

		// Validator.nullInput("val", val);
		// Validator.checkCondition("val. The value has to be a positve " + 8 *
		// numberOfBytes + "Bit Integer",
		// val.signum() != -1, val.bitLength() <= 8 * numberOfBytes);

		if (val.signum() == -1 || val.bitLength() > 8 * numberOfBytes)
			throw new IllegalArgumentException("The value has to be a positive " + 8 * numberOfBytes
					+ " Bit integer");

		byte[] tmp = new byte[numberOfBytes];
		int length = val.toByteArray().length;

		if (val.toByteArray().length > numberOfBytes) {
			System.arraycopy(val.toByteArray(), length - tmp.length, tmp, 0, numberOfBytes);
		} else {
			System.arraycopy(val.toByteArray(), 0, tmp, numberOfBytes - length, length);
		}

		return tmp;
	}

	// public static byte[] convertBigIntegerTo8ByteArray(BigInteger val) {
	// if (val.bitLength() > 64)
	// throw new IllegalArgumentException(
	// "BigInteger value bigger than 64 bit");
	// if (val.signum() < 0)
	// throw new IllegalArgumentException("Negative value not allowed");
	// byte[] buffer = new byte[8];
	// // copy values of the bigInt to the byte array if bigInt is large enough
	// for (int i = 0; i <= 7; i++) {
	// if (val.bitLength() >= (i * 8)) {
	// // if bigInt is long enough the value of the bigInt is written
	// // to the buffer
	// if (val.bitLength() > 63)
	// buffer[(7 - i)] = val.toByteArray()[i + 1];
	// else
	// buffer[(7 - i)] = val.toByteArray()[i];
	// } else {
	// // else write zeros to buffer
	// buffer[(7 - i)] = (byte) 0x0;
	// }
	// }
	// return buffer;
	// }

	// public static byte[] convertBigIntegerTo4ByteArray(BigInteger val) {
	// if (val.bitLength() > 32)
	// throw new IllegalArgumentException(
	// "BigInteger value bigger than 32 bit");
	// if (val.signum() < 0)
	// throw new IllegalArgumentException("Negative value not allowed");
	// byte[] buffer = new byte[4];
	// // copy values of the bigInt to the byte array if bigInt is large enough
	// for (int i = 0; i <= 3; i++) {
	// if (val.bitLength() >= (i * 8)) {
	// // if bigInt is long enough the value of the bigInt is written
	// // to the buffer
	// if (val.bitLength() > 31)
	// buffer[(3 - i)] = val.toByteArray()[i + 1];
	// else
	// buffer[(3 - i)] = val.toByteArray()[i];
	// } else {
	// // else write zeros to buffer
	// buffer[(3 - i)] = (byte) 0x0;
	// }
	// }
	// return buffer;
	// }

	public static BigInteger convertByteArrayToBigInt(byte[] byteArray) {
		// byte[] buffer = new byte[9]; // first byte = 0x00 in order to get an
		// unsigned value
		// System.arraycopy(byteArray, byteArray.length - 8, buffer, 1, 8);
		return new BigInteger(1, byteArray);
	}

	public static byte[] convertIntTo2ByteArray(int val) {
		if (val > ByteUtils.MAX_UNSIGNED_SHORT) {
			throw new IllegalArgumentException("Integer value bigger than " + ByteUtils.MAX_UNSIGNED_SHORT);
		}
		// if (val < 0)
		// throw new IllegalArgumentException("Negative integer not allowed");
		byte[] buffer = new byte[2];

		buffer[0] = (byte) (val >>> 8);
		buffer[1] = (byte) val;

		return buffer;
	}

	public static byte[] convertLongTo4ByteArray(long val) {
		if (val > ByteUtils.MAX_UNSIGNED_INTEGER) {
			throw new IllegalArgumentException("Long value bigger than " + ByteUtils.MAX_UNSIGNED_INTEGER);
		}
		if (val < 0) {
			throw new IllegalArgumentException("Negative long not allowed");
		}
		byte[] buffer = new byte[4];

		buffer[0] = (byte) (val >>> 24);
		buffer[1] = (byte) (val >>> 16);
		buffer[2] = (byte) (val >>> 8);
		buffer[3] = (byte) val;

		return buffer;
	}

	public static byte[] convertSignedIntTo4ByteArray(int val) {
		byte[] buffer = new byte[4];

		buffer[0] = (byte) (val >>> 24);
		buffer[1] = (byte) (val >>> 16);
		buffer[2] = (byte) (val >>> 8);
		buffer[3] = (byte) val;

		return buffer;
	}

	public static byte[] getBytesFromFile(File file) throws IOException {
		FileInputStream in = null;
		try {
			in = new FileInputStream(file);
			return ByteUtils.getBytesFromStream(in);
		} finally {
			ByteUtils.closeStream(in);
		}
	}

	/**
	 * Reads all bytes from <code>in</code> until the end of the stream has been
	 * reached. Before returning the read bytes the stream is closed.
	 * 
	 * @param in
	 * @return
	 * @throws IOException
	 */
	public static byte[] getBytesFromStream(InputStream in) throws IOException {
		try {
			ByteArrayOutputStream byteArrayOutputStream = new ByteArrayOutputStream(in.available());
			byte[] buffer = new byte[16384];

			for (int len = in.read(buffer); len > 0; len = in.read(buffer)) {
				byteArrayOutputStream.write(buffer, 0, len);
			}
			return byteArrayOutputStream.toByteArray();

		} finally {
			in.close();
		}
	}

	public static String getHex(byte[] raw) {

		return Hex.encodeHexString(raw);
	}

	public static boolean hasDuplicates(Object[] array) {

		HashSet<Object> set = new HashSet<Object>();
		ArrayList<Object> lst = new ArrayList<Object>();
		Collections.addAll(lst, array);
		set.addAll(lst);

		return array.length != set.size();
	}

	/**
	 * A simple way to test if we are currently in an OSGi environment. The
	 * current implementation bases on the {@link ClassLoader} and it's package
	 * name which should contain a ".osgi." (is true for Equinox - other OSGi
	 * environments may be different!).
	 * 
	 * @return
	 */
	public static boolean testForOSGi() {
		return ByteUtils.class.getClassLoader().getClass().getName().contains(".osgi.");
	}

	public static void writeBytesToFile(File f, byte[] data) throws IOException {
		FileOutputStream fo = null;
		try {
			fo = new FileOutputStream(f);
			fo.write(data);
		} finally {
			ByteUtils.closeStream(fo);
		}
	}

	public static byte[] hexStringToByteArray(String s) {
		int len = s.length();
		byte[] data = new byte[len / 2];
		for (int i = 0; i < len; i += 2) {
			data[i / 2] = (byte) ((Character.digit(s.charAt(i), 16) << 4) + Character.digit(s.charAt(i + 1),
					16));
		}
		return data;
	}

	/**
	 * load from resources
	 * 
	 * @param file
	 * @return
	 */
	public InputStream loadFile(String file) {
		InputStream f = getClass().getResourceAsStream(file);
		// When executed as unit test, config file resides in another directory
		if (f == null) {
			try {
				f = new FileInputStream("resources" + file);
			} catch (FileNotFoundException e) {
				return null;
			}
		}
		return f;
	}

	public static byte[] concatBytes(byte[]... bytes) {

		int size = 0;
		for (int i = 0; i < bytes.length; i++) {
			size += bytes[i].length;
		}
		byte[] concatBytes = new byte[size];
		int pointer = 0;
		for (int i = 0; i < bytes.length; i++) {
			System.arraycopy(bytes[i], 0, concatBytes, pointer, bytes[i].length);
			pointer += bytes[i].length;
		}
		return concatBytes;
	}

	public static double round(Double a, int n) {
		double b = Math.pow(10.0, n);
		return Math.round(a * b) / b;
	}

	public static byte[] toLittleEndian(byte[] bigEndianBytes) {
		byte[] littleEndianBytes = revertByteArray(bigEndianBytes);
		return littleEndianBytes;
	}

	public static byte[] toBigEndian(byte[] littleEndianBytes) {
		byte[] bigEndianBytes = revertByteArray(littleEndianBytes);
		return bigEndianBytes;
	}

	public static byte[] trim(byte[] littleEndianBytes) {

		int i = 0;
		for (byte b : littleEndianBytes) {
			if (b != 0x00)
				break;
			i++;
		}
		if (i == 0)
			return littleEndianBytes;
		else {
			byte[] tmp = new byte[littleEndianBytes.length - i];
			System.arraycopy(littleEndianBytes, i, tmp, 0, tmp.length);
			return tmp;
		}
	}

	/**
	 * @param bigEndianBytes
	 * @return
	 */
	private static byte[] revertByteArray(byte[] input) {

		byte[] tmp = new byte[input.length];
		int size = 0;
		boolean skipZeros = true;
		for (int i = 0; i < input.length; i++) {
			if (input[i] == 0 && skipZeros) {
			} else {
				tmp[input.length - 1 - i] = input[i];
				size++;
				skipZeros = false;
			}
		}
		if (size == tmp.length) {
			return tmp;
		} else {
			byte[] tmp2 = new byte[size];
			System.arraycopy(tmp, 0, tmp2, 0, tmp2.length);
			return tmp2;
		}
	}

	public static void print(String name, byte[] bytes) {
		StringBuilder sb = new StringBuilder();
		sb.append(name);
		sb.append(" - (");
		sb.append(bytes.length);
		sb.append(" Bytes)\n");
		int i = 0;
		for (byte b : bytes) {
			sb.append("0x");
			sb.append(Hex.encodeHex(new byte[] { b }, false));
			sb.append("U, ");
			if (++i == 16) {
				i = 0;
				sb.append("\n");
			}
		}
		System.out.println(sb.toString().subSequence(0, sb.toString().length() - ((i == 0) ? 3 : 2)));
		System.out.println();
		System.out.println();
	}

	public static DataInputStream bytesAsStream(byte[] bytes) {

		ByteArrayInputStream bin = new ByteArrayInputStream(bytes);
		DataInputStream in = new DataInputStream(bin);
		try {
			bin.close();
		} catch (IOException e) {
			e.printStackTrace();
		}
		return in;
	}

	public static byte[] randomBytes(int i) {

		byte[] tmp = new byte[i];
		new Random().nextBytes(tmp);
		return tmp;
	}
}
