package de.fraunhofer.sit.c2x.pki.ca.provider.jdbc;

import java.security.KeyStore;
import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.ArrayList;
import java.util.List;

import javax.net.ssl.HttpsURLConnection;
import javax.net.ssl.KeyManagerFactory;
import javax.net.ssl.SSLContext;
import javax.net.ssl.TrustManager;
import javax.net.ssl.X509TrustManager;

import org.apache.commons.codec.binary.Hex;
import org.apache.log4j.Logger;

import com.mysql.jdbc.Statement;

import de.fraunhofer.sit.c2x.pki.ca.core.logging.InjectLogger;
import de.fraunhofer.sit.c2x.pki.ca.measuring.ItsStationType;
import de.fraunhofer.sit.c2x.pki.ca.provider.ProviderException;
import de.fraunhofer.sit.c2x.pki.ca.provider.entities.Constants;
import de.fraunhofer.sit.c2x.pki.ca.provider.entities.Host;
import de.fraunhofer.sit.c2x.pki.ca.provider.entities.KnownCA;
import de.fraunhofer.sit.c2x.pki.ca.provider.interfaces.KnownCaProvider;

public class JDBCKnownCaProvider extends AbstractMysqlConnection implements KnownCaProvider {

	@Override
	public String getName() {
		return "KnownCa provider";
	}

	@InjectLogger
	private Logger logger;

	private boolean initialized = false;

	public JDBCKnownCaProvider() {
	}

	public void updateLoadedX509ClientCerts() throws ProviderException {
		if (initialized == false) {
			final TrustManager[] trustAllCerts = new TrustManager[] { new X509TrustManager() {

				public java.security.cert.X509Certificate[] getAcceptedIssuers() {
					return null;
				}

				public void checkClientTrusted(java.security.cert.X509Certificate[] certs, String authType) {
				}

				public void checkServerTrusted(java.security.cert.X509Certificate[] certs, String authType) {
				}

			} };

			try {
				// try to load key and certificate into keystore
				KeyStore ks = KeyStore.getInstance("JKS");
				ks.load(null, "".toCharArray());
				KeyManagerFactory keyManagerFactory = KeyManagerFactory.getInstance("SunX509");

				for (KnownCA ca : getAll(0, 1000)) {
					if (ca.getX509ClientCertString().isEmpty() == false && ca.getX509ClientCert() != null
							&& ca.getRsaClientKey() != null && ca.getRsaClientKeyPassword() != null) {
						ks.setKeyEntry(ca.getX509ClientCert().getSubjectDN().getName(), ca.getRsaClientKey(),
								ca.getRsaClientKeyPassword().toCharArray(),
								new java.security.cert.Certificate[] { ca.getX509ClientCert() });
						keyManagerFactory.init(ks, ca.getRsaClientKeyPassword().toCharArray());

						if (logger.isDebugEnabled())
							logger.debug("X.509 cert added to client key store: "
									+ ca.getX509ClientCert().getSubjectDN().toString());
					}
				}

				final SSLContext sslContext = SSLContext.getInstance("SSL");
				sslContext.init(keyManagerFactory.getKeyManagers(), trustAllCerts,
						new java.security.SecureRandom());
				HttpsURLConnection.setDefaultSSLSocketFactory(sslContext.getSocketFactory());

				initialized = true;
			} catch (final Exception e) {
				throw new ProviderException("Unable to load keystore with X.509 certificate: ", e);
			}
		}
	}

	@Override
	public boolean save(Host hostInfo, String caType, byte[] certId, byte[] certificate,
			String x509ClientCert, String rsaClientKey, String rsaClientKeyPassword) throws ProviderException {
		
		if(caType.equals(ItsStationType.RCA.toString()) && get(ItsStationType.RCA.toString()).length > 0)
			throw new ProviderException("One RCA already availabe in DB. Please delete this knwon RCA first.");
		
		String sql = String.format(
				"INSERT INTO %s (%s, %s, %s, %s, %s, %s, %s, %s, %s, %s) VALUES (?,?,?,?,?,?,?,?,?,?);",
				Constants.KNOWN_CAS__TABLE, Constants.HOST__COL__ADDRESS, Constants.HOST__COL__PORT_HTTP,
				Constants.HOST__COL__PORT_UDP, Constants.HOST__COL__WSDL_URL,
				Constants.KNOWN_CAS__COL__CA_TYPE, Constants.KNOWN_CAS__COL__CERT_ID,
				Constants.KNOWN_CAS__COL__CERTIFICATE, Constants.KNOWN_CAS__COL__X509_CLIENT_CERT,
				Constants.KNOWN_CAS__COL__RSA_CLIENT_KEY, Constants.KNOWN_CAS__COL__RSA_CLIENT_KEY_PASSWORD);
		Connection con = getConnection();
		PreparedStatement st = null;
		try {
			st = con.prepareStatement(sql);
			st.setString(1, hostInfo.getAdress());

			if (hostInfo.getHttpPort() != null)
				st.setInt(2, hostInfo.getHttpPort());
			else
				st.setInt(2, 0);

			if (hostInfo.getUdpPort() != null)
				st.setInt(3, hostInfo.getUdpPort());
			else
				st.setInt(3, 0);

			st.setString(4, hostInfo.getWsdlUrl());
			st.setString(5, caType);
			st.setBytes(6, certId);
			st.setBytes(7, certificate);
			st.setString(8, x509ClientCert);
			st.setString(9, rsaClientKey);
			st.setString(10, rsaClientKeyPassword);
			st.execute();

			try {
				updateLoadedX509ClientCerts();
			} catch (Exception e) {
				logger.error("Unable to update keystore");
			}

			return true;
		} catch (SQLException e) {
			logger.error("New KnownCA not stored: " + e.getMessage());
			throw new ProviderException(e);
		} finally {
			closeStatement(st);
			closeConnection(con);
		}
	}

	@Override
	public KnownCA get(byte[] certId) throws ProviderException {
		String sql = String.format("SELECT * FROM %s WHERE %s = ?", Constants.KNOWN_CAS__TABLE,
				Constants.KNOWN_CAS__COL__CERT_ID, certId);
		Connection con = getConnection();
		PreparedStatement st = null;
		ResultSet result = null;
		try {
			st = con.prepareStatement(sql);
			st.setBytes(1, certId);
			st.execute();
			result = st.getResultSet();

			while (result.next()) {
				Host host = new Host();
				KnownCA knownCA = new KnownCA();
				host.setAdress(result.getString(Constants.HOST__COL__ADDRESS));
				host.setHttpPort(result.getShort(Constants.HOST__COL__PORT_HTTP));
				host.setUdpPort(result.getShort(Constants.HOST__COL__PORT_UDP));
				host.setWsdlUrl(result.getString(Constants.HOST__COL__WSDL_URL));

				knownCA.setHostInfo(host);
				knownCA.setCaType(result.getString(Constants.KNOWN_CAS__COL__CA_TYPE));
				knownCA.setCertId(result.getBytes(Constants.KNOWN_CAS__COL__CERT_ID));
				knownCA.setCertificate(result.getBytes(Constants.KNOWN_CAS__COL__CERTIFICATE));
				knownCA.setX509ClientCert(result.getString(Constants.KNOWN_CAS__COL__X509_CLIENT_CERT));
				knownCA.setRsaClientKey(result.getString(Constants.KNOWN_CAS__COL__RSA_CLIENT_KEY));
				knownCA.setRsaClientKeyPassword(result
						.getString(Constants.KNOWN_CAS__COL__RSA_CLIENT_KEY_PASSWORD));
				return knownCA;
			}
			if (logger.isDebugEnabled())
				logger.debug("KnownCa with certId " + Hex.encodeHexString(certId) + " not found in DB");
			return null;
		} catch (SQLException e) {
			logger.error(e);
			throw new ProviderException(e);
		} finally {
			closeStatement(st);
			closeConnection(con);
			closeResultSet(result);
		}
	}

	@Override
	public KnownCA[] getAll(int offset, int limit) throws ProviderException {
		String sql = String.format("SELECT * FROM %s ORDER BY %s DESC", Constants.KNOWN_CAS__TABLE,
				Constants.KNOWN_CAS__COL__CA_TYPE);
		Connection con = getConnection();
		PreparedStatement st = null;
		ResultSet result = null;
		try {
			st = con.prepareStatement(sql);
			st.execute();
			result = st.getResultSet();

			List<KnownCA> tmp = new ArrayList<>();
			while (result.next()) {

				Host host = new Host();
				KnownCA knownCA = new KnownCA();
				host.setAdress(result.getString(Constants.HOST__COL__ADDRESS));
				host.setHttpPort(result.getShort(Constants.HOST__COL__PORT_HTTP));
				host.setUdpPort(result.getShort(Constants.HOST__COL__PORT_UDP));
				host.setWsdlUrl(result.getString(Constants.HOST__COL__WSDL_URL));

				knownCA.setHostInfo(host);
				knownCA.setCaType(result.getString(Constants.KNOWN_CAS__COL__CA_TYPE));
				knownCA.setCertId(result.getBytes(Constants.KNOWN_CAS__COL__CERT_ID));
				knownCA.setCertificate(result.getBytes(Constants.KNOWN_CAS__COL__CERTIFICATE));
				knownCA.setX509ClientCert(result.getString(Constants.KNOWN_CAS__COL__X509_CLIENT_CERT));
				knownCA.setRsaClientKey(result.getString(Constants.KNOWN_CAS__COL__RSA_CLIENT_KEY));
				knownCA.setRsaClientKeyPassword(result
						.getString(Constants.KNOWN_CAS__COL__RSA_CLIENT_KEY_PASSWORD));

				tmp.add(knownCA);
			}
			return tmp.toArray(new KnownCA[tmp.size()]);
		} catch (SQLException e) {
			logger.error(e);
			throw new ProviderException(e);
		} finally {
			closeStatement(st);
			closeConnection(con);
			closeResultSet(result);
		}
	}
	
	@Override
	public KnownCA getKnownRCA() throws ProviderException {
		KnownCA[] knownRCAs = get(ItsStationType.RCA.toString());
		if(knownRCAs.length == 0)
			throw new ProviderException("No RCA known. Please insert RCA as known CA");
		if(knownRCAs.length > 1)
			throw new ProviderException("More than one known RCAs found in DB");
		return knownRCAs[0];
	}

	@Override
	public KnownCA[] get(String caType) throws ProviderException {
		String sql = String.format("SELECT * FROM %s WHERE %s = '%s' ORDER BY %s DESC",
				Constants.KNOWN_CAS__TABLE, Constants.KNOWN_CAS__COL__CA_TYPE, caType,
				Constants.KNOWN_CAS__COL__CA_TYPE);
		Connection con = getConnection();
		PreparedStatement st = null;
		ResultSet result = null;
		try {
			st = con.prepareStatement(sql);
			st.execute();
			result = st.getResultSet();

			List<KnownCA> tmp = new ArrayList<>();
			while (result.next()) {

				Host host = new Host();
				KnownCA knownCA = new KnownCA();
				host.setAdress(result.getString(Constants.HOST__COL__ADDRESS));
				host.setHttpPort(result.getShort(Constants.HOST__COL__PORT_HTTP));
				host.setUdpPort(result.getShort(Constants.HOST__COL__PORT_UDP));
				host.setWsdlUrl(result.getString(Constants.HOST__COL__WSDL_URL));

				knownCA.setHostInfo(host);
				knownCA.setCaType(result.getString(Constants.KNOWN_CAS__COL__CA_TYPE));
				knownCA.setCertId(result.getBytes(Constants.KNOWN_CAS__COL__CERT_ID));
				knownCA.setCertificate(result.getBytes(Constants.KNOWN_CAS__COL__CERTIFICATE));
				knownCA.setX509ClientCert(result.getString(Constants.KNOWN_CAS__COL__X509_CLIENT_CERT));
				knownCA.setRsaClientKey(result.getString(Constants.KNOWN_CAS__COL__RSA_CLIENT_KEY));
				knownCA.setRsaClientKeyPassword(result
						.getString(Constants.KNOWN_CAS__COL__RSA_CLIENT_KEY_PASSWORD));

				tmp.add(knownCA);
			}
			return tmp.toArray(new KnownCA[tmp.size()]);
		} catch (SQLException e) {
			logger.error(e);
			throw new ProviderException(e);
		} finally {
			closeStatement(st);
			closeConnection(con);
			closeResultSet(result);
		}
	}

	@Override
	public int removeKnownCA(byte[] certId) throws ProviderException {
		String sql = String.format("DELETE FROM %s WHERE %s = ?;", Constants.KNOWN_CAS__TABLE,
				Constants.KNOWN_CAS__COL__CERT_ID);
		Connection con = getConnection();
		PreparedStatement st = null;
		try {
			st = con.prepareStatement(sql, Statement.RETURN_GENERATED_KEYS);
			st.setBytes(1, certId);

			int result = st.executeUpdate();

			try {
				updateLoadedX509ClientCerts();
			} catch (Exception e) {
				logger.error("Unable to update keystore");
			}

			return result;
		} catch (SQLException e) {
			if (logger.isDebugEnabled())
				logger.debug("Cannot delete KnownCA: " + Hex.encodeHexString(certId), e);
			throw new ProviderException("Cannot delete KnownCA: " + Hex.encodeHexString(certId), e);
		} finally {
			closeStatement(st);
			closeConnection(con);
		}
	}

}
