package de.fraunhofer.sit.c2x.pki.ca.provider.entities;

import java.io.BufferedInputStream;
import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.io.StringReader;
import java.security.KeyPair;
import java.security.PrivateKey;
import java.security.Security;
import java.security.cert.Certificate;
import java.security.cert.CertificateException;
import java.security.cert.CertificateFactory;
import java.security.cert.X509Certificate;
import java.security.interfaces.RSAPrivateKey;

import javax.persistence.Column;
import javax.persistence.EmbeddedId;
import javax.persistence.Entity;
import javax.persistence.Table;

import org.bouncycastle.jce.provider.BouncyCastleProvider;
import org.bouncycastle.jce.provider.JCERSAPrivateCrtKey;
import org.bouncycastle.openssl.PEMReader;
import org.bouncycastle.openssl.PasswordFinder;

import de.fraunhofer.sit.c2x.pki.ca.crypto.DefaultPasswordFinder;

@Entity
@Table(name = Constants.KNOWN_CAS__TABLE)
public class KnownCA {

	@EmbeddedId
	private Host hostInfo;

	@Column(name = Constants.KNOWN_CAS__COL__CA_TYPE, nullable = false)
	private String caType;

	@Column(name = Constants.KNOWN_CAS__COL__CERT_ID, nullable = true, columnDefinition = "varbinary(10)")
	private byte[] certId;

	@Column(name = Constants.KNOWN_CAS__COL__CERTIFICATE, nullable = true, columnDefinition = "blob")
	private byte[] certificate;

	@Column(name = Constants.KNOWN_CAS__COL__X509_CLIENT_CERT, columnDefinition = "text", nullable = true)
	private String x509ClientCert;

	@Column(name = Constants.KNOWN_CAS__COL__RSA_CLIENT_KEY, columnDefinition = "text", nullable = true)
	private String rsaClientKey;

	@Column(name = Constants.KNOWN_CAS__COL__RSA_CLIENT_KEY_PASSWORD, columnDefinition = "varchar(30)", nullable = true)
	private String rsaClientKeyPassword;

	public KnownCA() {
	}

	public String getRsaClientKeyPassword() {
		return rsaClientKeyPassword;
	}

	public void setRsaClientKeyPassword(String rsaClientKeyPassword) {
		this.rsaClientKeyPassword = rsaClientKeyPassword;
	}

	public Host getHostInfo() {
		return hostInfo;
	}

	public String getX509ClientCertString() {
		return x509ClientCert;
	}

	public String getRsaClientKeyString() {
		return rsaClientKey;
	}

	public RSAPrivateKey getRsaClientKey() throws IOException {
		if (rsaClientKey != null && !rsaClientKey.isEmpty())
			return createKeyFromString(rsaClientKey, rsaClientKeyPassword);
		else
			return null;
	}

	public void setRsaClientKey(String rsaClientKey) {
		this.rsaClientKey = rsaClientKey;
	}

	public X509Certificate getX509ClientCert() throws IOException {
		if (x509ClientCert != null && !x509ClientCert.isEmpty())
			return createX509FromString(x509ClientCert);
		else
			return null;
	}

	public void setX509ClientCert(String x509ClientCert) {
		this.x509ClientCert = x509ClientCert;
	}

	public void setHostInfo(Host hostInfo) {
		this.hostInfo = hostInfo;
	}

	public byte[] getCertId() {
		return certId;
	}

	public void setCertId(byte[] certId) {
		this.certId = certId;
	}

	public byte[] getCertificate() {
		return certificate;
	}

	public void setCertificate(byte[] certificate) {
		this.certificate = certificate;
	}

	public String getCaType() {
		return caType;
	}

	public void setCaType(String caType) {
		this.caType = caType;
	}

	public static RSAPrivateKey createKeyFromString(String keyString, String password) throws IOException {

		if (!keyString.contains("-----BEGIN RSA PRIVATE KEY-----")
				&& !keyString.contains("-----END RSA PRIVATE KEY-----")) {
			throw new IOException("RSA private key has wrong format. Key must contain "
					+ "\"-----BEGIN RSA PRIVATE KEY-----\" and \"-----END RSA PRIVATE KEY-----\"");
		}
		Security.addProvider(new BouncyCastleProvider());

		PasswordFinder passwordFinder = new DefaultPasswordFinder(password.toCharArray());
		KeyPair kp = (KeyPair) new PEMReader(new StringReader(keyString), passwordFinder).readObject();
		PrivateKey key = kp.getPrivate();
		
		if (key.getClass() == JCERSAPrivateCrtKey.class)
			return (RSAPrivateKey) key;
		else
			throw new IOException("key is not a RSA private key: " + key.getClass());
	}

	public static X509Certificate createX509FromString(String certificate) throws IOException {
		if (!certificate.contains("-----BEGIN CERTIFICATE-----")
				&& !certificate.contains("-----END CERTIFICATE-----")) {
			throw new IOException("X.509 certificate has wrong format. Certificate must contain "
					+ "\"-----BEGIN CERTIFICATE-----\" and \"-----END CERTIFICATE-----\"");
		}

		X509Certificate x509Cert = null;
		byte[] certificateBytes = certificate.getBytes();
		ByteArrayInputStream bais;
		BufferedInputStream bis;
		try {
			CertificateFactory cf = CertificateFactory.getInstance("X.509");
			bais = new ByteArrayInputStream(certificateBytes);
			bis = new BufferedInputStream(bais);
			while (bis.available() > 0) {
				Certificate cert = cf.generateCertificate(bis);
				if (cert.getType() != "X.509")
					throw new CertificateException("Certificate type " + cert.getType()
							+ " not supported for trust store");
				x509Cert = (X509Certificate) cert;
			}
			x509Cert.checkValidity();
		} catch (CertificateException e) {
			throw new IOException("Unable to decode X.509 certificate", e);
		}
		bais.close();
		bis.close();
		return x509Cert;
	}
}
