package de.fraunhofer.sit.c2x.pki.ca.module.webserver.servlets;

import java.sql.Timestamp;
import java.text.DecimalFormat;
import java.util.Calendar;
import java.util.TimeZone;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

import org.apache.log4j.Logger;

import com.google.inject.Inject;

import de.fraunhofer.sit.c2x.pki.ca.core.logging.InjectLogger;
import de.fraunhofer.sit.c2x.pki.ca.measuring.MeasuringKey;
import de.fraunhofer.sit.c2x.pki.ca.measuring.MeasuringStatistics;
import de.fraunhofer.sit.c2x.pki.ca.measuring.MeasuringStatisticsEntry;
import de.fraunhofer.sit.c2x.pki.ca.module.webserver.interfaces.HtmlProvider;
import de.fraunhofer.sit.c2x.pki.ca.provider.ProviderException;
import de.fraunhofer.sit.c2x.pki.ca.provider.interfaces.ConfigProvider;
import de.fraunhofer.sit.c2x.pki.ca.provider.interfaces.MeasuringProvider;

/**
 * A simple servlet that outputs the content of the file log/pki.log via HTTP
 * 
 * <ul>
 * <li>10 seconds: <b>/log?period=10s</b></li>
 * <li>10 minutes: <b>/log?period=10m</b></li>
 * <li>10 hours: <b>/log?period=10h</b></li>
 * </ul>
 * 
 * @author Jan Peter Stotz (jan-peter.stotz@sit.fraunhofer.de) Norbert Bissmeyer
 *         (norbert.bissmeyer@sit.fraunhofer.de)
 * 
 */
public class Statistics extends AbstractServlet {

	/**
	 * 
	 */
	private static final long serialVersionUID = 1L;

	@InjectLogger
	private Logger logger;
	public static final String URL = "/statistics";
	public static final String TITLE = "Statistics";
	public static final String DATE_PATTERN_LOG = "yyyy-MM-dd HH:mm:ss,SSS";
	public static final DecimalFormat f = new DecimalFormat("#0.00");

	@Inject
	private MeasuringProvider measuringProvider;
	
	@Inject
	private ConfigProvider configProvider;

	@Inject
	public Statistics(HtmlProvider htmlProvider) {
		super(htmlProvider);
	}

	@Override
	public String getUrl() {
		return URL;
	}

	@Override
	public String getTitle() {
		return TITLE;
	}

	@Override
	protected String contentHtml(HttpServletRequest req, HttpServletResponse resp) {
		TimeZone tz = TimeZone.getTimeZone(configProvider.get("webpageLogTimeZone"));
		Calendar c = Calendar.getInstance(tz);
		
		String userInfo = "";
		Timestamp start = null;
		Timestamp end = null;
		if (req.getParameter("start") != null && req.getParameter("start").length() > 0) {
			// TODO: read start time from web page
		} else {
			
			start = new Timestamp(c.getTimeInMillis());
		}

		if (req.getParameter("end") != null && req.getParameter("end").length() > 0) {
			// TODO read end time from web page
		} else {
			end = new Timestamp(c.getTimeInMillis());
		}

		StringBuilder sb = new StringBuilder();
		MeasuringStatistics stats = null;

		try {
			stats = measuringProvider.getStatistics(start, end);
		} catch (ProviderException e) {
			logger.error(e.getMessage());
			userInfo = "<div id=\"errorbox\">Unable to get statistics from DB</div>";
		}
		if (stats == null || stats.getStatistics() == null)
			userInfo = "<div id=\"errorbox\">Unable to get statistics from DB</div>";

		if (userInfo != "") {
			sb.append("<br/><br/>" + userInfo);
		} else {
			String group = "";
			for (String key : stats.getStatistics().keySet()) {
				if (MeasuringKey.getGroup(key).equals(group) == false) {
					if (group.isEmpty() == false) {
						sb.append("</td></tr></table>\n<br />\n");
					}
					group = MeasuringKey.getGroup(key);
					sb.append("<h2>" + group
							+ "</h2><table border=\"0\" style=\"border: solid 1px #ccc;\">\n"
							+ "<tr><td style=\"width:620px; vertical-align:top; margin:10px;\">\n");
				}

				sb.append("<table border=\"0\" style=\"margin-bottom:10px;\">\n"
						+ "<tr style=\"background-color:#ddd;\"><td style=\"width:360px; vertical-align:top;\"><strong>"
						+ key.replace('_', ' ')
						+ ":</strong> </td><td style=\"width:80px; vertical-align:top;\"><strong>Min</strong></td><td style=\"width:80px; vertical-align:top;\"><strong>Mean</strong></td><td style=\"width:80px; vertical-align:top;\"><strong>Max</strong></td></tr>\n");
				for (MeasuringStatisticsEntry entry : stats.getStatistics().get(key)) {
					if (entry != null && entry.getNumberProcesses() > 0) {
						sb.append("<tr><td>" + entry.getNumberRequestedElements() + " elements per request ("
								+ entry.getNumberProcesses() + " measures)</td><td>"
								+ f.format(entry.getMinTime()) + " ms</td><td>"
								+ f.format(entry.getMeanTime()) + " ms</td><td>"
								+ f.format(entry.getMaxTime()) + " ms</td></tr>\n");
					}
				}
				sb.append("</table>\n");
			}
			sb.append("</td></tr></table>\n<br />\n");
		}

		sb.append("");

		return sb.toString();
	}

	@Override
	public boolean isProtected() {
		return true;
	}

	@Override
	public String[] getAllowedRoles() {
		return new String[] { UserRole.ADMINISTRATOR.toString(), UserRole.REGISTRATION_USER.toString(),
				UserRole.DEVELOPER.toString(), UserRole.MAINTAINER.toString(), UserRole.USER.toString() };
	}

	@Override
	public Category getCategory() {
		return Category.CA_INFO;
	}
}
