package de.fraunhofer.sit.c2x.pki.ca.module.webserver.servlets;

import java.io.BufferedReader;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStreamReader;
import java.io.OutputStream;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.TimeZone;

import javax.servlet.ServletException;
import javax.servlet.http.HttpServlet;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

import org.apache.commons.lang3.StringEscapeUtils;
import org.apache.log4j.Logger;
import org.eclipse.jetty.server.UserIdentity;

import com.google.inject.Inject;
import com.google.inject.name.Named;

import de.fraunhofer.sit.c2x.pki.ca.core.logging.InjectLogger;
import de.fraunhofer.sit.c2x.pki.ca.module.webserver.DBLoginService;
import de.fraunhofer.sit.c2x.pki.ca.module.webserver.interfaces.Servlet;
import de.fraunhofer.sit.c2x.pki.ca.provider.interfaces.ConfigProvider;

public class LogRawServlet extends HttpServlet implements Servlet {
	private static final long serialVersionUID = 1L;

	@InjectLogger
	private Logger logger;

	@Inject
	private ConfigProvider configProvider;

	@Inject
	@Named(value = "caName")
	private String caName;

	@Inject
	private DBLoginService loginService;

	public static final String URL = "/rawlog";
	public static final String DATE_PATTERN_LOG = "yyyy-MM-dd HH:mm:ss,SSS";

	@Override
	protected void doGet(HttpServletRequest req, HttpServletResponse response) throws ServletException,
			IOException {

		String periodValue = configProvider.get("webpageLogDefaultTimePeriod").replaceAll("[a-z]*", "");
		String periodUnit = configProvider.get("webpageLogDefaultTimePeriod").replaceAll("[0-9]*", "");
		String period = periodValue + "" + periodUnit;
		if (req.getParameter("period") != null && req.getParameter("period").length() > 0) {
			period = req.getParameter("period");
		} else if (req.getParameter("periodValue") != null && req.getParameter("periodValue").length() > 0
				&& req.getParameter("periodUnit") != null && req.getParameter("periodUnit").length() > 0) {
			periodValue = req.getParameter("periodValue");
			periodUnit = req.getParameter("periodUnit");
			period = periodValue + "" + periodUnit;
		}

		OutputStream out = response.getOutputStream();

		/* Send response header with Content-Length. */
		response.setStatus(200);

		// check whether raw output is requested
		if (req.getParameter("user") != null && req.getParameter("user").length() > 0
				&& req.getParameter("password") != null && req.getParameter("password").length() > 0) {

			// try to register session with given user and password
			UserIdentity userIdentity = loginService.login(req.getParameter("user"),
					req.getParameter("password"));
			boolean isUserInRole = false;
			for (String role : getAllowedRoles()) {
				if (userIdentity.isUserInRole(role, null))
					isUserInRole = true;
			}

			if (isUserInRole == false) {
				logger.error("User " + req.getParameter("user") + " not authenticated to read raw logs");
				// if given user and password is not authorized
				// then set HTML status to 401 = Unauthorized
				response.setStatus(401);
			} else {
				String logStr = "";
				TimeZone tz = TimeZone.getTimeZone(configProvider.get("webpageLogTimeZone"));
				try {
					logStr = outputLog(period, tz);
				} catch (IOException e) {
					logger.error(e);
					logStr += "<div id=\"errorbox\">Unable to read logfile</div>\n";
				}
				// write log file to output stream
				out.write(logStr.getBytes());
				/* Close. */
				out.flush();
				out.close();
				return;
			}
		}
	}

	protected String outputLog(String period, TimeZone tz) throws IOException {
		String str = "";
		if (period == null) {
			// set default period if nothing is given
			period = configProvider.get("webpageLogDefaultTimePeriod");
		}
		Date startPeriod = LogServlet.getStartDate(period, tz);

		FileInputStream fin = null;
		BufferedReader br = null;
		try {
			fin = new FileInputStream(configProvider.get("log4jFile"));
			br = new BufferedReader(new InputStreamReader(fin));
			String s = null;
			if (startPeriod == null) {
				while ((s = br.readLine()) != null) {
					if (s.contains("[" + caName))
						str += StringEscapeUtils.escapeHtml4(s) + "\n";
				}
			} else {
				boolean after = false;
				SimpleDateFormat df = new SimpleDateFormat(DATE_PATTERN_LOG);
				df.setTimeZone(tz);
				while ((s = br.readLine()) != null) {
					if ((!after)) {
						boolean dateAfter = false;
						if (s.length() > 23) {
							String date = s.substring(0, 23);
							try {
								dateAfter = df.parse(date).after(startPeriod);
							} catch (ParseException e) {
								System.err.println(e);
							}
						}
						if (!dateAfter)
							continue;
						after = true;
					}
					if (s.contains("[" + caName))
						str += StringEscapeUtils.escapeHtml4(s) + "\n";
				}
			}
			return str;
		} finally {
			if (fin != null)
				fin.close();
			if (br != null)
				br.close();
		}
	}

	@Override
	public String getUrl() {
		return URL;
	}

	@Override
	public String getTitle() {
		return "";
	}

	@Override
	public boolean isProtected() {
		return false;
	}

	@Override
	public String[] getAllowedRoles() {
		return new String[] { UserRole.ADMINISTRATOR.toString(), UserRole.REGISTRATION_USER.toString(),
				UserRole.DEVELOPER.toString(), UserRole.MAINTAINER.toString(), UserRole.USER.toString() };
	}

	@Override
	public Category getCategory() {
		// TODO Auto-generated method stub
		return null;
	}
}
