package de.fraunhofer.sit.c2x.pki.ca.module.udp;

import java.net.DatagramPacket;
import java.net.DatagramSocket;
import java.net.SocketException;

import org.apache.log4j.Logger;

import com.google.inject.Inject;
import com.google.inject.name.Named;

import de.fraunhofer.sit.c2x.pki.ca.core.interfaces.Service;
import de.fraunhofer.sit.c2x.pki.ca.core.interfaces.ThreadPool;
import de.fraunhofer.sit.c2x.pki.ca.core.logging.InjectLogger;

/**
 * @author Daniel Quanz (daniel.quanz@sit.fraunhofer.de)
 */
public class UdpServer extends Thread implements Service {

	@InjectLogger
	private Logger logger;

	private final ThreadPool threadPool;
	private DatagramSocket socket;

	private final Integer port;

	private final Integer maxPacketSize;

	private boolean shutdownSocket;

	private boolean isStarted;

	private final UdpPacketHandlerFactory udpPacketHandlerFactory;

	@Inject
	public UdpServer(ThreadPool threadPool,
			UdpPacketHandlerFactory udpPacketHandlerFactory,
			@Named(value = "udp_port") Integer port,
			@Named(value = "maxPacketSize") Integer maxPacketSize) {
		this.udpPacketHandlerFactory = udpPacketHandlerFactory;
		this.threadPool = threadPool;
		this.port = port;
		this.maxPacketSize = maxPacketSize;
	}

	@Override
	public void run() {
		try {
			this.socket = new DatagramSocket(port);
			logger.debug(getName() + " listens on "
					+ this.socket.getLocalAddress() + " port "
					+ this.socket.getLocalPort() + ". Waiting for packets");
			while (!this.socket.isClosed()) {
				byte[] buf = new byte[maxPacketSize];
				DatagramPacket packet = new DatagramPacket(buf, maxPacketSize);
				this.socket.receive(packet);
				logger.debug(getName() + " received new packet (size: "
						+ packet.getLength() + " bytes) from "
						+ packet.getSocketAddress());

				// create packet handler in a new thread
				if (!this.threadPool.get().isShutdown()) {
					this.threadPool.get().execute(
							udpPacketHandlerFactory.create(socket, packet));
				}
			}
		} catch (SocketException e) {
			if (this.shutdownSocket) {
				logger.info("Server socket closed at port " + port);
			} else {
				logger.error("Server socket closed at port " + port, e);
			}
		} catch (Exception e) {
			logger.error(e.getMessage(), e);
		} finally {
			this.close();
		}
	}

	public void close() {
		this.shutdownSocket = true;
		if (this.socket != null) {
			this.socket.close();
		}
	}

	/**
	 * Closes the job queue, processes the remaining queued jobs and then closes
	 * the communication Socket.
	 */
	public void closeAndWait() {
		this.shutdownSocket = true;
		if (this.socket != null) {
			this.socket.close();
		}
	}

	// /**
	// * Called if the {@link #jobQueue} is full
	// */
	// @Override
	// public void rejectedExecution(Runnable r, ThreadPoolExecutor executor) {
	// Abstract1609Dot2PacketHandler sph = (Abstract1609Dot2PacketHandler) r;
	// byte[] error = sph.generateErrorMessage(null, "jobQueue is full");
	// sph.respondMessage(error);
	// }

	@Override
	public synchronized void startService() {

		start();
		isStarted = true;
		logger.info("Datagram socket server started");
	}

	@Override
	public synchronized void stopService() {
		close();

	}

	@Override
	public boolean isStarted() {
		return isStarted;
	}
}
