package de.fraunhofer.sit.c2x.pki.etsi_ts103097v1114.utils;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.DataInputStream;
import java.io.DataOutputStream;
import java.io.IOException;
import java.nio.charset.Charset;
import java.util.ArrayList;

import org.apache.commons.codec.DecoderException;
import org.apache.commons.codec.binary.Hex;
import org.apache.log4j.Logger;
import org.bouncycastle.crypto.AsymmetricCipherKeyPair;
import org.bouncycastle.crypto.params.ECPrivateKeyParameters;

import com.google.inject.Inject;

import de.fraunhofer.sit.c2x.pki.ca.certificates.CertificateBuilder;
import de.fraunhofer.sit.c2x.pki.ca.certificates.RootCertificateResult;
import de.fraunhofer.sit.c2x.pki.ca.certificates.datacontainers.GeographicRegionDataContainer;
import de.fraunhofer.sit.c2x.pki.ca.certificates.datacontainers.PsidSspPriorityDataContainer;
import de.fraunhofer.sit.c2x.pki.ca.core.exceptions.HandlerException;
import de.fraunhofer.sit.c2x.pki.ca.core.logging.InjectLogger;
import de.fraunhofer.sit.c2x.pki.ca.crypto.KeyGenerator;
import de.fraunhofer.sit.c2x.pki.ca.provider.ProviderException;
import de.fraunhofer.sit.c2x.pki.ca.provider.entities.CaCertificate;
import de.fraunhofer.sit.c2x.pki.ca.utils.WaveUtils;
import de.fraunhofer.sit.c2x.pki.ca.validator.region.GeorgraphicRegionValidator;
import de.fraunhofer.sit.c2x.pki.etsi_ts103097v1114.handler.adapter.CertificateAdapter;
import de.fraunhofer.sit.c2x.pki.etsi_ts103097v1114.impl.Certificate;
import de.fraunhofer.sit.c2x.pki.etsi_ts103097v1114.impl.EccPoint;
import de.fraunhofer.sit.c2x.pki.etsi_ts103097v1114.impl.EccPointTypeImpl.EccPointType;
import de.fraunhofer.sit.c2x.pki.etsi_ts103097v1114.impl.EcdsaSignature;
import de.fraunhofer.sit.c2x.pki.etsi_ts103097v1114.impl.GeographicRegion;
import de.fraunhofer.sit.c2x.pki.etsi_ts103097v1114.impl.HashedId8;
import de.fraunhofer.sit.c2x.pki.etsi_ts103097v1114.impl.Opaque;
import de.fraunhofer.sit.c2x.pki.etsi_ts103097v1114.impl.PublicKey;
import de.fraunhofer.sit.c2x.pki.etsi_ts103097v1114.impl.PublicKeyAlgorithmImpl.PublicKeyAlgorithm;
import de.fraunhofer.sit.c2x.pki.etsi_ts103097v1114.impl.RegionTypeImpl.RegionType;
import de.fraunhofer.sit.c2x.pki.etsi_ts103097v1114.impl.Signature;
import de.fraunhofer.sit.c2x.pki.etsi_ts103097v1114.impl.SignerInfo;
import de.fraunhofer.sit.c2x.pki.etsi_ts103097v1114.impl.SignerInfoTypeImpl.SignerInfoType;
import de.fraunhofer.sit.c2x.pki.etsi_ts103097v1114.impl.SubjectAssurance;
import de.fraunhofer.sit.c2x.pki.etsi_ts103097v1114.impl.SubjectAttribute;
import de.fraunhofer.sit.c2x.pki.etsi_ts103097v1114.impl.SubjectAttributeTypeImpl.SubjectAttributeType;
import de.fraunhofer.sit.c2x.pki.etsi_ts103097v1114.impl.SubjectInfo;
import de.fraunhofer.sit.c2x.pki.etsi_ts103097v1114.impl.SubjectTypeImpl.SubjectType;
import de.fraunhofer.sit.c2x.pki.etsi_ts103097v1114.impl.SymmetricAlgorithmImpl.SymmAlgorithm;
import de.fraunhofer.sit.c2x.pki.etsi_ts103097v1114.impl.Time32;
import de.fraunhofer.sit.c2x.pki.etsi_ts103097v1114.impl.ValidityRestriction;
import de.fraunhofer.sit.c2x.pki.etsi_ts103097v1114.impl.ValidityRestrictionTypeImpl.ValidityRestrictionType;
import de.fraunhofer.sit.c2x.pki.etsi_ts103097v1114.validator.time.ExpirationTimeValidator;

public class EtsiCertificateBuilderInvalidContent implements CertificateBuilder {

	@InjectLogger
	private Logger logger;

	@Inject
	private GeorgraphicRegionValidator regionValidator;

	private EtsiPKCS11Utils etsiPKCS11Utils = new EtsiPKCS11Utils();

	// test extensions for HSM:
	// 1. LTCA ok
	// 2. PCA ok
	// 3. ROOT CA Probleme aufgrund "testverification" -nope
	// 4. CRL Signer ok

	// TODO TEST sign methods!
	// 1. LTCA
	// 2. PCA
	// 3. ROOTCA - Problem, hier wird Private key als signingkeypair
	// zurckgegeben
	// 4. CRL Signer

	@Override
	public de.fraunhofer.sit.c2x.pki.ca.provider.interfaces.Certificate createNewLongTermCaCertificate(
			String signingKeyX, String signingKeyY, String encryptionKeyX, String encryptionKeyY,
			String version, String algorithm, String symmetricAlgorithm, String compression,
			String startDate, String endDate, ArrayList<PsidSspPriorityDataContainer> psidSet,
			ECPrivateKeyParameters rootPrivateKey, byte[] rootCertificate, boolean addRootCertDigestAsSigner,
			String assuranceLevel, String subjectName, ArrayList<GeographicRegionDataContainer> regions)
			throws DecoderException, IOException, ProviderException, HandlerException {

		return createNewLongTermCaCertificate(signingKeyX, signingKeyY, encryptionKeyX, encryptionKeyY,
				version, algorithm, symmetricAlgorithm, compression, startDate, endDate, psidSet,
				rootPrivateKey, rootCertificate, addRootCertDigestAsSigner, assuranceLevel, subjectName,
				regions, false, null, null, null);
	}

	public de.fraunhofer.sit.c2x.pki.ca.provider.interfaces.Certificate createNewLongTermCaCertificate(
			String signingKeyX, String signingKeyY, String encryptionKeyX, String encryptionKeyY,
			String version, String algorithm, String symmetricAlgorithm, String compression,
			String startDate, String endDate, ArrayList<PsidSspPriorityDataContainer> psidSet,
			ECPrivateKeyParameters rootPrivateKey, byte[] rootCertificate, boolean addRootCertDigestAsSigner,
			String assuranceLevel, String subjectName, ArrayList<GeographicRegionDataContainer> regions,
			boolean useHSM, char[] hsmLogin, String alias, String dllPath) throws DecoderException,
			IOException, ProviderException, NumberFormatException, HandlerException {

		Certificate cert = new Certificate();

		// SubjectInfo
		SubjectInfo subjectInfo = new SubjectInfo();
		subjectInfo.setSubjectType(SubjectType.ENROLLMENT_AUTHORITY);
		Opaque subjectNameOpaque = new Opaque(subjectName.getBytes(Charset.forName("UTF-8")));
		subjectInfo.setSubjectName(subjectNameOpaque);
		cert.setSubjectInfo(subjectInfo);
		// check
		if (signingKeyX == null || signingKeyY == null || encryptionKeyX == null || encryptionKeyY == null)
			return null;

		SymmAlgorithm symmAlgorithm = SymmAlgorithm.AES_128_CCM;

		Opaque xv = new Opaque();
		xv.set(Hex.decodeHex(signingKeyX.toCharArray()));
		xv.setArray(true);

		Opaque yv = new Opaque();
		yv.set(Hex.decodeHex(signingKeyY.toCharArray()));
		yv.setArray(true);

		Opaque xe = new Opaque();
		xe.set(Hex.decodeHex(encryptionKeyX.toCharArray()));
		xe.setArray(true);

		Opaque ye = new Opaque();
		ye.set(Hex.decodeHex(encryptionKeyY.toCharArray()));
		ye.setArray(true);

		// check
		if (xv.get().length != 32 || yv.get().length != 32 || xe.get().length != 32 || ye.get().length != 32)
			return null;

		EccPoint pkvecc = new EccPoint();
		pkvecc.setType(EccPointType.UNCOMPRESSED);
		pkvecc.setX(xv);
		pkvecc.setY(yv);

		EccPoint pkeecc = new EccPoint();
		pkeecc.setType(EccPointType.UNCOMPRESSED);
		pkeecc.setX(xe);
		pkeecc.setY(ye);

		PublicKey pkv = new PublicKey();
		pkv.setAlgorithm(PublicKeyAlgorithm.ECDSA_NISTP256_WITH_SHA256);
		pkv.setPublicKey(pkvecc);

		PublicKey pke = new PublicKey();
		pke.setAlgorithm(PublicKeyAlgorithm.ECIES_NISTP256);
		pke.setPublicKey(pkeecc);
		pke.setSupportedSymmAlg(symmAlgorithm);

		// Subject Attributes
		ArrayList<SubjectAttribute> subjectAttributeList = new ArrayList<>();
		// Assurance Level
		if (assuranceLevel != null && !assuranceLevel.isEmpty()) {
			byte[] assuranceBytes = EtsiPerrmissionUtils
					.subjectAssuranceInttoByte(new Integer(assuranceLevel));
			SubjectAssurance sa = new SubjectAssurance(assuranceBytes);
			SubjectAttribute s = new SubjectAttribute(sa);
			subjectAttributeList.add(s);
		}
		// AIDs
		ArrayList<SubjectAttribute> psidSubjectAttributeLists = EtsiPerrmissionUtils
				.psidSetToSubjectAttributeArray(psidSet);
		boolean hasAid = false;
		for (SubjectAttribute sa : psidSubjectAttributeLists) {
			if (sa.getType() == SubjectAttributeType.ITS_AID_SSP_LIST
					|| sa.getType() == SubjectAttributeType.PRIORITY_ITS_AID_LIST
					|| sa.getType() == SubjectAttributeType.PRIORITY_SSP_LIST)
				throw new HandlerException("Invalid subject attribute type: " + sa.getType());
			else if (sa.getType() == SubjectAttributeType.ITS_AID_LIST)
				hasAid = true;
			subjectAttributeList.add(sa);
		}
		if (hasAid == false)
			throw new HandlerException("No AID given");

		// Verification Key
		subjectAttributeList.add(new SubjectAttribute(pkv, SubjectAttributeType.VERIFICATION_KEY));
		// Encryption Key
		subjectAttributeList.add(new SubjectAttribute(pke, SubjectAttributeType.ENCRYPTION_KEY));

		// add subject attributes
		SubjectAttribute[] subjectAttributes = new SubjectAttribute[subjectAttributeList.size()];
		for (int i = 0; i < subjectAttributeList.size(); i++) {
			subjectAttributes[i] = subjectAttributeList.get(i);
		}
		cert.setSubjectAttributes(subjectAttributes);

		Time32 startValidity;
		Time32 endValidity;
		try {
			startValidity = new Time32(startDate);
			endValidity = new Time32(endDate);
		} catch (Exception e) {
			return null;
		}

		if (startValidity == null || endValidity == null)
			return null;

		ValidityRestriction[] validityRestrictions = new ValidityRestriction[2];
		validityRestrictions[0] = new ValidityRestriction();
		validityRestrictions[0].setType(ValidityRestrictionType.TIME_START_AND_END);
		validityRestrictions[0].setStartValidity(startValidity);
		validityRestrictions[0].setEndValidity(endValidity);

		ValidityRestriction region = new ValidityRestriction(new GeographicRegion(RegionType.NONE));
		validityRestrictions[1] = region;

		cert.setValidityRestrictions(validityRestrictions);

		// Signer Info
		Certificate rootCert = WaveUtils.getElementFromBytes(rootCertificate, Certificate.class);
		if (addRootCertDigestAsSigner) {
			cert.setSignerInfo(new SignerInfo(rootCert.getHashedId8()));
		} else {
			cert.setSignerInfo(new SignerInfo(rootCert));
		}

		Signature signature = new Signature();
		signature.setAlgorithm(PublicKeyAlgorithm.ECDSA_NISTP256_WITH_SHA256);
		EcdsaSignature ecdsaSignature = new EcdsaSignature();
		Signature sig = new Signature();

		if (!useHSM) {
			sig = EtsiCryptoUtils.sign(cert, PublicKeyAlgorithm.ECDSA_NISTP256_WITH_SHA256, rootPrivateKey);
		} else {
			sig = etsiPKCS11Utils.signEtsiSignature(cert, "Sha256withECDSA", alias, hsmLogin, dllPath);
		}

		ecdsaSignature.setR(sig.getEcdsaSignature().getR());
		ecdsaSignature.setS(sig.getEcdsaSignature().getS());
		signature.setEcdsaSignature(ecdsaSignature);
		cert.setSignature(signature);

		// verify signature of newly created certificate
		if (addRootCertDigestAsSigner) {
			if (cert.verfiySignature(rootCert) == false)
				throw new HandlerException("Signature of certificate invalid");
		} else {
			if (cert.verfiySignature() == false)
				throw new HandlerException("Signature of certificate invalid");
		}

		CaCertificate caCert = new CaCertificate();

		ByteArrayOutputStream certOut = new ByteArrayOutputStream();
		cert.writeData(new DataOutputStream(certOut));
		caCert.setCertificate(certOut.toByteArray());
		certOut.close();

		logger.info("Certificate generated: " + new String(Hex.encodeHex(caCert.getCertificate())));

		return caCert;

	}

	@Override
	public de.fraunhofer.sit.c2x.pki.ca.provider.interfaces.Certificate createNewPseudonymCaCertificate(
			String signingKeyX, String signingKeyY, String encryptionKeyX, String encryptionKeyY,
			String version, String algorithm, String symmetricAlgorithm, String compression,
			String startDate, String endDate, ArrayList<PsidSspPriorityDataContainer> psidSet,
			ECPrivateKeyParameters rootPrivateKey, byte[] rootCertificate, boolean addRootCertDigestAsSigner,
			String assuranceLevel, String subjectName, ArrayList<GeographicRegionDataContainer> regions)
			throws DecoderException, IOException, ProviderException, HandlerException {

		return createNewPseudonymCaCertificate(signingKeyX, signingKeyY, encryptionKeyX, encryptionKeyY,
				version, algorithm, symmetricAlgorithm, compression, startDate, endDate, psidSet,
				rootPrivateKey, rootCertificate, addRootCertDigestAsSigner, assuranceLevel, subjectName,
				regions, false, null, null, null);

	}

	public de.fraunhofer.sit.c2x.pki.ca.provider.interfaces.Certificate createNewPseudonymCaCertificate(
			String signingKeyX, String signingKeyY, String encryptionKeyX, String encryptionKeyY,
			String version, String algorithm, String symmetricAlgorithm, String compression,
			String startDate, String endDate, ArrayList<PsidSspPriorityDataContainer> psidSet,
			ECPrivateKeyParameters rootPrivateKey, byte[] rootCertificate, boolean addRootCertDigestAsSigner,
			String assuranceLevel, String subjectName, ArrayList<GeographicRegionDataContainer> regions,
			boolean useHSM, char[] hsmLogin, String alias, String dllPath) throws DecoderException,
			IOException, ProviderException, NumberFormatException, HandlerException {

		Certificate cert = new Certificate();

		// SubjectInfo
		SubjectInfo subjectInfo = new SubjectInfo();
		subjectInfo.setSubjectType(SubjectType.AUTHORIZATION_AUTHORITY);
		Opaque subjectNameOpaque = new Opaque(subjectName.getBytes(Charset.forName("UTF-8")));
		subjectInfo.setSubjectName(subjectNameOpaque);
		cert.setSubjectInfo(subjectInfo);

		// Signer Info
		Certificate rootCert = WaveUtils.getElementFromBytes(rootCertificate, Certificate.class);
		if (addRootCertDigestAsSigner) {
			cert.setSignerInfo(new SignerInfo(rootCert.getHashedId8()));
		} else {
			throw new HandlerException(
					"Option to add certificate as signer not allowed in ETSI TS 103 097 v1.1.13");
			// signerInfo[0] = new SignerInfo(rootCert);
		}

		// check
		if (signingKeyX == null || signingKeyY == null || encryptionKeyX == null || encryptionKeyY == null)
			return null;

		SymmAlgorithm symmAlgorithm = SymmAlgorithm.AES_128_CCM;

		Opaque xv = new Opaque();
		xv.set(Hex.decodeHex(signingKeyX.toCharArray()));
		xv.setArray(true);

		Opaque yv = new Opaque();
		yv.set(Hex.decodeHex(signingKeyY.toCharArray()));
		yv.setArray(true);

		Opaque xe = new Opaque();
		xe.set(Hex.decodeHex(encryptionKeyX.toCharArray()));
		xe.setArray(true);

		Opaque ye = new Opaque();
		ye.set(Hex.decodeHex(encryptionKeyY.toCharArray()));
		ye.setArray(true);

		// check
		if (xv.get().length != 32 || yv.get().length != 32 || xe.get().length != 32 || ye.get().length != 32)
			return null;

		EccPoint pkvecc = new EccPoint();
		pkvecc.setType(EccPointType.UNCOMPRESSED);
		pkvecc.setX(xv);
		pkvecc.setY(yv);

		EccPoint pkeecc = new EccPoint();
		pkeecc.setType(EccPointType.UNCOMPRESSED);
		pkeecc.setX(xe);
		pkeecc.setY(ye);

		PublicKey pkv = new PublicKey();
		pkv.setAlgorithm(PublicKeyAlgorithm.ECDSA_NISTP256_WITH_SHA256);
		pkv.setPublicKey(pkvecc);

		PublicKey pke = new PublicKey();
		pke.setAlgorithm(PublicKeyAlgorithm.ECIES_NISTP256);
		pke.setPublicKey(pkeecc);
		pke.setSupportedSymmAlg(symmAlgorithm);

		// Subject Attributes
		ArrayList<SubjectAttribute> subjectAttributeList = new ArrayList<>();
		// Assurance Level
		if (assuranceLevel != null && !assuranceLevel.isEmpty()) {
			byte[] assuranceBytes = EtsiPerrmissionUtils
					.subjectAssuranceInttoByte(new Integer(assuranceLevel));
			SubjectAssurance sa = new SubjectAssurance(assuranceBytes);
			SubjectAttribute s = new SubjectAttribute(sa);
			subjectAttributeList.add(s);
		}
		// AIDs
		ArrayList<SubjectAttribute> psidSubjectAttributeLists = EtsiPerrmissionUtils
				.psidSetToSubjectAttributeArray(psidSet);
		boolean hasAid = false;
		for (SubjectAttribute sa : psidSubjectAttributeLists) {
			if (sa.getType() == SubjectAttributeType.ITS_AID_SSP_LIST
					|| sa.getType() == SubjectAttributeType.PRIORITY_ITS_AID_LIST
					|| sa.getType() == SubjectAttributeType.PRIORITY_SSP_LIST)
				throw new HandlerException("Invalid subject attribute type: " + sa.getType());
			else if (sa.getType() == SubjectAttributeType.ITS_AID_LIST)
				hasAid = true;
			subjectAttributeList.add(sa);
		}
		if (hasAid == false)
			throw new HandlerException("No AID given");

		// Verification Key
		subjectAttributeList.add(new SubjectAttribute(pkv, SubjectAttributeType.VERIFICATION_KEY));
		// Encryption Key
		subjectAttributeList.add(new SubjectAttribute(pke, SubjectAttributeType.ENCRYPTION_KEY));

		// add subject attributes
		SubjectAttribute[] subjectAttributes = new SubjectAttribute[subjectAttributeList.size()];
		for (int i = 0; i < subjectAttributeList.size(); i++) {
			subjectAttributes[i] = subjectAttributeList.get(i);
		}
		cert.setSubjectAttributes(subjectAttributes);

		// Validity Restrictions
		// -> Date Restrictions

		Time32 startValidity;
		Time32 endValidity;
		try {
			startValidity = new Time32(startDate);
			endValidity = new Time32(endDate);
		} catch (Exception e) {
			return null;
		}

		if (startValidity == null || endValidity == null)
			return null;

		ValidityRestriction[] validityRestrictions = new ValidityRestriction[2];
		validityRestrictions[0] = new ValidityRestriction();
		validityRestrictions[0].setType(ValidityRestrictionType.TIME_START_AND_END);
		validityRestrictions[0].setStartValidity(startValidity);
		validityRestrictions[0].setEndValidity(endValidity);

		// check validity time
		ExpirationTimeValidator expirationTimeValidator = new ExpirationTimeValidator();
		expirationTimeValidator.validate(validityRestrictions);
		if (expirationTimeValidator.validate(validityRestrictions) == false)
			throw new HandlerException("Expiration time of PCA certificate not valid. Start time = "
					+ startValidity + "; End time = " + endValidity);

		// set region restriction
		if (regions == null || regions.size() == 0) {
			ValidityRestriction region = new ValidityRestriction(new GeographicRegion(RegionType.NONE));
			validityRestrictions[1] = region;
		} else {
			if (regions.size() > 1)
				throw new HandlerException("More than one region restriction not supported.");

			GeographicRegion requestedRegion = EtsiRegionUtils.regionContainerToGeographicRegion(regions
					.get(0));

			// do not check region restriction
			ValidityRestriction region = new ValidityRestriction(requestedRegion);
			validityRestrictions[1] = region;
		}

		cert.setValidityRestrictions(validityRestrictions);

		// create signrature
		Signature signature = new Signature();
		signature.setAlgorithm(PublicKeyAlgorithm.ECDSA_NISTP256_WITH_SHA256);
		EcdsaSignature ecdsaSignature = new EcdsaSignature();

		Signature sig = new Signature();
		if (useHSM) {
			sig = etsiPKCS11Utils.signEtsiSignature(cert, "Sha256withECDSA", alias, hsmLogin, dllPath);
		} else {
			sig = EtsiCryptoUtils.sign(cert, PublicKeyAlgorithm.ECDSA_NISTP256_WITH_SHA256, rootPrivateKey);

		}
		EccPoint r = sig.getEcdsaSignature().getR();
		ecdsaSignature.setR(r);
		Opaque s = sig.getEcdsaSignature().getS();
		ecdsaSignature.setS(s);
		signature.setEcdsaSignature(ecdsaSignature);
		cert.setSignature(signature);

		// verify signature of newly created certificate
		if (cert.verfiySignature(rootCert) == false)
			throw new HandlerException("Signature of certificate invalid");

		CaCertificate caCert = new CaCertificate();

		ByteArrayOutputStream certOut = new ByteArrayOutputStream();
		cert.writeData(new DataOutputStream(certOut));
		caCert.setCertificate(certOut.toByteArray());
		certOut.close();

		return caCert;
	}

	@Override
	public RootCertificateResult createNewRootCertificate(String startDate, String endDate,
			String assuranceLevel, String subjectName, ArrayList<GeographicRegionDataContainer> regions)
			throws IOException, HandlerException {
		return createNewRootCertificate(startDate, endDate, assuranceLevel, subjectName, regions, false,
				null, null, null);
	}

	@Override
	public RootCertificateResult createNewRootCertificate(String startDate, String endDate,
			String assuranceLevel, String subjectName, ArrayList<GeographicRegionDataContainer> regions,
			boolean useHsm, char[] hsmLogin, String alias, String dllPath) throws IOException,
			HandlerException {
		return createNewRootCertificate(startDate, endDate, assuranceLevel, subjectName, null, regions,
				false, null, null, null);
	}

	public RootCertificateResult createNewRootCertificate(String startDate, String endDate,
			String assuranceLevel, String subjectName, ArrayList<PsidSspPriorityDataContainer> psidSet,
			ArrayList<GeographicRegionDataContainer> regions) throws IOException, NumberFormatException,
			HandlerException {

		return createNewRootCertificate(startDate, endDate, assuranceLevel, subjectName, psidSet, regions,
				false, null, null, null);
	}

	public RootCertificateResult createNewRootCertificate(String startDate, String endDate,
			String assuranceLevel, String subjectName, ArrayList<PsidSspPriorityDataContainer> psidSet,
			ArrayList<GeographicRegionDataContainer> regions, boolean useHsm, char[] hsmLogin, String alias,
			String dllPath) throws IOException, NumberFormatException, HandlerException {

		Certificate cert = new Certificate();

		// SubjectInfo
		SubjectInfo subjectInfo = new SubjectInfo();
		subjectInfo.setSubjectType(SubjectType.ROOT_CA);
		Opaque subjectNameOpaque = new Opaque(subjectName.getBytes(Charset.forName("UTF-8")));
		subjectInfo.setSubjectName(subjectNameOpaque);
		cert.setSubjectInfo(subjectInfo);

		// Create new (uncompressed) signing and encryption key pairs
		EccPoint pkvecc;
		AsymmetricCipherKeyPair signingKeyPair = null;
		if (!useHsm) {
			signingKeyPair = KeyGenerator.getInstance().generateECKeyPair("P-256");
			pkvecc = new EccPoint(signingKeyPair.getPublic());

		} else {
			pkvecc = new EccPoint(etsiPKCS11Utils.convertKey(etsiPKCS11Utils.getPublicPKCS11(alias, dllPath,
					hsmLogin)));
		}

		PublicKey pkv = new PublicKey();
		pkv.setAlgorithm(PublicKeyAlgorithm.ECDSA_NISTP256_WITH_SHA256);
		pkv.setPublicKey(pkvecc);

		// also for hsm use, only signing pk is stored on HSM
		AsymmetricCipherKeyPair encryptionKeyPair = KeyGenerator.getInstance().generateECKeyPair("P-256");
		EccPoint pkeecc = new EccPoint(encryptionKeyPair.getPublic());
		PublicKey pke = new PublicKey();
		pke.setPublicKey(pkeecc);

		SymmAlgorithm symmAlgorithm = SymmAlgorithm.AES_128_CCM;
		pke.setAlgorithm(PublicKeyAlgorithm.ECIES_NISTP256);
		pke.setSupportedSymmAlg(symmAlgorithm);

		ArrayList<SubjectAttribute> subjectAttributeList = new ArrayList<>();
		// Assurance Level
		if (assuranceLevel != null && !assuranceLevel.isEmpty()) {
			byte[] assuranceBytes = EtsiPerrmissionUtils
					.subjectAssuranceInttoByte(new Integer(assuranceLevel));
			SubjectAssurance sa = new SubjectAssurance(assuranceBytes);
			SubjectAttribute s = new SubjectAttribute(sa);
			subjectAttributeList.add(s);
		}
		// add AIDs to subject attribute list
		if (psidSet == null || psidSet.size() == 0)
			throw new HandlerException("No AIDs provided. Please add list of AIDs for root certificate.");
		ArrayList<SubjectAttribute> psidSubjectAttributeLists = EtsiPerrmissionUtils
				.psidSetToSubjectAttributeArray(psidSet);
		boolean hasAid = false;
		for (SubjectAttribute sa : psidSubjectAttributeLists) {
			if (sa.getType() == SubjectAttributeType.ITS_AID_SSP_LIST
					|| sa.getType() == SubjectAttributeType.PRIORITY_ITS_AID_LIST
					|| sa.getType() == SubjectAttributeType.PRIORITY_SSP_LIST)
				throw new HandlerException("Invalid subject attribute type: " + sa.getType());
			else if (sa.getType() == SubjectAttributeType.ITS_AID_LIST)
				hasAid = true;
			subjectAttributeList.add(sa);
		}
		if (hasAid == false)
			throw new HandlerException("No AID given");

		subjectAttributeList.add(new SubjectAttribute(pkv, SubjectAttributeType.VERIFICATION_KEY));
		if (!useHsm) {
			if (testVerification(signingKeyPair) == false)
				throw new HandlerException(
						"Generated verification key pair is not applicable to sign and verify");
		} else {
			if (etsiPKCS11Utils.testVerification(alias, hsmLogin, dllPath) == false) {
				throw new HandlerException("Stored keys on HSM for alias \"" + alias
						+ "\" verification key pair is not applicable to sign and verify");
			}

		}

		// Encryption Key
		subjectAttributeList.add(new SubjectAttribute(pke, SubjectAttributeType.ENCRYPTION_KEY));
		if (testEncrytion(encryptionKeyPair) == false)
			throw new HandlerException(
					"Generated encryption key pair is not applicable to encrypt and decrpyt");

		// add subject attributes
		SubjectAttribute[] subjectAttributes = new SubjectAttribute[subjectAttributeList.size()];
		for (int i = 0; i < subjectAttributeList.size(); i++) {
			subjectAttributes[i] = subjectAttributeList.get(i);
		}
		// pkv and pke added here
		cert.setSubjectAttributes(subjectAttributes);

		// Validity Restrictions
		// -> Date Restrictions

		Time32 startValidity;
		Time32 endValidity;
		try {
			startValidity = new Time32(startDate);
			endValidity = new Time32(endDate);
		} catch (Exception e) {
			return null;
		}

		if (startValidity == null || endValidity == null)
			return null;

		ValidityRestriction[] validityRestrictions = new ValidityRestriction[2];
		validityRestrictions[0] = new ValidityRestriction();
		validityRestrictions[0].setType(ValidityRestrictionType.TIME_START_AND_END);
		validityRestrictions[0].setStartValidity(startValidity);
		validityRestrictions[0].setEndValidity(endValidity);

		// set region restriction
		if (regions == null || regions.size() == 0) {
			ValidityRestriction region = new ValidityRestriction(new GeographicRegion(RegionType.NONE));
			validityRestrictions[1] = region;
		} else {
			if (regions.size() > 1)
				throw new HandlerException("More than one region restriction not supported.");

			GeographicRegion requestedRegion = EtsiRegionUtils.regionContainerToGeographicRegion(regions
					.get(0));

			// check region restriction
			boolean result = EtsiPerrmissionUtils.checkRegionRestrictions(requestedRegion, requestedRegion,
					regionValidator);
			if (result == false)
				throw new HandlerException(
						"Issuer has region restriction that does not allow to issue the requested certificate.");

			ValidityRestriction region = new ValidityRestriction(requestedRegion);
			validityRestrictions[1] = region;
		}

		cert.setValidityRestrictions(validityRestrictions);

		// check validity time
		ExpirationTimeValidator expirationTimeValidator = new ExpirationTimeValidator();
		if (expirationTimeValidator.validate(validityRestrictions) == false)
			throw new HandlerException("Expiration time of root certificate not valid. Start time = "
					+ startValidity + "; End time = " + endValidity);

		// Signer Info
		SignerInfo signerInfo = new SignerInfo();
		signerInfo.setType(SignerInfoType.SELF);
		cert.setSignerInfo(signerInfo);

		// Signature

		Signature signature = new Signature();
		signature.setAlgorithm(PublicKeyAlgorithm.ECDSA_NISTP256_WITH_SHA256);
		EcdsaSignature ecdsaSignature = new EcdsaSignature();

		Signature sig = new Signature();
		if (!useHsm) {
			sig = EtsiCryptoUtils.sign(cert, PublicKeyAlgorithm.ECDSA_NISTP256_WITH_SHA256,
					signingKeyPair.getPrivate());

		} else {
			sig = etsiPKCS11Utils.signEtsiSignature(cert, "Sha256withECDSA", alias, hsmLogin, dllPath);
		}

		ecdsaSignature.setR(sig.getEcdsaSignature().getR());
		ecdsaSignature.setS(sig.getEcdsaSignature().getS());
		signature.setEcdsaSignature(ecdsaSignature);
		cert.setSignature(signature);

		// verify signature of newly created certificate
		if (cert.verfiySignature() == false)
			throw new HandlerException("Signature of certificate invalid");

		CaCertificate caCert = new CaCertificate();
		ByteArrayOutputStream certOut = new ByteArrayOutputStream();
		cert.writeData(new DataOutputStream(certOut));
		byte[] certbyte = certOut.toByteArray();
		caCert.setCertificate(certbyte);
		certOut.close();

		return new RootCertificateResult(caCert, signingKeyPair, encryptionKeyPair);
	}

	@Override
	public String getCertificateAsHexString(de.fraunhofer.sit.c2x.pki.ca.provider.interfaces.Certificate cert)
			throws IOException {

		if (cert == null || cert.getCertificate() == null)
			return "";

		return Hex.encodeHexString(cert.getCertificate());
	}

	private boolean testVerification(AsymmetricCipherKeyPair signingKeyPair) {
		byte[] messageToSign = "This is a small test".getBytes();
		Signature signature = EtsiCryptoUtils.sign(messageToSign,
				PublicKeyAlgorithm.ECDSA_NISTP256_WITH_SHA256, signingKeyPair.getPrivate());
		try {
			return EtsiCryptoUtils.verify(messageToSign, signature, signingKeyPair.getPublic());
		} catch (Exception e) {
			return false;
		}
	}

	private boolean testEncrytion(AsymmetricCipherKeyPair encryptionKeyPair) {
		// FIXME
		// EtsiCryptoUtils.encryptAES(symmKey, nonce, messageToEncrypt)
		return true;
	}

	@Override
	public de.fraunhofer.sit.c2x.pki.ca.provider.interfaces.Certificate readCertificateFromBytes(
			byte[] certificate) throws IOException, ProviderException {
		ByteArrayInputStream bin = new ByteArrayInputStream(certificate);
		DataInputStream din = new DataInputStream(bin);
		Certificate cert = new Certificate(din);

		return new CertificateAdapter(cert);
	}

	@Override
	public de.fraunhofer.sit.c2x.pki.ca.provider.interfaces.Certificate createNewCrlSigner(
			String signingKeyX, String signingKeyY, String version, String algorithm, String compression,
			String startDate, String endDate, ECPrivateKeyParameters rootPrivateKey, byte[] rootCertificate,
			boolean addRootCertDigestAsSigner, String assuranceLevel, String subjectName)
			throws DecoderException, IOException, ProviderException, NumberFormatException, HandlerException {

		return createNewCrlSigner(signingKeyX, signingKeyY, version, algorithm, compression, startDate,
				endDate, rootPrivateKey, rootCertificate, addRootCertDigestAsSigner, assuranceLevel,
				subjectName, false, null, null, null);

	}

	public de.fraunhofer.sit.c2x.pki.ca.provider.interfaces.Certificate createNewCrlSigner(
			String signingKeyX, String signingKeyY, String version, String algorithm, String compression,
			String startDate, String endDate, ECPrivateKeyParameters rootPrivateKey, byte[] rootCertificate,
			boolean addRootCertDigestAsSigner, String assuranceLevel, String subjectName, boolean useHsm,
			char[] hsmLogin, String alias, String dllPath) throws DecoderException, IOException,
			ProviderException, NumberFormatException, HandlerException {

		Certificate cert = new Certificate();

		// Signer Info
		Certificate rootCert = WaveUtils.getElementFromBytes(rootCertificate, Certificate.class);
		if (addRootCertDigestAsSigner) {
			cert.setSignerInfo(new SignerInfo(rootCert.getHashedId8()));
		} else {
			cert.setSignerInfo(new SignerInfo(rootCert));
		}

		// SubjectInfo
		SubjectInfo subjectInfo = new SubjectInfo();
		subjectInfo.setSubjectType(SubjectType.CRL_SIGNER);
		Opaque subjectNameOpaque = new Opaque(subjectName.getBytes(Charset.forName("UTF-8")));
		subjectInfo.setSubjectName(subjectNameOpaque);
		cert.setSubjectInfo(subjectInfo);

		// check
		if (signingKeyX == null || signingKeyY == null)
			return null;

		Opaque xv = new Opaque();
		xv.set(Hex.decodeHex(signingKeyX.toCharArray()));
		xv.setArray(true);

		Opaque yv = new Opaque();
		yv.set(Hex.decodeHex(signingKeyY.toCharArray()));
		yv.setArray(true);

		// check
		if (xv.get().length != 32 || yv.get().length != 32)
			return null;

		EccPoint pkvecc = new EccPoint();
		pkvecc.setType(EccPointType.UNCOMPRESSED);
		pkvecc.setX(xv);
		pkvecc.setY(yv);

		PublicKey pkv = new PublicKey();
		pkv.setAlgorithm(PublicKeyAlgorithm.ECDSA_NISTP256_WITH_SHA256);
		pkv.setPublicKey(pkvecc);

		ArrayList<SubjectAttribute> subjectAttributeList = new ArrayList<>();
		// Assurance Level
		if (assuranceLevel != null && !assuranceLevel.isEmpty()) {
			byte[] assuranceBytes = EtsiPerrmissionUtils
					.subjectAssuranceInttoByte(new Integer(assuranceLevel));
			SubjectAssurance sa = new SubjectAssurance(assuranceBytes);
			SubjectAttribute s = new SubjectAttribute(sa);
			subjectAttributeList.add(s);
		}
		// no AIDs added

		// Verification Key
		subjectAttributeList.add(new SubjectAttribute(pkv, SubjectAttributeType.VERIFICATION_KEY));

		// add subject attributes
		SubjectAttribute[] subjectAttributes = new SubjectAttribute[subjectAttributeList.size()];
		for (int i = 0; i < subjectAttributeList.size(); i++) {
			subjectAttributes[i] = subjectAttributeList.get(i);
		}
		cert.setSubjectAttributes(subjectAttributes);

		// Validity Restrictions
		// -> Date Restrictions

		Time32 startValidity;
		Time32 endValidity;
		try {
			startValidity = new Time32(startDate);
			endValidity = new Time32(endDate);
		} catch (Exception e) {
			return null;
		}

		if (startValidity == null || endValidity == null)
			return null;

		ValidityRestriction[] validityRestrictions = new ValidityRestriction[2];
		validityRestrictions[0] = new ValidityRestriction(startValidity, endValidity);

		// set region restriction
		ValidityRestriction region = new ValidityRestriction(new GeographicRegion(RegionType.NONE));
		validityRestrictions[1] = region;

		// check validity time
		ExpirationTimeValidator expirationTimeValidator = new ExpirationTimeValidator();
		expirationTimeValidator.validate(validityRestrictions);
		if (expirationTimeValidator.validate(validityRestrictions) == false)
			throw new HandlerException("Expiration time of CRL signer certificate not valid. Start time = "
					+ startValidity + "; End time = " + endValidity);

		cert.setValidityRestrictions(validityRestrictions);

		// Signature
		Signature signature = new Signature();
		signature.setAlgorithm(PublicKeyAlgorithm.ECDSA_NISTP256_WITH_SHA256);
		EcdsaSignature ecdsaSignature = new EcdsaSignature();

		Signature sig = new Signature();
		if (!useHsm) {
			sig = EtsiCryptoUtils.sign(cert, PublicKeyAlgorithm.ECDSA_NISTP256_WITH_SHA256, rootPrivateKey);
		} else {

			sig = etsiPKCS11Utils.signEtsiSignature(cert, "SHA256withECDSA", alias, hsmLogin, dllPath);
		}

		ecdsaSignature.setR(sig.getEcdsaSignature().getR());
		ecdsaSignature.setS(sig.getEcdsaSignature().getS());
		signature.setEcdsaSignature(ecdsaSignature);
		cert.setSignature(signature);

		// verify signature of newly created certificate
		if (addRootCertDigestAsSigner) {
			if (cert.verfiySignature(rootCert) == false)
				throw new HandlerException("Signature of certificate invalid");
		} else {
			if (cert.verfiySignature() == false)
				throw new HandlerException("Signature of certificate invalid");
		}

		CaCertificate caCert = new CaCertificate();

		ByteArrayOutputStream certOut = new ByteArrayOutputStream();
		cert.writeData(new DataOutputStream(certOut));
		caCert.setCertificate(certOut.toByteArray());
		certOut.close();

		return caCert;

	}

	@Override
	public de.fraunhofer.sit.c2x.pki.ca.provider.interfaces.Certificate createNewPseudonymCertificate(
			String signingKeyX, String signingKeyY, String encryptionKeyX, String encryptionKeyY,
			String version, String algorithm, String symmetricAlgorithm, String compression,
			String startDate, String endDate, ArrayList<PsidSspPriorityDataContainer> psidSet,
			ECPrivateKeyParameters issuerPrivateKey, byte[] issuerCertificate, String assuranceLevel,
			ArrayList<GeographicRegionDataContainer> regions) throws DecoderException, IOException,
			ProviderException, HandlerException {
		// build certificate
		Certificate pc = new Certificate();

		// add subject info to new PC
		try {
			pc.setSubjectInfo(new SubjectInfo(SubjectType.AUTHORIZATION_TICKET, ""));
		} catch (IOException e) {
			logger.error("Unable to create subject info for new PC: " + e.getMessage());
			throw new HandlerException("Unable to create subject info for new PC", e);
		}

		// add signer to new PC
		Certificate issuerCert = WaveUtils.getElementFromBytes(issuerCertificate, Certificate.class);
		if (issuerCert.getSubjectInfo().getSubjectType() != SubjectType.AUTHORIZATION_AUTHORITY)
			throw new HandlerException(
					"Issuer of pseudonym certificate must be from type AUTHORIZATION_AUTHORITY. Given isser certificate is from type: "
							+ issuerCert.getSubjectInfo().getSubjectType());
		SignerInfo pcaSignerInfo;
		pcaSignerInfo = new SignerInfo(new HashedId8(issuerCert.getHashedId8().getCertID().get()));
		pc.setSignerInfo(pcaSignerInfo);

		// add subject attributes to new PC
		ArrayList<SubjectAttribute> subjectAttributes = new ArrayList<>();
		// Assurance Level
		if (assuranceLevel != null && !assuranceLevel.isEmpty()) {
			byte[] assuranceBytes = EtsiPerrmissionUtils
					.subjectAssuranceInttoByte(new Integer(assuranceLevel));
			SubjectAssurance sa = new SubjectAssurance(assuranceBytes);
			SubjectAttribute s = new SubjectAttribute(sa);
			subjectAttributes.add(s);
		}
		// AIDs
		ArrayList<SubjectAttribute> psidSubjectAttributeLists = EtsiPerrmissionUtils
				.psidSetToSubjectAttributeArray(psidSet);
		boolean hasAid = false;
		for (SubjectAttribute sa : psidSubjectAttributeLists) {
			if (sa.getType() == SubjectAttributeType.ITS_AID_SSP_LIST)
				hasAid = true;
			else if (sa.getType() == SubjectAttributeType.ITS_AID_LIST)
				throw new HandlerException("Invalid subject attribute type: " + sa.getType());

			subjectAttributes.add(sa);
		}
		if (hasAid == false)
			throw new HandlerException("No AID given");

		// check
		if (signingKeyX == null || signingKeyY == null)
			throw new HandlerException("Signing key must not be null");

		SymmAlgorithm symmAlgorithm = SymmAlgorithm.AES_128_CCM;

		Opaque xv = new Opaque();
		xv.set(Hex.decodeHex(signingKeyX.toCharArray()));
		xv.setArray(true);

		Opaque yv = new Opaque();
		yv.set(Hex.decodeHex(signingKeyY.toCharArray()));
		yv.setArray(true);

		if (xv.get().length != 32 || yv.get().length != 32)
			throw new HandlerException("Invalid key length of signing key: " + xv.get().length);

		EccPoint pkvecc = new EccPoint();
		pkvecc.setType(EccPointType.UNCOMPRESSED);
		pkvecc.setX(xv);
		pkvecc.setY(yv);

		PublicKey pkv = new PublicKey();
		pkv.setAlgorithm(PublicKeyAlgorithm.ECDSA_NISTP256_WITH_SHA256);
		pkv.setPublicKey(pkvecc);

		// copy verification public key
		subjectAttributes.add(new SubjectAttribute(pkv, SubjectAttributeType.VERIFICATION_KEY));

		if (encryptionKeyX != null && encryptionKeyY != null) {
			Opaque xe = new Opaque();
			xe.set(Hex.decodeHex(encryptionKeyX.toCharArray()));
			xe.setArray(true);

			Opaque ye = new Opaque();
			ye.set(Hex.decodeHex(encryptionKeyY.toCharArray()));
			ye.setArray(true);

			if (xe.get().length != 32 || ye.get().length != 32)
				throw new HandlerException("Invalid key length of encryption key: " + xe.get().length);

			EccPoint pkeecc = new EccPoint();
			pkeecc.setType(EccPointType.UNCOMPRESSED);
			pkeecc.setX(xe);
			pkeecc.setY(ye);

			PublicKey pke = new PublicKey();
			pke.setAlgorithm(PublicKeyAlgorithm.ECIES_NISTP256);
			pke.setPublicKey(pkeecc);
			pke.setSupportedSymmAlg(symmAlgorithm);

			// copy encryption public key
			subjectAttributes.add(new SubjectAttribute(pke, SubjectAttributeType.ENCRYPTION_KEY));
		}

		pc.setSubjectAttributes(subjectAttributes.toArray(new SubjectAttribute[subjectAttributes.size()]));

		// add validity restrictions to new PC
		Time32 startValidity;
		Time32 endValidity;
		try {
			startValidity = new Time32(startDate);
			endValidity = new Time32(endDate);
		} catch (Exception e) {
			return null;
		}

		if (startValidity == null || endValidity == null)
			return null;

		ValidityRestriction[] validityRestrictions = new ValidityRestriction[2];
		validityRestrictions[0] = new ValidityRestriction();
		validityRestrictions[0].setType(ValidityRestrictionType.TIME_START_AND_END);
		validityRestrictions[0].setStartValidity(startValidity);
		validityRestrictions[0].setEndValidity(endValidity);

		// check validity time
		ExpirationTimeValidator expirationTimeValidator = new ExpirationTimeValidator();
		expirationTimeValidator.validate(validityRestrictions);
		if (expirationTimeValidator.validate(validityRestrictions) == false)
			throw new HandlerException("Expiration time of pseudonym certificate not valid. Start time = "
					+ startValidity + "; End time = " + endValidity);

		// set region restriction
		if (regions == null || regions.size() == 0) {
			ValidityRestriction region = new ValidityRestriction(new GeographicRegion(RegionType.NONE));
			validityRestrictions[1] = region;
		} else {
			if (regions.size() > 1)
				throw new HandlerException("More than one region restriction not supported.");

			GeographicRegion requestedRegion = EtsiRegionUtils.regionContainerToGeographicRegion(regions
					.get(0));

			// do not check region restriction
			ValidityRestriction region = new ValidityRestriction(requestedRegion);
			validityRestrictions[1] = region;
		}

		pc.setValidityRestrictions(validityRestrictions);

		// sign PC
		Signature pcSignature;
		try {
			pcSignature = EtsiCryptoUtils.sign(pc, PublicKeyAlgorithm.ECDSA_NISTP256_WITH_SHA256,
					issuerPrivateKey);
		} catch (IOException e) {
			logger.error("Signing of new PC failed: " + e.getMessage());
			throw new HandlerException("Signing of new PC failed", e);
		}

		// add signature to new PC
		pc.setSignature(pcSignature);

		// verify signature of newly created certificate
		if (pc.verfiySignature(issuerCert) == false)
			throw new HandlerException("Signature of certificate invalid");

		byte[] pcBytes = WaveUtils.getBytesFromWaveType(pc);

		CaCertificate caCert = new CaCertificate();
		caCert.setCertificate(pcBytes);

		return caCert;
	}
}
