package de.fraunhofer.sit.c2x.pki.etsi_ts103097v1114.impl;

import java.io.DataInputStream;
import java.io.DataOutputStream;
import java.io.IOException;
import java.util.ArrayList;
import java.util.TimeZone;

import org.joda.time.DateTime;
import org.joda.time.DateTimeZone;
import org.joda.time.format.DateTimeFormat;
import org.joda.time.format.DateTimeFormatter;

import de.fraunhofer.sit.c2x.pki.etsi_ts103097v1114.serializer.Internal;
import de.fraunhofer.sit.c2x.pki.etsi_ts103097v1114.visitor.EtsiVisitor;

/**
 * 
 * @author Daniel Quanz (daniel.quanz@sit.fraunhofer.de)
 * 
 */
public class Time32 extends WaveElement {

	// ---- fields ----

	@Internal(order = 0)
	private UInt32 time32;

	// ---- fields ----

	// ---- internal variables ----

	private DateTime dateTime;

	public static final TimeZone TIME_ZONE_UTC = TimeZone.getTimeZone("UTC");

	/**
	 * Number of seconds that has be subtracted for changing the time
	 * calculation origin from the "JAVA epoch" to "IEEE 1609.2 Time32". NOTE:
	 * Due to leap seconds, the number of seconds from the Linux epoch of
	 * 00:00:00 UTC, 1 January, 1970 to the 1609.2 and 103 097 epoch of 00:00:00
	 * UTC, 1 January, 2004 is 1,072,915,234 seconds if the 34 leap seconds are
	 * considered. Otherwise the diff is 1,072,915,200 seconds.
	 * 
	 * <pre>
	 * Java time calculation origin:      1970-Jan-01
	 * IEEE 1609 time calculation origin: 2004-Jan-01
	 * </pre>
	 */
	// 00:00:00 UTC, 1 January, 2004
	public static final long TIME32_JAVA_DIFF = 1072915200;

	public static final long MAX_TIME32 = 0xFFFFFFFFL;

	// ---- constructors ----

	public Time32() {
		this(DateTime.now(DateTimeZone.UTC).getMillis() / 1000 * 1000);
	}

	public Time32(long timeMillis) {
		if (timeMillis < 0)
			throw new IllegalArgumentException("Time must be positive. Given value " + timeMillis
					+ " is invalid");
		int leap = computeLeapSeconds(timeMillis);
		this.dateTime = new DateTime(timeMillis, DateTimeZone.UTC);
		this.time32 = new UInt32(((timeMillis / 1000) - TIME32_JAVA_DIFF) + leap);
	}

	public Time32(int year, int month, int day, int hour, int minute, int second) {
		this(year, month, day, hour, minute, second, 0);
	}

	public Time32(int year, int month, int day, int hour, int minute, int second, int millis) {
		this(new DateTime(year, month, day, hour, minute, second, millis, DateTimeZone.UTC).getMillis());
	}

	public Time32(DateTime dateTime) {
		this(dateTime.getMillis());
	}

	public Time32(String date) throws Exception {
		this(getDateTimeFromString(date));
	}

	private static DateTime getDateTimeFromString(String date) {
		DateTime dateTime = null;

		if (date == null) {
			throw new IllegalArgumentException("Not a valid date/time: " + date);
		}

		try {
			DateTimeFormatter formatter = DateTimeFormat.forPattern("dd.MM.yyyy HH:mm:ss");
			dateTime = formatter.withZoneUTC().parseDateTime(date);

		} catch (Exception e) {

			try {
				DateTimeFormatter formatter2 = DateTimeFormat.forPattern("dd.MM.yyyy");
				if (dateTime == null)
					dateTime = formatter2.withZoneUTC().parseDateTime(date);
			} catch (Exception e2) {
				throw new IllegalArgumentException("Not a valid date/time: " + date);
			}
		}
		return dateTime;
	}

	public Time32(UInt32 time32Value) {
		this.time32 = time32Value;
		int leap = computeLeapSeconds((time32.get() + TIME32_JAVA_DIFF) * 1000);
		this.dateTime = new DateTime(((time32.get() + TIME32_JAVA_DIFF) * 1000 - leap * 1000), DateTimeZone.UTC);
	}

	public Time32(DataInputStream in) throws IOException {
		this.time32 = new UInt32(in);
		long timeMillis = (this.time32.get() + Time32.TIME32_JAVA_DIFF) * 1000;
		int leapSec = computeLeapSeconds(timeMillis);
		this.dateTime = new DateTime(timeMillis - leapSec * 1000, DateTimeZone.UTC);
	}

	// ---- accept ----

	public <T> T accept(EtsiVisitor<T> visitor) {
		return visitor.visit(this);
	}

	public void setTime32(UInt32 time32) {
		this.time32 = time32;
	}

	@Override
	public int writeData(DataOutputStream out) throws IOException {
		if (time32 == null) {
			throw new IllegalArgumentException();
		}
		return time32.writeData(out);
	}

	// ### Copied from IEEE ###

	/**
	 * @param javaTimeMillis
	 * @return
	 */
	protected static int computeLeapSeconds(long timeMillis) {
		Time32.init();
		int leapOffSet = 0;
		for (int i = Time32.LEAP_SECONDS.size() - 1; i >= 0; i--) {
			long leapSeconds = Time32.LEAP_SECONDS.get(i);
			if (timeMillis > leapSeconds) {
				leapOffSet = i + 1;
				break;
			}
		}
		return Time32.LEAP_BASE + leapOffSet - 32; // -32 leap seconds since
													// 2004
	}

	private static final int LEAP_BASE = 10;
	private static final ArrayList<Long> LEAP_SECONDS = new ArrayList<Long>();

	private static void set(long timeMillis) {
		Time32.LEAP_SECONDS.add(timeMillis);
	}

	private static void set(int year, boolean last) {
		int monthOfYear = last ? 12 : 6;
		int dayOfMonth = last ? 31 : 30;
		Time32.set(new DateTime(year, monthOfYear, dayOfMonth, 23, 59, 59, 999, DateTimeZone.UTC).getMillis());
	}

	private static void init() {
		if (Time32.LEAP_SECONDS.size() == 0) {
			Time32.set(1972, false);
			Time32.set(1972, true);
			Time32.set(1973, true);
			Time32.set(1974, true);
			Time32.set(1975, true);
			Time32.set(1976, true);
			Time32.set(1977, true);
			Time32.set(1978, true);
			Time32.set(1979, true);
			Time32.set(1981, false);
			Time32.set(1982, false);
			Time32.set(1983, false);
			Time32.set(1985, false);
			Time32.set(1987, true);
			Time32.set(1989, true);
			Time32.set(1990, true);
			Time32.set(1992, false);
			Time32.set(1993, false);
			Time32.set(1994, false);
			Time32.set(1995, true);
			Time32.set(1997, false);
			Time32.set(1998, true);
			Time32.set(2005, true);
			Time32.set(2008, true);
			Time32.set(2012, false);
		}
	}

	/**
	 * Number of seconds since 00:00:00 UTC, 1 January, 2004
	 * 
	 * @return [0..2<sup>32</sup>-1]
	 */
	public UInt32 getTime32() {
		return this.time32;
	}

	public boolean isFutureDate() {
		return this.time32.get() > new Time32().getTime32().get();
	}

	public boolean isPastDate() {
		return this.time32.get() < new Time32().getTime32().get();
	}

	public java.sql.Timestamp toSqlTimestamp() {
		int leap = computeLeapSeconds(this.time32.get() * 1000 + TIME32_JAVA_DIFF);
		long date = (this.time32.get() - leap) * 1000 + Time32.TIME32_JAVA_DIFF;
		return new java.sql.Timestamp(date);
	}

	/**
	 * Tests if this date is before the specified date.
	 * 
	 * @param when
	 * @return true if and only if the instant of time represented by this
	 *         Time32 object is strictly earlier than the instant represented by
	 *         when; false otherwise.
	 */
	public boolean before(Time32 when) {
		return this.getTime32().get() < when.getTime32().get();
	}

	/**
	 * @return the dateTime
	 */
	public DateTime getDateTime() {
		return this.dateTime;
	}

	public Time32 addSeconds(int seconds) {
		long newTime32 = this.time32.get() + seconds;
		return new Time32(new UInt32(newTime32));
	}

	public Time32 subtractSeconds(long seconds) {
		long newTime32 = this.time32.get() - seconds;
		if (newTime32 < 0) {
			throw new IllegalArgumentException("New time is smaller than 0");
		}
		return new Time32(new UInt32(newTime32));
	}

	public Time32 subtract(Time32 time32) {
		long newTime32 = this.time32.get() - time32.getTime32().get();
		if (newTime32 < 0) {
			throw new IllegalArgumentException("New time is smaller than 0");
		}
		return new Time32(new UInt32(newTime32));
	}

	public int getYear() {
		return this.dateTime.getYear();
	}

	public int getMonthOfYear() {
		return this.dateTime.getMonthOfYear();
	}

	public int getDayOfMonth() {
		return this.dateTime.getDayOfMonth();
	}

	public int getHourOfDay() {
		return this.dateTime.getHourOfDay();
	}

	public int getMinuteOfHour() {
		return this.dateTime.getMinuteOfHour();
	}

	public int getSecondOfMinute() {
		return this.dateTime.getSecondOfMinute();
	}

	public long getSeconds() {
		return (this.dateTime.getMillis() / 1000);
	}

	@Override
	public String toString() {
		return dateTime.toString();
	}
}