package de.fraunhofer.sit.c2x.pki.etsi_ts103097v1114.impl;

import java.io.DataInputStream;
import java.io.DataOutputStream;
import java.io.IOException;

import de.fraunhofer.sit.c2x.pki.ca.utils.WaveUtils;
import de.fraunhofer.sit.c2x.pki.etsi_ts103097v1114.impl.PublicKeyAlgorithmImpl.PublicKeyAlgorithm;
import de.fraunhofer.sit.c2x.pki.etsi_ts103097v1114.impl.SymmetricAlgorithmImpl.SymmAlgorithm;
import de.fraunhofer.sit.c2x.pki.etsi_ts103097v1114.visitor.EtsiVisitor;

/**
 * @author Daniel Quanz (daniel.quanz@sit.fraunhofer.de)
 */
public class RecipientInfo extends WaveElement {

	private HashedId8 certId;
	private PublicKeyAlgorithm pkEncryption;
	private EciesEncryptedKey encKey;
	private Opaque otherEncKey;

	public RecipientInfo() {
	}

	public RecipientInfo(HashedId8 certId, EciesEncryptedKey encKey) {
		this.certId = certId;
		this.pkEncryption = PublicKeyAlgorithm.ECIES_NISTP256;
		this.encKey = encKey;
	}

	public RecipientInfo(DataInputStream in, SymmAlgorithm symmAlg)
			throws IOException {
		certId = new HashedId8(in);
		pkEncryption = PublicKeyAlgorithmImpl.getInstance().getEnumType(
				in.readByte());
		if (pkEncryption == PublicKeyAlgorithm.ECIES_NISTP256)
			encKey = new EciesEncryptedKey(in, symmAlg);
		else
			otherEncKey = new Opaque(in);
	}

	public <T> T accept(EtsiVisitor<T> visitor) {
		return visitor.visit(this);
	}

	@Override
	public int writeData(DataOutputStream out) throws IOException {
		int written = WaveUtils.writeWave(out, certId);
		written += PublicKeyAlgorithmImpl.getInstance().writeData(out,
				pkEncryption);
		if (pkEncryption == PublicKeyAlgorithm.ECIES_NISTP256)
			written += WaveUtils.writeWave(out, encKey);
		else
			written += WaveUtils.writeWave(out, otherEncKey);
		return written;
	}

	public HashedId8 getHashedID() {
		return certId;
	}

	public PublicKeyAlgorithm getPkEncryption() {
		return pkEncryption;
	}

	public EciesEncryptedKey getEncKey() {
		return encKey;
	}

	public Opaque getOtherEncKey() {
		return otherEncKey;
	}

	public void setHashedId(HashedId8 certId) {
		this.certId = certId;
	}

	public void setPkEncryption(PublicKeyAlgorithm pkEncryption) {
		this.pkEncryption = pkEncryption;
	}

	public void setEncKey(EciesEncryptedKey encKey) {
		this.encKey = encKey;
	}

	public void setOtherEncKey(Opaque otherEncKey) {
		this.otherEncKey = otherEncKey;
	}

}
