package de.fraunhofer.sit.c2x.pki.etsi_ts103097v1114.impl;

import java.io.DataInputStream;
import java.io.DataOutputStream;
import java.io.IOException;

import de.fraunhofer.sit.c2x.pki.etsi_ts103097v1114.impl.PublicKeyAlgorithmImpl.PublicKeyAlgorithm;
import de.fraunhofer.sit.c2x.pki.etsi_ts103097v1114.impl.SymmetricAlgorithmImpl.SymmAlgorithm;
import de.fraunhofer.sit.c2x.pki.etsi_ts103097v1114.serializer.Internal;
import de.fraunhofer.sit.c2x.pki.etsi_ts103097v1114.visitor.EtsiVisitor;

/**
 * 
 * @author Daniel Quanz (daniel.quanz@sit.fraunhofer.de)
 * 
 */
public class PublicKey extends WaveElement {

	// ---- fields ----

	@Internal(order = 0)
	private PublicKeyAlgorithm algorithm;

	@Internal(order = 1)
	private EccPoint publicKey;

	@Internal(order = 2)
	private SymmAlgorithm supportedSymmAlg;

	@Internal(order = 3)
	private Opaque otherKey;

	// ---- constructors ----

	public PublicKey() {
	}
	
	public PublicKey(PublicKeyAlgorithm algorithm, EccPoint publicKey) throws IOException{
		if(algorithm != PublicKeyAlgorithm.ECDSA_NISTP256_WITH_SHA256)
			throw new IOException("Unexpted algorithm");
		this.algorithm = algorithm;
		this.publicKey = publicKey;
	}
	
	public PublicKey(PublicKeyAlgorithm algorithm, EccPoint publicKey, SymmAlgorithm smmAlgorithm) throws IOException{
		if(algorithm != PublicKeyAlgorithm.ECIES_NISTP256)
			throw new IOException("Unexpted pkAlgorithm");
		if(smmAlgorithm != SymmAlgorithm.AES_128_CCM)
			throw new IOException("Unexpted smmAlgorithm");
		this.algorithm = algorithm;
		this.publicKey = publicKey;
		this.supportedSymmAlg = smmAlgorithm;
	}

	public PublicKey(DataInputStream in) throws IOException {
		algorithm = PublicKeyAlgorithmImpl.getInstance().getEnumType(
				in.readByte());
		if (algorithm == PublicKeyAlgorithm.ECDSA_NISTP256_WITH_SHA256) {
			publicKey = new EccPoint(in, new UInt8(32), algorithm);
		} else if (algorithm == PublicKeyAlgorithm.ECIES_NISTP256) {
			supportedSymmAlg = SymmetricAlgorithmImpl.getInstance()
					.getEnumType(in.readByte());
			publicKey = new EccPoint(in, new UInt8(32), algorithm);
		} else {
			otherKey = new Opaque(in);
		}
	}

	// ---- accept ----

	public <T> T accept(EtsiVisitor<T> visitor) {
		return visitor.visit(this);
	}

	// ---- getter ----

	public PublicKeyAlgorithm getAlgorithm() {
		return this.algorithm;
	}

	public EccPoint getPublicKey() {
		return this.publicKey;
	}

	public SymmAlgorithm getSupportedSymmAlg() {
		return this.supportedSymmAlg;
	}

	public Opaque getOtherKey() {
		return this.otherKey;
	}

	// ---- setter ----

	public void setAlgorithm(PublicKeyAlgorithm algorithm) {
		this.algorithm = algorithm;
	}

	public void setPublicKey(EccPoint publicKey) {
		this.publicKey = publicKey;
	}

	public void setSupportedSymmAlg(SymmAlgorithm supportedSymmAlg) {
		this.supportedSymmAlg = supportedSymmAlg;
	}

	public void setOtherKey(Opaque otherKey) {
		this.otherKey = otherKey;
	}

	@Override
	public int writeData(DataOutputStream out) throws IOException {

		if (algorithm == null) {
			throw new IllegalArgumentException();
		}
		int written = PublicKeyAlgorithmImpl.getInstance().writeData(out,
				algorithm);
		if (algorithm == PublicKeyAlgorithm.ECDSA_NISTP256_WITH_SHA256) {
			if (publicKey == null)
				throw new IllegalArgumentException("public key missing");
			written += publicKey.writeData(out);
		} else if (algorithm == PublicKeyAlgorithm.ECIES_NISTP256) {
			if (publicKey == null || supportedSymmAlg == null)
				throw new IllegalArgumentException("public key or symmetric algorithm missing");

			written += SymmetricAlgorithmImpl.getInstance().writeData(out,
					supportedSymmAlg);
			written += publicKey.writeData(out);
		} else {
			if (otherKey == null)
				throw new IllegalArgumentException();
			written += otherKey.writeData(out);
		}
		return written;
	}
}