package de.fraunhofer.sit.c2x.pki.etsi_ts103097v1114.impl;

import java.io.DataInputStream;
import java.io.DataOutputStream;
import java.io.IOException;
import java.util.Arrays;

import org.bouncycastle.crypto.CipherParameters;
import org.bouncycastle.crypto.params.ECDomainParameters;
import org.bouncycastle.crypto.params.ECPublicKeyParameters;
import org.bouncycastle.jce.ECNamedCurveTable;
import org.bouncycastle.jce.spec.ECParameterSpec;
import org.bouncycastle.math.ec.ECCurve;
import org.bouncycastle.math.ec.ECPoint;

import de.fraunhofer.sit.c2x.pki.etsi_ts103097v1114.impl.EccPointTypeImpl.EccPointType;
import de.fraunhofer.sit.c2x.pki.etsi_ts103097v1114.impl.PublicKeyAlgorithmImpl.PublicKeyAlgorithm;
import de.fraunhofer.sit.c2x.pki.etsi_ts103097v1114.serializer.External;
import de.fraunhofer.sit.c2x.pki.etsi_ts103097v1114.serializer.Internal;
import de.fraunhofer.sit.c2x.pki.etsi_ts103097v1114.visitor.EtsiVisitor;

/**
 * 
 * @author Daniel Quanz (daniel.quanz@sit.fraunhofer.de)
 * 
 */
public class EccPoint extends WaveElement {

	// ---- fields ----

	@Internal(order = 0)
	private EccPointType type;

	@Internal(order = 1)
	private Opaque x;

	@Internal(order = 2)
	private Opaque y;

	@Internal(order = 3)
	private Opaque data;

	// ---- fields ----

	@External
	private PublicKeyAlgorithm algorithm;

	@External
	private UInt8 fieldSize;

	// ---- constructors ----

	public EccPoint() {
		this.fieldSize = new UInt8(32);
	}

	public EccPoint(EccPointType type, byte[] x, byte[] y, PublicKeyAlgorithm algorithm, int fieldSize) {
		super();
		this.type = type;
		if (x != null)
			this.x = new Opaque(x, true);
		if (y != null)
			this.y = new Opaque(y, true);
		this.algorithm = algorithm;
		this.fieldSize = new UInt8(fieldSize);
	}

	public EccPoint(DataInputStream in, UInt8 fieldSize, PublicKeyAlgorithm algorithm) throws IOException {
		this.type = EccPointTypeImpl.getInstance().getEnumType(in.readByte());
		this.fieldSize = fieldSize;
		this.algorithm = algorithm;
		this.x = new Opaque(in, fieldSize.get(), true);
		switch (type) {
		case X_COORDINATE_ONLY:
		case COMPRESSED_LSB_Y_0:
		case COMPRESSED_LSB_Y_1:
			break;
		case UNCOMPRESSED:
			this.y = new Opaque(in, fieldSize.get(), true);
			break;

		default:
			this.data = new Opaque(in);
			break;
		}
	}

	public EccPoint(CipherParameters pubKey) {

		ECPublicKeyParameters publicKey = (ECPublicKeyParameters) pubKey;

		this.fieldSize = new UInt8(publicKey.getParameters().getCurve().getFieldSize() / 8);
		this.checkFieldsize(this.fieldSize);

		byte[] q = publicKey.getQ().getEncoded();

		this.type = EccPointTypeImpl.getInstance().getEnumType(q[0]);

		byte[] tmp = new byte[this.fieldSize.get()];
		tmp = Arrays.copyOfRange(q, 1, this.fieldSize.get() + 1);
		this.x = new Opaque();
		this.x.set(tmp);
		this.x.setArray(true);

		if (this.getType() == EccPointType.UNCOMPRESSED) {
			// if (q.length != (2 * fieldSize.get() + 1))
			// throw new IllegalArgumentException("");

			tmp = new byte[this.fieldSize.get()];
			tmp = Arrays.copyOfRange(q, this.fieldSize.get() + 1, 2 * this.fieldSize.get() + 1);
			this.y = new Opaque();
			this.y.set(tmp);
			this.y.setArray(true);
		} else {
			this.y = null;
		}

	}

	// ---- accept ----

	public <T> T accept(EtsiVisitor<T> visitor) {
		return visitor.visit(this);
	}

	// ---- getter ----

	public EccPointType getType() {
		return this.type;
	}

	public Opaque getX() {
		return this.x;
	}

	public Opaque getY() {
		return this.y;
	}

	public Opaque getData() {
		return this.data;
	}

	public PublicKeyAlgorithm getAlgorithm() {
		return this.algorithm;
	}

	public UInt8 getFieldSize() {
		return this.fieldSize;
	}

	// ---- setter ----

	public void setType(EccPointType type) {
		this.type = type;
	}

	public void setX(Opaque x) {
		this.x = x;
	}

	public void setY(Opaque y) {
		this.y = y;
	}

	public void setData(Opaque data) {
		this.data = data;
	}

	public void setAlgorithm(PublicKeyAlgorithm algorithm) {
		this.algorithm = algorithm;
	}

	public void setFieldSize(UInt8 fieldSize) {
		this.fieldSize = fieldSize;
	}

	@Override
	public int writeData(DataOutputStream out) throws IOException {

		if (type == null || x == null) {
			throw new IllegalArgumentException(type + " - " + x);
		}
		if (!x.isArray()) {
			throw new IllegalArgumentException("x is not an array");
		}
		int written = 0;
		written += EccPointTypeImpl.getInstance().writeData(out, type);
		written += this.x.writeData(out);
		switch (type) {
		case X_COORDINATE_ONLY:
		case COMPRESSED_LSB_Y_0:
		case COMPRESSED_LSB_Y_1:
			break;
		case UNCOMPRESSED:
			if (y == null || !y.isArray() || y.getLength() != fieldSize.get())
				throw new IllegalArgumentException();
			written += this.y.writeData(out);
			break;

		default:
			if (data == null)
				throw new IllegalArgumentException();
			written += this.data.writeData(out);
			break;
		}
		return written;

	}

	// Utils

	private void checkFieldsize(UInt8 fieldSize) {
		if (fieldSize == null || fieldSize.get() != 28 && fieldSize.get() != 32) {
			throw new IllegalArgumentException("ECC PKAlgorithm with field size " + fieldSize
					+ " not supported");
		}
	}

	public ECPublicKeyParameters toECPublicKeyParameters() {

		// Global Params
		ECParameterSpec spec = ECNamedCurveTable.getParameterSpec("P-" + this.fieldSize.get() * 8 + "");
		ECCurve.Fp curve = (ECCurve.Fp) spec.getCurve();
		ECDomainParameters domain = new ECDomainParameters(curve, spec.getG(), spec.getN(), spec.getH(),
				spec.getSeed());

		boolean withCompression = this.type != EccPointType.UNCOMPRESSED;
		int size = withCompression ? this.fieldSize.get() + 1 : 2 * this.fieldSize.get() + 1;
		byte[] encoded = new byte[size];
		encoded[0] = this.type.getType();
		System.arraycopy(this.x.get(), 0, encoded, 1, this.fieldSize.get());
		if (!withCompression) {
			System.arraycopy(this.y.get(), 0, encoded, this.fieldSize.get() + 1, this.fieldSize.get());
		}

		ECPoint q = domain.getCurve().decodePoint(encoded);

		return new ECPublicKeyParameters(q, domain);
	}

}