package de.fraunhofer.sit.c2x.pki.etsi_ts103097v1114.handler.adapter;

import java.io.IOException;
import java.io.UnsupportedEncodingException;
import java.sql.Timestamp;

import de.fraunhofer.sit.c2x.pki.ca.provider.interfaces.Certificate;
import de.fraunhofer.sit.c2x.pki.etsi_ts103097v1114.TypeAndVersion;
import de.fraunhofer.sit.c2x.pki.etsi_ts103097v1114.impl.ValidityRestriction;
import de.fraunhofer.sit.c2x.pki.etsi_ts103097v1114.impl.ValidityRestrictionTypeImpl.ValidityRestrictionType;

/**
 * @author Daniel Quanz (daniel.quanz@sit.fraunhofer.de)
 */
public class CertificateAdapter implements Certificate {

	private final de.fraunhofer.sit.c2x.pki.etsi_ts103097v1114.impl.Certificate cert;

	public CertificateAdapter(
			de.fraunhofer.sit.c2x.pki.etsi_ts103097v1114.impl.Certificate cert) {
		this.cert = cert;
	}

	@Override
	public Timestamp getStartTime() throws IOException {
		for (ValidityRestriction restriction : cert.getValidityRestrictions()) {
			if ((restriction.getType() == ValidityRestrictionType.TIME_START_AND_DURATION || restriction
					.getType() == ValidityRestrictionType.TIME_START_AND_END)
					&& restriction.getStartValidity() != null)
				return restriction.getStartValidity().toSqlTimestamp();
		}
		throw new IOException("Start time not available");
	}

	@Override
	public Timestamp getExpirationTime() throws IOException {
		for (ValidityRestriction restriction : cert.getValidityRestrictions()) {
			if ((restriction.getType() == ValidityRestrictionType.TIME_END || restriction
					.getType() == ValidityRestrictionType.TIME_START_AND_END)
					&& restriction.getEndValidity() != null)
				return restriction.getEndValidity().toSqlTimestamp();
			if (restriction.getType() == ValidityRestrictionType.TIME_START_AND_DURATION
					&& restriction.getStartValidity() != null
					&& restriction.getDuration() != null)
				return restriction
						.getStartValidity()
						.addSeconds(
								restriction.getDuration()
										.getDurationInSeconds())
						.toSqlTimestamp();
		}
		throw new IOException("Expiration time not available");
	}

	@Override
	public byte[] getCertId() {
		return cert.getHashedId8().getCertID().get();
	}

	@Override
	public byte[] getCertificate() {
		try {
			return cert.getBytes();
		} catch (IOException e) {
			e.printStackTrace();
			return new byte[0];
		}
	}

	@Override
	public String getSubjectName() {
		try {
			return new String(cert.getSubjectInfo().getSubjectName().get(),
					"UTF-8");
		} catch (UnsupportedEncodingException e) {
			return "";
		}
	}

	@Override
	public byte[] getSignerCertId() throws IOException {
		if (cert.getSignerInfo() == null)
			throw new IOException("No signer info available");

		switch (cert.getSignerInfo().getType()) {
		case SELF:
			return new byte[0];
		case ID:
			return cert.getSignerInfo().getId().get();
		case CERTIFICATE:
			return cert.getSignerInfo().getCertificate().getHashedId8()
					.getCertID().get();
		case CERTIFICATE_CHAIN:
			return cert.getSignerInfo().getCertificates()[0].getHashedId8()
					.getCertID().get();
		case CERTIFICATE_DIGEST_WITH_ECDSAP256:
			return cert.getSignerInfo().getDigest().getCertID().get();
		case CERTIFICATE_DIGEST_WITH_OTHER_ALGORITHM:
			return cert.getSignerInfo().getDigest().getCertID().get();
		default:
			throw new IOException("Unexpected signer type");
		}
	}

	@Override
	public String getSubjectType() {
		return cert.getSubjectInfo().getSubjectType().toString();
	}

	@Override
	public String getCertTypeAndVersion() {
		return TypeAndVersion.getCertTypeAndVersion();
	}

}
