package de.fraunhofer.sit.c2x.pki.etsi_ts103097v1114.handler;

import java.io.ByteArrayInputStream;
import java.io.DataInputStream;
import java.io.IOException;
import java.security.Key;
import java.security.KeyStoreException;
import java.security.NoSuchAlgorithmException;
import java.security.cert.CertificateException;

import org.apache.commons.codec.binary.Hex;
import org.apache.log4j.Logger;
import org.bouncycastle.crypto.AsymmetricCipherKeyPair;
import org.bouncycastle.crypto.CipherParameters;
import org.bouncycastle.crypto.CryptoException;

import com.google.inject.Inject;

import de.fraunhofer.sit.c2x.pki.ca.core.logging.InjectLogger;
import de.fraunhofer.sit.c2x.pki.ca.crypto.KeyGenerator;
import de.fraunhofer.sit.c2x.pki.ca.crypto.encryption.ECIESCipher;
import de.fraunhofer.sit.c2x.pki.ca.crypto.keystore.KeyStoreImpl;
import de.fraunhofer.sit.c2x.pki.ca.crypto.keystore.KeystoreAdapter;
import de.fraunhofer.sit.c2x.pki.ca.module.webserver.servlets.datatypes.CaCertificateWebHandlerResult;
import de.fraunhofer.sit.c2x.pki.ca.module.webserver.servlets.interfaces.CaCertificateWebHandler;
import de.fraunhofer.sit.c2x.pki.ca.provider.interfaces.CaInfoProvider;
import de.fraunhofer.sit.c2x.pki.ca.provider.interfaces.ConfigProvider;
import de.fraunhofer.sit.c2x.pki.ca.provider.interfaces.KeystoreProvider;
import de.fraunhofer.sit.c2x.pki.ca.provider.interfaces.KeystoreProvider.KeyType;
import de.fraunhofer.sit.c2x.pki.ca.utils.WaveUtils;
import de.fraunhofer.sit.c2x.pki.etsi_ts103097v1114.impl.Certificate;
import de.fraunhofer.sit.c2x.pki.etsi_ts103097v1114.impl.EccPoint;
import de.fraunhofer.sit.c2x.pki.etsi_ts103097v1114.impl.EccPointTypeImpl.EccPointType;
import de.fraunhofer.sit.c2x.pki.etsi_ts103097v1114.impl.EciesEncryptedKey;
import de.fraunhofer.sit.c2x.pki.etsi_ts103097v1114.impl.EncryptionParameters;
import de.fraunhofer.sit.c2x.pki.etsi_ts103097v1114.impl.HashedId8;
import de.fraunhofer.sit.c2x.pki.etsi_ts103097v1114.impl.Opaque;
import de.fraunhofer.sit.c2x.pki.etsi_ts103097v1114.impl.PublicKey;
import de.fraunhofer.sit.c2x.pki.etsi_ts103097v1114.impl.PublicKeyAlgorithmImpl.PublicKeyAlgorithm;
import de.fraunhofer.sit.c2x.pki.etsi_ts103097v1114.impl.RecipientInfo;
import de.fraunhofer.sit.c2x.pki.etsi_ts103097v1114.impl.SignerInfo;
import de.fraunhofer.sit.c2x.pki.etsi_ts103097v1114.impl.SubjectTypeImpl.SubjectType;
import de.fraunhofer.sit.c2x.pki.etsi_ts103097v1114.impl.SymmetricAlgorithmImpl.SymmAlgorithm;
import de.fraunhofer.sit.c2x.pki.etsi_ts103097v1114.impl.UInt32;
import de.fraunhofer.sit.c2x.pki.etsi_ts103097v1114.utils.EtsiCryptoUtils;

public class EtsiPcaCertificateWebHandler implements CaCertificateWebHandler {

	@Inject
	private KeystoreProvider keystoreProvider;

	@Inject
	private CaInfoProvider<Certificate> caInfoProvider;

	@Inject
	private ConfigProvider configProvider;

	@InjectLogger
	Logger logger;

	/*
	 * (non-Javadoc)
	 * 
	 * @see
	 * de.fraunhofer.sit.c2x.pki.ieee1609_2_d9.handler.LtcaCertificateWebHandler
	 * #prepareLtcaCertificateCreation()
	 */
	@Override
	public CaCertificateWebHandlerResult prepareCaCertificateCreation() throws KeyStoreException,
			NoSuchAlgorithmException, CertificateException, IOException {

		/*
		 * Variables to be given to the root CA: - signing public key -
		 * encryption public key - certificate identifier -> primary key of
		 * database table
		 */

		// Create new (uncompressed) signing and encryption key pairs
		AsymmetricCipherKeyPair signingKeyPair = KeyGenerator.getInstance().generateECKeyPair("P-256");

		AsymmetricCipherKeyPair encryptionKeyPair = KeyGenerator.getInstance().generateECKeyPair("P-256");

		// Extract Public Keys
		PublicKey signingKey = new PublicKey(PublicKeyAlgorithm.ECDSA_NISTP256_WITH_SHA256, new EccPoint(
				signingKeyPair.getPublic()));

		PublicKey encryptionKey = new PublicKey(PublicKeyAlgorithm.ECIES_NISTP256, new EccPoint(
				encryptionKeyPair.getPublic()), SymmAlgorithm.AES_128_CCM);

		// Add keys to the keystore
		long id = keystoreProvider.addKeystore(new KeystoreAdapter(new KeyStoreImpl(signingKeyPair,
				encryptionKeyPair, configProvider.get("keyStorePassword"))));

		if (logger.isDebugEnabled())
			logger.debug("Temporary keystore with ID " + id + " created that contains the private keys");

		// test whether the keys can be used to encrypt and decrypt data using ECIES
		try {
			testEncryption(encryptionKey, encryptionKeyPair.getPrivate());
		} catch (CryptoException e) {
			throw new IOException("Encrytion test with new keys failed", e);
		}

		// x, y, (uncompressed), (ieee/etsi), algorithmus:
		// ECDSA_NISTP256_WITH_SHA_256

		String signingKeyX = new String(Hex.encodeHex(signingKey.getPublicKey().getX().get()));

		String signingKeyY = new String(Hex.encodeHex(signingKey.getPublicKey().getY().get()));

		String encryptionKeyX = new String(Hex.encodeHex(encryptionKey.getPublicKey().getX().get()));

		String encryptionKeyY = new String(Hex.encodeHex(encryptionKey.getPublicKey().getY().get()));

		String version = "ETSI TS 103 097 V1.1.1"; // TODO Read Config/Flag

		String algorithm = signingKey.getAlgorithm().toString();

		String symmetricAlgorithm = encryptionKey.getSupportedSymmAlg().toString();

		String compression = EccPointType.UNCOMPRESSED.toString();

		CaCertificateWebHandlerResult result = new CaCertificateWebHandlerResult(id, signingKeyX,
				signingKeyY, encryptionKeyX, encryptionKeyY, version, algorithm, symmetricAlgorithm,
				compression);

		// return variables for webpage

		return result;

		// TODO

		/*
		 * The LTCA administrator should now send the variables to the RCA
		 * administrator and wait for response The RCA administrator has to
		 * return the certificate identifier and a signed certificate
		 */
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see
	 * de.fraunhofer.sit.c2x.pki.ieee1609_2_d9.handler.LtcaCertificateWebHandler
	 * #handleLtcaCertificateString(int, java.lang.String)
	 */
	@Override
	public boolean handleCaCertificateString(long primaryKey, String ltcaCertificate) throws Exception {

		byte[] certBytes = Hex.decodeHex(ltcaCertificate.toCharArray());

		// TODO handleCaCertificateString - checks

		// Parse the certificate bytes. If the parsing fails, the handling
		// will be aborted and "false" returned
		Certificate cert = new Certificate(new DataInputStream(new ByteArrayInputStream(certBytes)));
		
		if (cert.getVersionAndType().get() != new Certificate().getVersionAndType().get())
			throw new Exception("Invalid certificate version: " + cert.getVersionAndType().get()
					+ ". Only version " + new Certificate().getVersionAndType().get() + " supported");

		// Check DB if the is there is no certId associated with the given
		// primaryKey yet

		boolean tempCaCert = keystoreProvider.hasTemporaryCaCertificate(primaryKey);
		if (!tempCaCert)
			throw new Exception("The given id is either not valid or already used.");

		// Calculate certId from certificate
		HashedId8 certId = HashedId8.generateFromCertificate(certBytes);

		// Analyze and validate the certificate
		if (cert.getSubjectInfo().getSubjectType() != SubjectType.AUTHORIZATION_AUTHORITY)
			throw new Exception("The given certificate is not a PCA certificate");

		// TODO Check for correct root signature and check crl

		// TODO Analyze and validate the certificate -> compare public keys!

		// Store the certificate in the DB and link the certId of the
		// keystore
		caInfoProvider.setCaCertificate(cert);
		keystoreProvider.linkCaCertificate(primaryKey, certId.getCertID().get());

		// test whether the keys can be used to encrypt and decrypt data using ECIES
		try {
			CipherParameters privateEncryptionKey = keystoreProvider.getKey(certId.getCertID().get(),
					KeyType.ENCRYPTION_KEY);
			PublicKey publicEncryptionKey = cert.getEncryptionKey();
			testEncryption(publicEncryptionKey, privateEncryptionKey);
		} catch (CryptoException e) {
			throw new IOException("Encrytion test with new keys failed", e);
		}

		logger.info("Certificate " + new String(cert.getSubjectInfo().getSubjectName().get()) + "/"
				+ Hex.encodeHex(certId.getCertID().get()).toString() + " successfully stored in DB!");

		// No problems -> return true
		return true;
	}

	private void testEncryption(PublicKey receiverEncryptionPublicKey,
			CipherParameters receiverEncrpytionPrivateKey) throws IOException, CryptoException {
		// create sender and receiver keys
		AsymmetricCipherKeyPair senderEncryptionKeyPair = KeyGenerator.getInstance().generateECKeyPair(
				"P-256");
		EccPoint ltcEncryptionEccPublicKey = new EccPoint(senderEncryptionKeyPair.getPublic());

		PublicKey senderEncryptionPublicKey = new PublicKey(PublicKeyAlgorithm.ECIES_NISTP256,
				ltcEncryptionEccPublicKey, SymmAlgorithm.AES_128_CCM);

		// create test HashedId8 of signer
		HashedId8 receiverCertId = new HashedId8();
		receiverCertId.set(new Opaque(new byte[8], true));

		// temp variables
		RecipientInfo recipientInfo;
		EncryptionParameters encryptionParameters;
		byte[] aesCcmCiphertext;

		// encrypt signer info
		SignerInfo signerInfo = new SignerInfo(receiverCertId);

		Key symmetricKeyToEncrypt = KeyGenerator.getInstance().generateAesKey();

		byte[] messageToEncrypt = WaveUtils.getBytesFromWaveType(signerInfo);
		Opaque nonce = EtsiCryptoUtils.randomOpaqeArray(12);
		// encrypt signer data with symmetric key
		aesCcmCiphertext = EtsiCryptoUtils.encryptAES(symmetricKeyToEncrypt, nonce, messageToEncrypt);
		// encrypt symmetric key with own private key and the receiver's public
		// key
		ECIESCipher cipher = EtsiCryptoUtils.encryptECIES(symmetricKeyToEncrypt, senderEncryptionKeyPair
				.getPrivate(), receiverEncryptionPublicKey.getPublicKey().toECPublicKeyParameters());

		EccPoint senderEccEncryptionPublicKey = senderEncryptionPublicKey.getPublicKey();

		SymmAlgorithm symmAlgorithm = senderEncryptionPublicKey.getSupportedSymmAlg();
		EciesEncryptedKey encKey = new EciesEncryptedKey(senderEccEncryptionPublicKey,
				new Opaque(cipher.getCBytes(), true), new Opaque(cipher.getTBytes(), true), symmAlgorithm,
				new UInt32(16));
		recipientInfo = new RecipientInfo(receiverCertId, encKey);
		encryptionParameters = new EncryptionParameters(receiverEncryptionPublicKey.getSupportedSymmAlg(),
				nonce);

		// decrypt signer info for test purposes
		encKey = recipientInfo.getEncKey();
		senderEccEncryptionPublicKey = encKey.getV();
		cipher = new ECIESCipher(encKey.getC().get(), encKey.getT().get());

		// decrypt the symmetric key by using own private key and provided
		// public key
		Key symmetricKey = EtsiCryptoUtils.decryptECIES(cipher, receiverEncrpytionPrivateKey,
				senderEccEncryptionPublicKey.toECPublicKeyParameters());

		// decrypt the signer info by using the symmetric key
		Opaque decryptedMessage = EtsiCryptoUtils.decryptAES(symmetricKey, aesCcmCiphertext,
				encryptionParameters.getNonce());
		SignerInfo decryptedSignerInfo = WaveUtils.getElementFromBytes(decryptedMessage.get(),
				SignerInfo.class);

		if (signerInfo.equals(decryptedSignerInfo)) {
			if (logger.isDebugEnabled())
				logger.debug("Encryption test successfull");
		}
	}
}
