/**
 * 
 */
package de.fraunhofer.sit.c2x.pki.ca.validator.region;

import de.fraunhofer.sit.c2x.pki.ca.validator.region.shape.DoubleVector2d;


/**
 * UTM Zohnenfeld f&uuml;r das Versuchsgebiet: 32U<br>
 * <ul>
 * <li>48-56 N&ouml;rdlicher Breite (&phi;, latitude)</li>
 * <li>6-12 &Ouml;stlicher Lnge (&lambda;, longitude)</li>
 * </ul>
 * 
 * @author Attila Jaeger - TU Darmstadt <br>
 *         jaeger@iss.tu-darmstadt.de
 * @version 0.1.1
 * 
 */
public class UTMCoordinate extends DoubleVector2d {

	/** Standard UTM Zone 32U */
	public final static String DefaultUTMZone = "32U";

	/** Rechenkostante */
	private final static double rad2deg = 180.0 / Math.PI;

	/** Konstanter verzerrungsfaktor 0.9996 */
	public final static double factor = 0.9996;

	private String zone; // simplification for simTD

	/**
	 * For simTD the UTM Zone is set to 32U.
	 * 
	 * @param easting
	 * @param northing
	 */
	public UTMCoordinate(double easting, double northing, String zone) {
		super(easting, northing);
		this.zone = zone; // simplification for simTD
	}

	/**
	 * Converts this UTM coordinate in to WGS84
	 * <p>
	 * To optimize performance for simTD, this works only for UTMZone 32U
	 * 
	 * @return This coordinate as WGS84
	 */
	public WGSCoordinate toWGS() {
		double northing = y;
		double easting = x;
		String zone = this.zone;

		double k0 = factor;
		double a = WGSCoordinate.EquatorialRadius;
		double eccSquared = WGSCoordinate.eccentricitySquared;

		double e1 = (1 - Math.sqrt(1 - eccSquared)) / (1 + Math.sqrt(1 - eccSquared));
		// double e1 = 0.0016792203888649744; // simplification for simTD

		double x = easting - 500000.0; // remove 500,000 meter offset for
										// longitude
		double y = northing;

		int zoneNumber = Integer.valueOf(zone.substring(0, 2));
		// int zoneNumber = 32; // simplification for simTD
		char zoneLetter = zone.charAt(2);
		// char zoneLetter = 'U'; // simplification for simTD

//		int NorthernHemisphere; // 1 for northern hemispher, 0 for southern
		if (zoneLetter - 'N' >= 0) {
//			NorthernHemisphere = 1;// point is in northern hemisphere
		} else {
//			NorthernHemisphere = 0;// point is in southern hemisphere
			y -= 10000000.0;// remove 10,000,000 meter offset used for southern
							// hemisphere
		}

		double longOrigin = (zoneNumber - 1) * 6 - 180 + 3; // +3 puts origin in
															// middle of zone
		// double longOrigin = 9; // simplification for simTD

		double eccPrimeSquared = (eccSquared) / (1 - eccSquared);
		// double eccPrimeSquared = 0.006739496752268451; // simplification for
		// simTD

		double M = y / k0;
		double mu = M
				/ (a * (1 - eccSquared / 4 - 3 * eccSquared * eccSquared / 64 - 5 * eccSquared * eccSquared
						* eccSquared / 256));

		double phi1Rad = mu + (3 * e1 / 2 - 27 * e1 * e1 * e1 / 32) * Math.sin(2 * mu)
				+ (21 * e1 * e1 / 16 - 55 * e1 * e1 * e1 * e1 / 32) * Math.sin(4 * mu)
				+ (151 * e1 * e1 * e1 / 96) * Math.sin(6 * mu);
//		double phi1 = phi1Rad * rad2deg;

		double N = a / Math.sqrt(1 - eccSquared * Math.sin(phi1Rad) * Math.sin(phi1Rad));
		double T = Math.tan(phi1Rad) * Math.tan(phi1Rad);
		double C = eccPrimeSquared * Math.cos(phi1Rad) * Math.cos(phi1Rad);
		double R = a * (1 - eccSquared)
				/ Math.pow(1 - eccSquared * Math.sin(phi1Rad) * Math.sin(phi1Rad), 1.5);
		double D = x / (N * k0);

		double latitude = phi1Rad
				- (N * Math.tan(phi1Rad) / R)
				* (D * D / 2 - (5 + 3 * T + 10 * C - 4 * C * C - 9 * eccPrimeSquared) * D * D * D * D / 24 + (61
						+ 90 * T + 298 * C + 45 * T * T - 252 * eccPrimeSquared - 3 * C * C)
						* D * D * D * D * D * D / 720);
		latitude = latitude * rad2deg;

		double longitude = (D - (1 + 2 * T + C) * D * D * D / 6 + (5 - 2 * C + 28 * T - 3 * C * C + 8
				* eccPrimeSquared + 24 * T * T)
				* D * D * D * D * D / 120)
				/ Math.cos(phi1Rad);
		longitude = longOrigin + longitude * rad2deg;

		return new WGSCoordinate(latitude, longitude);
	}

	/**
	 * Berechnet die UTM Zone anhand von Longitude und Latitude
	 * 
	 * @param longitude
	 *            Longitude
	 * @param latitude
	 *            Latitude
	 * @return UTMZone
	 */
	public static String calcUTMZone(double longitude, double latitude) {

		int zoneNumber = ((int) longitude + 180) / 6 + 1;

		// Special zone for Norway
		if (latitude >= 56.0 && latitude < 64.0 && longitude >= 3.0 && longitude < 12.0)
			zoneNumber = 32;

		// Special zones for Svalbard
		if (latitude >= 72.0 && latitude < 84.0) {
			if (longitude >= 0.0 && longitude < 9.0)
				zoneNumber = 31;
			else if (longitude >= 9.0 && longitude < 21.0)
				zoneNumber = 33;
			else if (longitude >= 21.0 && longitude < 33.0)
				zoneNumber = 35;
			else if (longitude >= 33.0 && longitude < 42.0)
				zoneNumber = 37;
		}

		char letterDesignator;

		if (84 >= latitude && latitude >= 72)
			letterDesignator = 'X';
		else if (72 > latitude && latitude >= 64)
			letterDesignator = 'W';
		else if (64 > latitude && latitude >= 56)
			letterDesignator = 'V';
		else if (56 > latitude && latitude >= 48)
			letterDesignator = 'U';
		else if (48 > latitude && latitude >= 40)
			letterDesignator = 'T';
		else if (40 > latitude && latitude >= 32)
			letterDesignator = 'S';
		else if (32 > latitude && latitude >= 24)
			letterDesignator = 'R';
		else if (24 > latitude && latitude >= 16)
			letterDesignator = 'Q';
		else if (16 > latitude && latitude >= 8)
			letterDesignator = 'P';
		else if (8 > latitude && latitude >= 0)
			letterDesignator = 'N';
		else if (0 > latitude && latitude >= -8)
			letterDesignator = 'M';
		else if (-8 > latitude && latitude >= -16)
			letterDesignator = 'L';
		else if (-16 > latitude && latitude >= -24)
			letterDesignator = 'K';
		else if (-24 > latitude && latitude >= -32)
			letterDesignator = 'J';
		else if (-32 > latitude && latitude >= -40)
			letterDesignator = 'H';
		else if (-40 > latitude && latitude >= -48)
			letterDesignator = 'G';
		else if (-48 > latitude && latitude >= -56)
			letterDesignator = 'F';
		else if (-56 > latitude && latitude >= -64)
			letterDesignator = 'E';
		else if (-64 > latitude && latitude >= -72)
			letterDesignator = 'D';
		else if (-72 > latitude && latitude >= -80)
			letterDesignator = 'C';
		else
			letterDesignator = '?'; // This is here as an error flag to show
									// thatthe Latitude is outside the UTM
									// limits

		return (zoneNumber < 10 ? "0" : "") + zoneNumber + letterDesignator;
	}

	/**
	 * @return
	 */
	public String getZone() {
		return zone;
	}

}