<?xml version="1.0"?>
<xsl:stylesheet version="1.0"
                xmlns:xsl="http://www.w3.org/1999/XSL/Transform"
                xmlns:date="http://exslt.org/dates-and-times"
                extension-element-prefixes="date"                
                >
 <xsl:variable name="base-time"   select="'2018-01-01'"/>
 <xsl:variable name="local-region"    select="250"/>
 <xsl:variable name="base-latitude"   select="436169490.0"/>
 <xsl:variable name="base-longitude"  select="70533080.0"/>
<!--
 <xsl:variable name="base-latitude"   select="0"/>
 <xsl:variable name="base-longitude"  select="0"/>
-->
 <xsl:variable name="koeff-latitude"  select="89.833458"/>  <!--tenth of microseconds per metter-->
 <xsl:variable name="koeff-longitude" select="89.833458"/>  <!--tenth of microseconds per metter-->

 <xsl:variable name="AID-CAM"     select="36"/>
 <xsl:variable name="AID-DENM"    select="37"/>
 <xsl:variable name="AID-SPAT"    select="137"/>
 <xsl:variable name="AID-MAP"     select="138"/>
 <xsl:variable name="AID-IVI"     select="139"/>
 <xsl:variable name="AID-TLC"     select="140"/>
 <xsl:variable name="AID-GN-MGMT" select="141"/>

 <xsl:template match = "/">
  <xsl:apply-templates select="certificate"/>
 </xsl:template>
 
 <xsl:template match = "certificate">
  <EtsiTs103097Certificate>
    <xsl:apply-templates select="version"/>
    <type><explicit/></type>
    <xsl:apply-templates select="signer"/>
    <xsl:apply-templates select="subject"/>
    <xsl:apply-templates select="signature"/>
  </EtsiTs103097Certificate>
 </xsl:template>

 <xsl:template match = "version">
<!--   <version><xsl:value-of select = "." /></version> -->
   <version>3</version>
 </xsl:template>
 
 <xsl:template match = "signer">
  <xsl:choose>
  <xsl:when test="@type='digest'">
    <issuer>
     <xsl:attribute name="name">
       <xsl:value-of select="@name" />
     </xsl:attribute>
     <xsl:comment><xsl:value-of select = "@name" /></xsl:comment>
     <xsl:choose>
      <xsl:when test="contains(@alg, '384')">
        <sha384AndDigest/>
      </xsl:when>
      <xsl:otherwise>
        <sha256AndDigest/>
      </xsl:otherwise>
     </xsl:choose>
    </issuer>
    </xsl:when>
    <xsl:when test="@type='self'">
     <issuer>
       <self><sha256/></self>
     </issuer>
    </xsl:when>
    </xsl:choose>
 </xsl:template>

 <xsl:template mode="sha256Digest" match = "@name">
   <!-- TODO: should calculate sha256 of external certificate -->
   <xsl:value-of select = "." />
 </xsl:template>
 
 <xsl:template mode="sha384Digest" match = "@name">
   <!-- TODO: should calculate sha384 of external certificate -->
   <xsl:value-of select = "." />
 </xsl:template>

 <xsl:template match = "subject">
  <xsl:variable name="eeType">
   <xsl:choose>
    <xsl:when test="@type = 'AT'">0</xsl:when>
    <xsl:when test="@type = 'AA'">0</xsl:when>
    <xsl:when test="@type = 'ROOT'">0</xsl:when>
    <xsl:when test="@type = 'EC'">1</xsl:when>
    <xsl:when test="@type = 'EA'">1</xsl:when>
    <xsl:otherwise>2</xsl:otherwise>
   </xsl:choose>
  </xsl:variable>
 
  <toBeSigned>
   <id><none></none></id>
   <cracaId>00 00 00</cracaId>
   <crlSeries>0</crlSeries>

   <xsl:apply-templates select="../validity/restriction[@type='time']"/>
   <xsl:apply-templates mode="region" select="../validity/restriction[@type='region']"/>
   <xsl:apply-templates select="attribute[@type='assurance_level']"/>
   <xsl:apply-templates select="attribute[@type='its_aid_ssp_list']"/>
   <xsl:apply-templates select="attribute[@type='its_aid_list']">
    <xsl:with-param name="eeType" select="$eeType"/>
   </xsl:apply-templates>
   <xsl:apply-templates select="attribute[@type='encryption_key']"/>
   <xsl:apply-templates select="attribute[@type='verification_key']"/>
  </toBeSigned>
 </xsl:template>
 
 <xsl:template match="attribute[@type='assurance_level']">
  <xsl:variable name="level" select="assurance/@level"/>
  <xsl:variable name="confidence">
    <xsl:choose>
      <xsl:when test="assurance/@confidence">
        <xsl:value-of select="assurance/@confidence" />
      </xsl:when>
      <xsl:otherwise>
       <xsl:text>0</xsl:text>
      </xsl:otherwise>
    </xsl:choose>
  </xsl:variable>
  <assuranceLevel>
    <xsl:call-template name="ConvertDecToHex">
      <xsl:with-param name="index" select="$level * 32 + $confidence"/>
    </xsl:call-template>
  </assuranceLevel>
  <xsl:comment>level=<xsl:value-of select="$level"/> confidence=<xsl:value-of select="$confidence"/> </xsl:comment>
 </xsl:template>
 
 <xsl:template match="attribute[@type='its_aid_list']">
  <xsl:param name="eeType"/>
  <certIssuePermissions>
   <PsidGroupPermissions>
    <subjectPermissions>
     <explicit>
      <xsl:apply-templates mode="certIssuePermissions" select="aid"/>
     </explicit>
    </subjectPermissions>
    <minChainLength>1</minChainLength>
    <chainLengthRange>0</chainLengthRange>
    <eeType><xsl:value-of select="$eeType"/></eeType>
   </PsidGroupPermissions>
  </certIssuePermissions>
 </xsl:template>
 
 <xsl:template mode="certIssuePermissions" match="aid">
  <PsidSspRange>
   <xsl:choose>
    <xsl:when test="@value = 'CAM' or @value=$AID-CAM">
     <xsl:comment>CAM</xsl:comment>
     <psid><xsl:value-of select="$AID-CAM"/></psid>
     <sspRange>
      <bitmapSspRange>
       <sspValue>01 FF FF</sspValue>
       <sspBitmask>01 FF FF</sspBitmask>
      </bitmapSspRange>
     </sspRange>
    </xsl:when>
    <xsl:when test="@value = 'DENM' or @value=$AID-DENM">
     <xsl:comment>DENM</xsl:comment>
     <psid><xsl:value-of select="$AID-DENM"/></psid>
     <sspRange>
      <bitmapSspRange>
       <sspValue>01 FF FF FF</sspValue>
       <sspBitmask>01 FF FF FF</sspBitmask>
      </bitmapSspRange>
     </sspRange>
    </xsl:when>
    <xsl:when test="@value = 'TLM' or @value = 'SPAT' or @value=$AID-SPAT">
     <xsl:comment>SPAT</xsl:comment>
     <psid><xsl:value-of select="$AID-SPAT"/></psid>
     <sspRange>
      <bitmapSspRange>
       <sspValue>01 FF FF FF</sspValue>
       <sspBitmask>01 FF FF FF</sspBitmask>
      </bitmapSspRange>
     </sspRange>
    </xsl:when>
    <xsl:when test="@value = 'RLT' or @value = 'MAP' or @value=$AID-MAP">
     <xsl:comment>MAP</xsl:comment>
     <psid><xsl:value-of select="$AID-MAP"/></psid>
     <sspRange>
      <bitmapSspRange>
       <sspValue>01 FF FF FF</sspValue>
       <sspBitmask>01 FF FF FF</sspBitmask>
      </bitmapSspRange>
     </sspRange>
    </xsl:when>
    <xsl:when test="@value = 'IVI' or @value=$AID-IVI">
     <xsl:comment>IVI</xsl:comment>
     <psid><xsl:value-of select="$AID-IVI"/></psid>
     <sspRange>
      <bitmapSspRange>
       <sspValue>01 FF FF FF</sspValue>
       <sspBitmask>01 FF FF FF</sspBitmask>
      </bitmapSspRange>
     </sspRange>
    </xsl:when>
    <xsl:when test="@value = 'TLC' or @value=$AID-TLC">
     <xsl:comment>TLC</xsl:comment>
     <psid><xsl:value-of select="$AID-TLC"/></psid>
     <sspRange>
      <bitmapSspRange>
       <sspValue>01 FF FF FF</sspValue>
       <sspBitmask>01 FF FF FF</sspBitmask>
      </bitmapSspRange>
     </sspRange>
    </xsl:when>
    <xsl:when test="@value = 'GN-MGMT' or @value = $AID-GN-MGMT">
     <xsl:comment>GN-MGMT</xsl:comment>
     <psid><xsl:value-of select="$AID-GN-MGMT"/></psid>
    </xsl:when>
    <xsl:otherwise>
     <psid><xsl:value-of select="@value"/></psid>
    </xsl:otherwise>
    </xsl:choose>
   </PsidSspRange>
 </xsl:template>

 
 <xsl:template match="attribute[@type='its_aid_ssp_list']">
   <appPermissions>
    <xsl:apply-templates mode="appPermissions" select="ssp"/>
   </appPermissions>
 </xsl:template>
 <xsl:template mode="appPermissions" match="ssp">
   <PsidSsp>
     <xsl:choose>
      <xsl:when test="@aid = 'CAM' or @aid=$AID-CAM">
       <xsl:comment>CAM</xsl:comment>
       <psid><xsl:value-of select="$AID-CAM"/></psid>
       <ssp><bitmapSsp><xsl:value-of select="." /></bitmapSsp></ssp>
      </xsl:when>
      <xsl:when test="@aid = 'DENM' or @aid=$AID-DENM">
       <xsl:comment>DENM</xsl:comment>
       <psid><xsl:value-of select="$AID-DENM"/></psid>
       <ssp><bitmapSsp><xsl:value-of select="." /></bitmapSsp></ssp>
      </xsl:when>
      <xsl:when test="@aid = 'SPAT' or @aid='TLM' or @aid=$AID-SPAT">
       <xsl:comment>SPAT</xsl:comment>
       <psid><xsl:value-of select="$AID-SPAT"/></psid>
       <ssp><bitmapSsp><xsl:value-of select="." /></bitmapSsp></ssp>
      </xsl:when>
      <xsl:when test="@aid = 'MAP' or @aid='RLT' or @aid=$AID-MAP">
       <xsl:comment>MAP</xsl:comment>
       <psid><xsl:value-of select="$AID-MAP"/></psid>
       <ssp><bitmapSsp><xsl:value-of select="." /></bitmapSsp></ssp>
      </xsl:when>
      <xsl:when test="@aid = 'IVI' or @aid=$AID-IVI">
       <xsl:comment>IVI</xsl:comment>
       <psid><xsl:value-of select="$AID-IVI"/></psid>
       <ssp><bitmapSsp><xsl:value-of select="." /></bitmapSsp></ssp>
      </xsl:when>
      <xsl:when test="@aid = 'TLC' or @aid=$AID-TLC">
       <xsl:comment>TLC</xsl:comment>
       <psid><xsl:value-of select="$AID-TLC"/></psid>
       <ssp><bitmapSsp><xsl:value-of select="." /></bitmapSsp></ssp>
      </xsl:when>
      <xsl:when test="@aid = 'GN-MGMT' or @aid=$AID-GN-MGMT">
       <xsl:comment>GN-MGMT</xsl:comment>
       <psid><xsl:value-of select="$AID-GN-MGMT"/></psid>
       <ssp><bitmapSsp><xsl:value-of select="." /></bitmapSsp></ssp>
      </xsl:when>
      <xsl:otherwise>
       <psid><xsl:value-of select="@aid"/></psid>
       <xsl:if test="string-length(.) &gt; 0">
        <ssp><opaque><xsl:value-of select="." /></opaque></ssp>
       </xsl:if>
      </xsl:otherwise>
     </xsl:choose>
   </PsidSsp>
 </xsl:template>

 <xsl:template match="restriction[@type='time']">
   <validityPeriod>
     <xsl:apply-templates mode="validityperiod" select="@start"/>
     <xsl:apply-templates mode="validityperiod" select="@end">
      <xsl:with-param name="start" select="@start"/>
     </xsl:apply-templates>
     <xsl:apply-templates mode="validityperiod" select="@duration">
      <xsl:with-param name="start" select="@start"/>
     </xsl:apply-templates>
   </validityPeriod>
 </xsl:template>

 <xsl:template mode="validityperiod" match="@start">
   <xsl:variable name="start">
     <xsl:call-template name="time-value">
       <xsl:with-param name="value" select = "." /> 
       <xsl:with-param name="base"  select = "$base-time" /> 
     </xsl:call-template>
   </xsl:variable>
   <start>
     <xsl:comment><xsl:value-of select="date:add('2004-01-01', concat('PT',$start,'S'))"/></xsl:comment>
     <xsl:value-of select="$start"/>
   </start>
 </xsl:template>

 <xsl:template name="time-value">
   <xsl:param name = "value" />
   <xsl:param name = "base" />
   <xsl:choose>
     <xsl:when test="starts-with($value,'+')">
       <xsl:comment> <xsl:value-of select="concat('P',translate(substring($value,2),'ymd','YMD'))"/> </xsl:comment>
       <xsl:value-of select="date:seconds(date:add($base, 
                             concat('P',translate(substring($value,2),'ymd','YMD'))
                             )) - date:seconds('2004-01-01')"/>
     </xsl:when>
     <xsl:when test="starts-with($value,'-')">
       <xsl:comment> <xsl:value-of select="concat('-P',translate(substring($value,2),'ymd','YMD'))"/></xsl:comment>
       <xsl:value-of select="date:seconds(date:add($base, 
                             concat('-P',translate(substring($value,2),'ymd','YMD'))
                             )) - date:seconds('2004-01-01')"/>
     </xsl:when>
     <xsl:when test="$value = 'cur'">
       <xsl:value-of select="date:seconds(date:date()) - date:seconds('2004-01-01')"/>
     </xsl:when>
     <xsl:otherwise>
       <xsl:value-of select="date:seconds($value) - date:seconds('2004-01-01')"/>
     </xsl:otherwise>
   </xsl:choose>
 </xsl:template>


 <xsl:template mode="validityperiod" match="@end">
   <xsl:param name = "start" />
   <xsl:variable name="vstart">
    <xsl:call-template name="time-value">
      <xsl:with-param name="value" select="$start"/>
      <xsl:with-param name="base"  select="$base-time"/>
    </xsl:call-template>
   </xsl:variable>
   <xsl:variable name="vend">
    <xsl:call-template name="time-value">
      <xsl:with-param name="value" select="."/>
      <xsl:with-param name="base"  select="$base-time"/>
    </xsl:call-template>
   </xsl:variable>

   <duration><hours>
     <xsl:comment><xsl:value-of select="date:add('2004-01-01', concat('PT',$vend,'S'))"/></xsl:comment>
     <xsl:value-of select="($vend - $vstart) div 3600"/>
   </hours></duration>
 </xsl:template>

 <xsl:template mode="validityperiod" match="@duration">
   <xsl:param name = "start" />
   <xsl:variable name="duration">
    <xsl:call-template name="time-value">
      <xsl:with-param name="value" select="."/>
      <xsl:with-param name="base"  select="0"/>
    </xsl:call-template>
   </xsl:variable>
   <duration><hours>
     <xsl:comment><xsl:value-of select="date:add('2004-01-01', concat('PT',$start + $duration,'S'))"/></xsl:comment>
     <xsl:value-of select="$duration div 3600"/>
   </hours></duration>
 </xsl:template>

 <xsl:template mode="region" match="none">
 </xsl:template>

 <xsl:template mode="region" match="circle">
  <region>
    <circularRegion>
      <center>
        <latitude>
         <xsl:call-template name="geo-value">
           <xsl:with-param name="base"  select="$base-latitude"/>
           <xsl:with-param name="koeff" select="$koeff-latitude"/>
           <xsl:with-param name="value" select="@latitude"/>
         </xsl:call-template>
        </latitude>
        <longitude>
         <xsl:call-template name="geo-value">
           <xsl:with-param name="base"  select="$base-longitude"/>
           <xsl:with-param name="koeff" select="$koeff-longitude"/>
           <xsl:with-param name="value" select="@longitude"/>
         </xsl:call-template>
        </longitude>
      </center>
      <radius><xsl:value-of select="@radius"/></radius>
    </circularRegion>
  </region>
 </xsl:template>

 <xsl:template mode="region" match="rectangle">
  <xsl:variable name="nw" select="location[position() mod 2 = 1]" />
  <xsl:variable name="se" select="location[position() mod 2 = 0]" />
  <region>
    <rectangularRegion>
      <xsl:for-each select="$nw">
        <xsl:call-template name="rectangular-region">
          <xsl:with-param name="nw" select="$nw"/>
          <xsl:with-param name="se" select="$se"/>
          <xsl:with-param name="pos" select="position()"/>
        </xsl:call-template>
      </xsl:for-each>
    </rectangularRegion>
  </region>
 </xsl:template>

 <xsl:template name="rectangular-region">
   <xsl:param name="nw"/>
   <xsl:param name="se"/>
   <xsl:param name="pos"/>
   <RectangularRegion>
     <northWest>
       <latitude>
         <xsl:call-template name="geo-value">
           <xsl:with-param name="base"  select="$base-latitude"/>
           <xsl:with-param name="koeff" select="$koeff-latitude"/>
           <xsl:with-param name="value" select="$nw[$pos]/@latitude"/>
         </xsl:call-template>
       </latitude>
       <longitude>
         <xsl:call-template name="geo-value">
           <xsl:with-param name="base"  select="$base-longitude"/>
           <xsl:with-param name="koeff" select="$koeff-longitude"/>
           <xsl:with-param name="value" select="$nw[$pos]/@longitude"/>
         </xsl:call-template>
       </longitude>
     </northWest>
     <southEast>
       <latitude>
         <xsl:call-template name="geo-value">
           <xsl:with-param name="base"  select="$base-latitude"/>
           <xsl:with-param name="koeff" select="$koeff-latitude"/>
           <xsl:with-param name="value" select="$se[$pos]/@latitude"/>
         </xsl:call-template>
       </latitude>
       <longitude>
         <xsl:call-template name="geo-value">
           <xsl:with-param name="base"  select="$base-longitude"/>
           <xsl:with-param name="koeff" select="$koeff-longitude"/>
           <xsl:with-param name="value" select="$se[$pos]/@longitude"/>
         </xsl:call-template>
       </longitude>
     </southEast>
   </RectangularRegion>
 </xsl:template>

 <xsl:template mode="region" match="polygon">
  <region>
    <polygonalRegion>
      <xsl:for-each select="location">
        <TwoDLocation>
         <latitude>
           <xsl:call-template name="geo-value">
             <xsl:with-param name="base"  select="$base-latitude"/>
             <xsl:with-param name="koeff" select="$koeff-latitude"/>
             <xsl:with-param name="value" select="@latitude"/>
           </xsl:call-template>
         </latitude>
         <longitude>
           <xsl:call-template name="geo-value">
             <xsl:with-param name="base"  select="$base-longitude"/>
             <xsl:with-param name="koeff" select="$koeff-longitude"/>
             <xsl:with-param name="value" select="@longitude"/>
           </xsl:call-template>
         </longitude>
        </TwoDLocation>
      </xsl:for-each>
    </polygonalRegion>
  </region>
 </xsl:template>

 <xsl:template mode="region" match="id">
  <region>
    <identifiedRegion>
      <xsl:choose>
        <xsl:when test="@local != '0'">
          <countryAndRegions>
           <xsl:call-template name="country-only">
             <xsl:with-param name="id" select="@id"/>
           </xsl:call-template>
           <regions>
             <INTEGER>@local</INTEGER>
           </regions>
          </countryAndRegions>
        </xsl:when>
        <xsl:otherwise>
           <xsl:call-template name="country-only">
             <xsl:with-param name="id" select="@id"/>
           </xsl:call-template>
        </xsl:otherwise>
      </xsl:choose>
    </identifiedRegion>
  </region>
 </xsl:template>

 <xsl:template name="geo-value">
   <xsl:param    name="base"/>
   <xsl:param    name="value"/>
   <xsl:param    name="koeff"/>
   <xsl:variable name="km" select="substring-before"/>
   <xsl:choose>
     <xsl:when test="starts-with($value,'-')">
      <xsl:call-template name="geo-value">
        <xsl:with-param name="base"  select="$base"/>
        <xsl:with-param name="koeff" select="-1 * $koeff"/>
        <xsl:with-param name="value" select="substring($value,2)"/>
      </xsl:call-template>
     </xsl:when>
     <xsl:when test="starts-with($value,'+')">
      <xsl:call-template name="geo-value">
        <xsl:with-param name="base"  select="$base"/>
        <xsl:with-param name="koeff" select="$koeff"/>
        <xsl:with-param name="value" select="substring($value,2)"/>
      </xsl:call-template>
     </xsl:when>
     <xsl:otherwise>
       <xsl:choose>
         <xsl:when test="substring($value,string-length($value)-1) = 'km'">
           <xsl:value-of select="floor($base + ( $koeff * 1000 * substring($value,1,string-length($value)-2) ))"/>
         </xsl:when>
         <xsl:when test="substring($value,string-length($value)) = 'm'">
           <xsl:value-of select="floor($base + ( $koeff * substring($value,1,string-length($value)-1) ))"/>
         </xsl:when>
        <xsl:otherwise>
           <xsl:value-of select="floor($base + ( 10000000.0 * $value))"/>
        </xsl:otherwise>
       </xsl:choose>
     </xsl:otherwise>
   </xsl:choose>
 </xsl:template>


 <xsl:template name="country-only">
   <xsl:param name="id"/>
     <xsl:choose>
       <xsl:when test="$id='RGN_LOCAL'">
         <xsl:call-template name="country-only">
          <xsl:with-param name="id" select="$local-region"/>
         </xsl:call-template>
       </xsl:when>
       <xsl:otherwise>
         <countryOnly><xsl:value-of select="$id"/></countryOnly>
       </xsl:otherwise>
     </xsl:choose>
 </xsl:template>

 <xsl:template match="attribute[@type='encryption_key']">
  <encryptionKey>
   <xsl:apply-templates select="public_key"  mode="encryption_key"/>
  </encryptionKey>
 </xsl:template>

 <xsl:template match="attribute[@type='verification_key']">
 <verifyKeyIndicator>
  <verificationKey>
   <xsl:apply-templates select="public_key"  mode="verification_key"/>
  </verificationKey>
 </verifyKeyIndicator>
 </xsl:template>
 
 <xsl:template match="public_key" mode="verification_key">
  <xsl:choose>
   <xsl:when test="contains(@algorithm, 'brainpool') and contains(@algorithm, '256')">
    <ecdsaBrainpoolP256r1>
     <xsl:apply-templates select="ecc_point"/>
    </ecdsaBrainpoolP256r1>
   </xsl:when>
   <xsl:when test="contains(@algorithm, 'brainpool') and contains(@algorithm, '384')">
    <ecdsaBrainpoolP384r1>
     <xsl:apply-templates select="ecc_point">
      <xsl:with-param name="algorithm" select="'384'"/>
     </xsl:apply-templates>
    </ecdsaBrainpoolP384r1>
   </xsl:when>
   <xsl:otherwise>
    <ecdsaNistP256>
     <xsl:apply-templates select="ecc_point"/>
    </ecdsaNistP256>
   </xsl:otherwise>
  </xsl:choose>
 </xsl:template>

 <xsl:template match="public_key" mode="encryption_key">
  <supportedSymmAlg>
   <xsl:choose>
    <xsl:when test="@sym_alg != 'aes_128_ccm' and @sym_alg != 0">
    </xsl:when>
    <xsl:otherwise>
     <aes128Ccm/>
    </xsl:otherwise>
   </xsl:choose>
  </supportedSymmAlg>
  <publicKey>
   <xsl:choose>
    <xsl:when test="@algorithm != 'ecies_nistp256' and @algorithm != 0">
    </xsl:when>
    <xsl:otherwise>
     <eciesNistP256>
      <xsl:apply-templates select="ecc_point"/>
     </eciesNistP256>
    </xsl:otherwise>
   </xsl:choose>
  </publicKey>
 </xsl:template>

 <xsl:template match="ecc_point">
  <xsl:param name="algorithm" select="'256'"/>
  <xsl:choose>
   <xsl:when test="@type = 'x_coordinate_only' or @type = 0">
    <x-only>
     <xsl:value-of select="."/>
    </x-only>
   </xsl:when>
   <xsl:when test="@type = 'compressed' or @type = 'compressed_y0' or @type = 2">
    <compressed-y-0>
     <xsl:value-of select="."/>
    </compressed-y-0>
   </xsl:when>
   <xsl:when test="@type = 'compressed_y1' or @type = 3">
    <compressed-y-1>
     <xsl:value-of select="."/>
    </compressed-y-1>
   </xsl:when>
   <xsl:when test="@type = 'uncompressed' or @type = 4">
    <xsl:element name="{concat('uncompressedP', $algorithm)}">
     <x><xsl:value-of select="x"/></x>
     <y><xsl:value-of select="y"/></y>
    </xsl:element>
   </xsl:when>
  </xsl:choose>
 </xsl:template>
 
 <xsl:template match = "signature">
   <signature>
     <xsl:choose>
       <xsl:when test="contains(@algorithm, 'brainpool') and contains(@algorithm, '256')">
         <ecdsaBrainpoolP256r1Signature>
           <xsl:call-template name="signature-point">
             <xsl:with-param name="algorithm" select="@algorithm"/>
           </xsl:call-template> 
         </ecdsaBrainpoolP256r1Signature>
       </xsl:when>
       <xsl:when test="contains(@algorithm, 'brainpool') and contains(@algorithm, '384')">
         <ecdsaBrainpoolP384r1Signature>
           <xsl:call-template name="signature-point">
             <xsl:with-param name="algorithm" select="@algorithm"/>
           </xsl:call-template> 
         </ecdsaBrainpoolP384r1Signature>
       </xsl:when>
       <xsl:otherwise>
         <ecdsaNistP256Signature>
           <xsl:call-template name="signature-point">
             <xsl:with-param name="algorithm" select="@algorithm"/>
           </xsl:call-template> 
         </ecdsaNistP256Signature>
       </xsl:otherwise>
     </xsl:choose>
   </signature>
 </xsl:template>

 <xsl:template name = "signature-point">
   <rSig>
     <xsl:choose>
       <xsl:when test="ecc_point">
         <xsl:apply-templates match="ecc_point"/>
       </xsl:when>
       <xsl:otherwise>
         <x-only/>
       </xsl:otherwise>
     </xsl:choose>
   </rSig>
   <sSig/>
  </xsl:template>


  <xsl:template name="ConvertDecToHex">
    <xsl:param name="index" />
    <xsl:if test="$index > 0">
      <xsl:call-template name="ConvertDecToHex">
        <xsl:with-param name="index" select="floor($index div 16)" />
      </xsl:call-template>
      <xsl:choose>
        <xsl:when test="$index mod 16 &lt; 10">
          <xsl:value-of select="$index mod 16" />
        </xsl:when>
        <xsl:otherwise>
          <xsl:choose>
            <xsl:when test="$index mod 16 = 10">A</xsl:when>
            <xsl:when test="$index mod 16 = 11">B</xsl:when>
            <xsl:when test="$index mod 16 = 12">C</xsl:when>
            <xsl:when test="$index mod 16 = 13">D</xsl:when>
            <xsl:when test="$index mod 16 = 14">E</xsl:when>
            <xsl:when test="$index mod 16 = 15">F</xsl:when>
            <xsl:otherwise>A</xsl:otherwise>
          </xsl:choose>
        </xsl:otherwise>
      </xsl:choose>
    </xsl:if>
  </xsl:template>

</xsl:stylesheet>