/*!
 * \file      udp_layer.hh
 * \brief     Header file for ITS UDP/IP protocol layer definition.
 * \author    ETSI STF525
 * \copyright ETSI Copyright Notification
 *            No part may be reproduced except as authorized by written permission.
 *            The copyright and the foregoing restriction extend to reproduction in all media.
 *            All rights reserved.
 * \version   0.1
 */
#pragma once

#include <arpa/inet.h>
#if !defined(LINUX)
#include <netdb.h>
struct iphdr {
    #if __BYTE_ORDER == __LITTLE_ENDIAN
        u_int8_t    ihl:4,
                version:4;
    #elif __BYTE_ORDER == __BIG_ENDIAN
        u_int8_t    version:4,
                ihl:4;
    #else
        #error  "Please fix <asm/byteorder.h>"
    #endif
          u_int8_t   tos;
          u_int16_t  tot_len;
          u_int16_t  id;
          u_int16_t  frag_off;
          u_int8_t   ttl;
          u_int8_t   protocol;
          u_int16_t  check;
          struct  in_addr  saddr;
          struct  in_addr  daddr;
          //The options start here.
};
struct udphdr {
         u_int16_t   source;
         u_int16_t   dest;
         u_int16_t   len;
         u_int16_t   check;
};
#else // LINUX
#include <linux/ip.h>
#include <linux/udp.h>
#endif // LINUX

#include "t_layer.hh"
#include "params.hh"

/*!
 * \class udp_layer
 * \brief  This class provides description of ITS UDP/IP protocol layer
 */
class udp_layer : public layer {
  params _params;            //! Layer parameters
  struct iphdr* _iphdr;      //! IP layer description
  struct udphdr* _udphdr;    //! UDP layer description
  struct sockaddr_in _saddr; //! Source socket address description
  struct sockaddr_in _daddr; //! Destination socket address description

  /*!
   * \fn unsigned short inet_check_sum(const void *buf, size_t hdr_len, const unsigned short p_initial_sum = 0);
   * \brief Compute the UDP checksum
   * \param[in] p_buffer The data to be sent
   * \param[in] p_header_length The UDP header length
   * \param [in] p_initial_sum The initial checksum value. Default: 0
   * \return The checksum value
   */
  unsigned short inet_check_sum(const void *p_buffer, size_t p_header_length, const unsigned short p_initial_sum = 0);

public: //! \publicsection
  /*!
   * \brief Specialised constructor
   *        Create a new instance of the udp_layer class
   * \param[in] p_type \todo
   * \param[in] p_param \todo
   */
  udp_layer(const std::string & p_type, const std::string & p_param);
  /*!
   * \brief Default destructor
   */
  virtual ~udp_layer() {}

  /*!
   * \virtual
   * \fn void send_data(OCTETSTRING& data, params& params);
   * \brief Send bytes formated data to the lower layers
   * \param[in] p_data The data to be sent
   * \param[in] p_params Some parameters to overwrite default value of the lower layers parameters
   */
  virtual void send_data(OCTETSTRING& data, params& params);
  /*!
   * \virtual
   * \fn void receive_data(OCTETSTRING& data, params& params);
   * \brief Receive bytes formated data from the lower layers
   * \param[in] p_data The bytes formated data received
   * \param[in] p_params Some lower layers parameters values when data was received
   */
  virtual void receive_data(OCTETSTRING& data, params& info);
}; // End of class udp_layer

