package de.fraunhofer.sit.c2x.pki.ca.module.webserver.servlets;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

import org.apache.commons.codec.DecoderException;
import org.apache.commons.codec.binary.Hex;
import org.apache.log4j.Logger;

import com.google.inject.Inject;

import de.fraunhofer.sit.c2x.pki.ca.core.logging.InjectLogger;
import de.fraunhofer.sit.c2x.pki.ca.module.webserver.WebServerWithClientAuth;
import de.fraunhofer.sit.c2x.pki.ca.module.webserver.interfaces.HtmlProvider;
import de.fraunhofer.sit.c2x.pki.ca.provider.ProviderException;
import de.fraunhofer.sit.c2x.pki.ca.provider.interfaces.PPEExtendedProvider;
import de.fraunhofer.sit.c2x.pki.ca.utils.ByteUtils;

/**
 * A simple servlet that outputs the certificate content of issued certificates.
 * Following arguments are allowed per GET
 * 
 * <ul>
 * <li>CertId8 or CertId10: <b>/cacert?cert=HEX or Base64</b></li>
 * </ul>
 * 
 * @author Norbert Bissmeyer (norbert.bissmeyer@sit.fraunhofer.de)
 * 
 */
public class PPEDeletionServlet extends AbstractServlet {

	private static final long serialVersionUID = 1L;
	protected static final String URL = "/deletePPE";
	protected static final String TITLE = "Delete pseudonym intervals";

	@InjectLogger
	private Logger logger;
	private final PPEExtendedProvider ppeProvider;

	@Inject
	WebServerWithClientAuth webServerWithClientAuth;

	@Inject
	public PPEDeletionServlet(HtmlProvider htmlProvider, PPEExtendedProvider ppeProvider) {
		super(htmlProvider);
		this.ppeProvider = ppeProvider;
	}

	@Override
	public String getUrl() {
		return URL;
	}

	@Override
	public String getTitle() {
		return TITLE;
	}

	@Override
	protected String contentHtml(HttpServletRequest req, HttpServletResponse resp) {
		StringBuilder sb = new StringBuilder();
		try {
			if (req.getParameter("cert") != null && req.getParameter("cert").length() > 0) {
				byte[] certId = Hex.decodeHex(req.getParameter("cert").toCharArray());
				byte[] region = new byte[0];
				if (req.getParameter("region") != null && req.getParameter("region").length() > 0) {
					region = Hex.decodeHex(req.getParameter("region").toCharArray());
				}

				if (req.getParameter("delete") != null && req.getParameter("delete").equals("true")) {
					ppeProvider.remove(certId);
					if (region.length > 0)
						sb.append("<div id=\"ackbox\">Pseudonym interval removed! certId:"
								+ ByteUtils.getHex(certId) + " region=" + ByteUtils.getHex(region) + "</div>");
					else
						sb.append("<div id=\"ackbox\">Pseudonym interval removed! certId:"
								+ ByteUtils.getHex(certId) + "</div>");
				}
			}
		} catch (ProviderException | DecoderException e) {
			logger.warn(e);
			sb.append("<div id=\"errorbox\">Unable to delete pseudonym intervals from DB</div>");
		}

		// Prepare the form
		sb.append("<form name=\"certForm\" action=\"\" method=\"get\" onsubmit=\"return askBeforeDelete('Do you really want delete these entries?');\"><br />\n");

		String certIdString = req.getParameter("cert");
		if (certIdString == null)
			certIdString = "";
		String regionString = req.getParameter("region");
		if (regionString == null)
			regionString = "";

		sb.append("<table>\n");
		sb.append("<tr>\n<td style=\"width:250px;\">\n");
		sb.append("LTC certificate ID (HEX encoded HashedId8): ");
		sb.append("</td><td><input type=\"hidden\" name=\"delete\" value=\"true\"/>\n");
		sb.append("<input type=\"text\" id=\"cert\" name=\"cert\" value=\"" + certIdString
				+ "\" maxlength=\"80\" style=\"width:250px;\" />\n");
		sb.append("<input type=\"submit\" value=\"Delete\" /></td></tr></table></form>");

		return sb.toString();
	}

	@Override
	public boolean isProtected() {
		return true;
	}

	@Override
	public String[] getAllowedRoles() {
		return new String[] { UserRole.ADMINISTRATOR.toString(), UserRole.REGISTRATION_USER.toString(),
				UserRole.DEVELOPER.toString() };
	}

	@Override
	public Category getCategory() {
		return Category.LTC;
	}
}
